<?php

declare(strict_types=1);

namespace Drupal\mcp_client\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\mcp_client\McpServerInterface;
use Drupal\mcp_client\ValueObject\ToolCollection;
use Drupal\mcp_client\ValueObject\ToolInterface;
use Drupal\mcp_client\ValueObject\ToolCollectionInterface;

/**
 * Defines the MCP Server entity type.
 *
 * @ConfigEntityType(
 *   id = "mcp_server",
 *   label = @Translation("MCP Server"),
 *   label_collection = @Translation("MCP Servers"),
 *   label_singular = @Translation("MCP Server"),
 *   label_plural = @Translation("MCP Servers"),
 *   label_count = @PluralTranslation(
 *     singular = "@count MCP Server",
 *     plural = "@count MCP Servers",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\mcp_client\McpServerListBuilder",
 *     "form" = {
 *       "add" = "Drupal\mcp_client\Form\McpServerForm",
 *       "edit" = "Drupal\mcp_client\Form\McpServerForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *   },
 *   config_prefix = "mcp_server",
 *   admin_permission = "administer mcp_server",
 *   links = {
 *     "collection" = "/admin/structure/mcp-server",
 *     "add-form" = "/admin/structure/mcp-server/add",
 *     "edit-form" = "/admin/structure/mcp-server/{mcp_server}",
 *     "delete-form" = "/admin/structure/mcp-server/{mcp_server}/delete",
 *   },
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "endpoint",
 *     "transport_type",
 *     "stdio_command",
 *     "stdio_env",
 *     "stdio_cwd",
 *     "timeout",
 *     "http_headers",
 *     "tools",
 *   },
 * )
 */
final class McpServer extends ConfigEntityBase implements McpServerInterface {

  /**
   * The MCP Server ID.
   */
  protected string $id;

  /**
   * The MCP Server label.
   */
  protected string $label;

  /**
   * The MCP Server description.
   */
  protected string $description = '';

  /**
   * The MCP server endpoint URL.
   */
  protected string $endpoint = '';

  /**
   * The transport type (http, stdio).
   */
  protected string $transport_type = 'http';

  /**
   * STDIO command (for stdio transport).
   */
  protected ?string $stdio_command = NULL;

  /**
   * STDIO environment variables.
   *
   * @var array<string, string>
   */
  protected array $stdio_env = [];

  /**
   * STDIO working directory.
   */
  protected ?string $stdio_cwd = NULL;

  /**
   * HTTP timeout in seconds.
   */
  protected int $timeout = 30;

  /**
   * HTTP headers.
   *
   * @var array<string, string>
   */
  protected array $http_headers = [];

  /**
   * The tools configuration.
   *
   * @var array<int, array<string, mixed>>
   */
  protected array $tools = [];

  /**
   * Cached tool collection.
   *
   * @var \Drupal\mcp_client\ValueObject\ToolCollectionInterface|null
   */
  private ?ToolCollectionInterface $toolCollection = NULL;

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    return $this->description;
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription(string $description): self {
    $this->description = $description;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getEndpoint(): string {
    return $this->endpoint;
  }

  /**
   * {@inheritdoc}
   */
  public function setEndpoint(string $endpoint): self {
    $this->endpoint = $endpoint;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTransportType(): string {
    return $this->transport_type;
  }

  /**
   * {@inheritdoc}
   */
  public function setTransportType(string $transport_type): self {
    $this->transport_type = $transport_type;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStdioCommand(): ?string {
    return $this->stdio_command;
  }

  /**
   * {@inheritdoc}
   */
  public function setStdioCommand(?string $stdio_command): self {
    $this->stdio_command = $stdio_command;
    return $this;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-return array<string, string>
   */
  public function getStdioEnv(): array {
    return $this->stdio_env;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<string, string> $stdio_env
   */
  public function setStdioEnv(array $stdio_env): self {
    $this->stdio_env = $stdio_env;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStdioCwd(): ?string {
    return $this->stdio_cwd;
  }

  /**
   * {@inheritdoc}
   */
  public function setStdioCwd(?string $stdio_cwd): self {
    $this->stdio_cwd = $stdio_cwd;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTimeout(): int {
    return $this->timeout;
  }

  /**
   * {@inheritdoc}
   */
  public function setTimeout(int $timeout): self {
    $this->timeout = $timeout;
    return $this;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-return array<string, string>
   */
  public function getHttpHeaders(): array {
    return $this->http_headers;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<string, string> $http_headers
   */
  public function setHttpHeaders(array $http_headers): self {
    $this->http_headers = $http_headers;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTools(): ToolCollectionInterface {
    if ($this->toolCollection === NULL) {
      $this->toolCollection = ToolCollection::fromArray($this->tools);
    }
    return $this->toolCollection;
  }

  /**
   * {@inheritdoc}
   */
  public function setTools(ToolCollectionInterface $tools): self {
    $this->toolCollection = $tools;
    $this->tools = $tools->toArray();
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTool(string $name): ?ToolInterface {
    return $this->getTools()->get($name);
  }

  /**
   * {@inheritdoc}
   */
  public function hasTool(string $name): bool {
    return $this->getTools()->has($name);
  }

}
