<?php

declare(strict_types=1);

namespace Drupal\mcp_client\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\CachedDiscoveryClearerInterface;
use Drupal\Core\Render\Markup;
use Drupal\mcp_client\Entity\McpServer;
use Drupal\mcp_client\Service\McpClientFactory;
use Drupal\mcp_client\ValueObject\Tool;
use Drupal\mcp_client\ValueObject\ToolCollection;
use Drupal\tool\Tool\ToolOperation;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * MCP Server form.
 */
final class McpServerForm extends EntityForm {

  /**
   * Constructs a new McpServerForm.
   *
   * @param \Drupal\mcp_client\Service\McpClientFactory $clientFactory
   *   The MCP client factory service.
   * @param \Drupal\Core\Plugin\CachedDiscoveryClearerInterface $pluginCacheClearer
   *   The plugin cache clearer service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger service.
   */
  public function __construct(
    protected McpClientFactory $clientFactory,
    protected CachedDiscoveryClearerInterface $pluginCacheClearer,
    protected LoggerChannelInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new self(
      $container->get('mcp_client.client_factory'),
      $container->get('plugin.cache_clearer'),
      $container->get('logger.channel.mcp_client')
    );
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<string, mixed> $form
   * @phpstan-param \Drupal\Core\Form\FormStateInterface $form_state
   * @phpstan-return array<string, mixed>
   */
  public function form(array $form, FormStateInterface $form_state): array {

    $form = parent::form($form, $form_state);

    /** @var \Drupal\mcp_client\Entity\McpServer $entity */
    $entity = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#required' => TRUE,
      '#description' => $this->t('Label for the server for internal usage. Data name will prefix all tools.'),
      '#attributes' => [
        'placeholder' => 'My MCP Server',
      ],
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => [McpServer::class, 'load'],
      ],
      '#disabled' => !$entity->isNew(),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $entity->status(),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $entity->get('description'),
      '#description' => $this->t('A description of the server for internal purposes.'),
      '#rows' => 2,
    ];

    // Transport configuration fieldset.
    $form['transport'] = [
      '#type' => 'details',
      '#title' => $this->t('Transport Configuration'),
      '#open' => TRUE,
    ];

    $form['transport']['transport_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Transport Type'),
      '#options' => [
        'http' => $this->t('Streamable HTTP (recommended - works with modern MCP servers)'),
        'stdio' => $this->t('STDIO - Local process (for command-line MCP servers)'),
      ],
      '#default_value' => $entity->get('transport_type') ?? 'http',
      '#required' => TRUE,
      '#description' => $this->t('<strong>Use Streamable HTTP for remote MCP servers.</strong> STDIO is only for local command-line servers and should be used with caution.'),
      '#ajax' => [
        'callback' => '::transportTypeCallback',
        'wrapper' => 'transport-settings-wrapper',
        'event' => 'change',
      ],
    ];

    $form['transport']['settings'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'transport-settings-wrapper'],
    ];

    // Get current transport type (from AJAX or entity).
    $transport_type = $form_state->getValue(['transport', 'transport_type'])
      ?? $entity->get('transport_type')
      ?? 'http';

    // HTTP settings (use endpoint URL).
    if ($transport_type === 'http') {
      $form['transport']['settings']['endpoint'] = [
        '#type' => 'url',
        '#title' => $this->t('Endpoint URL'),
        '#default_value' => $entity->get('endpoint') ?? '',
        '#required' => TRUE,
        '#description' => $this->t('The HTTP endpoint URL (e.g., https://example.com/mcp).'),
        '#attributes' => [
          'placeholder' => 'https://example.com/mcp',
        ],
      ];

      $form['transport']['settings']['timeout'] = [
        '#type' => 'number',
        '#title' => $this->t('Timeout (seconds)'),
        '#default_value' => $entity->get('timeout') ?? 30,
        '#min' => 1,
        '#max' => 300,
        '#description' => $this->t('Connection timeout in seconds.'),
      ];

      // Advanced HTTP headers (collapsed).
      $form['transport']['settings']['advanced'] = [
        '#type' => 'details',
        '#title' => $this->t('Advanced Settings'),
        '#open' => FALSE,
      ];

      // HTTP Headers.
      $http_headers = $entity->get('http_headers') ?? [];

      $form['transport']['settings']['advanced']['http_headers'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('HTTP Headers'),
        '#tree' => TRUE,
        '#description' => $this->t('Add custom HTTP headers to send with requests.'),
      ];

      // Authorization header using Key module.
      $form['transport']['settings']['advanced']['http_headers']['Authorization_key'] = [
        '#type' => 'key_select',
        '#title' => $this->t('Authorization Key'),
        '#default_value' => $http_headers['Authorization_key'] ?? '',
        '#description' => $this->t('Select a key containing the authorization token.'),
        '#empty_option' => $this->t('- None -'),
      ];

      $form['transport']['settings']['advanced']['http_headers']['Authorization_prefix'] = [
        '#type' => 'select',
        '#title' => $this->t('Authorization Prefix'),
        '#options' => [
          'Bearer ' => $this->t('Bearer'),
          'Token ' => $this->t('Token'),
          'Basic ' => $this->t('Basic'),
          '' => $this->t('None (use raw token)'),
        ],
        '#default_value' => $http_headers['Authorization_prefix'] ?? 'Bearer ',
        '#description' => $this->t('Select the authorization scheme prefix.'),
        '#states' => [
          'visible' => [
            ':input[name="http_headers[Authorization_key]"]' => ['!value' => ''],
          ],
        ],
      ];
    }

    // STDIO settings (for local process execution).
    if ($transport_type === 'stdio') {
      $form['transport']['settings']['stdio_command'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Command'),
        '#default_value' => $entity->get('stdio_command') ?? '',
        '#required' => TRUE,
        '#description' => $this->t('Command to execute the MCP server (e.g., "node /path/to/server.js" or "python /path/to/server.py").'),
        '#attributes' => [
          'placeholder' => 'node /path/to/mcp-server.js',
        ],
      ];

      $form['transport']['settings']['stdio_cwd'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Working Directory'),
        '#default_value' => $entity->get('stdio_cwd') ?? '',
        '#description' => $this->t('Optional working directory for the process. Leave empty to use the system default.'),
        '#attributes' => [
          'placeholder' => '/path/to/working/directory',
        ],
      ];

      // Environment variables.
      $stdio_env = $entity->get('stdio_env') ?? [];

      $form['transport']['settings']['stdio_env'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Environment Variables'),
        '#tree' => TRUE,
        '#prefix' => '<div id="stdio-env-wrapper">',
        '#suffix' => '</div>',
      ];

      // Get the number of env vars from form state or entity.
      $num_env_vars = $form_state->get('num_env_vars');
      if ($num_env_vars === NULL) {
        $num_env_vars = !empty($stdio_env) ? count($stdio_env) : 1;
        $form_state->set('num_env_vars', $num_env_vars);
      }

      // Add existing env vars or empty fields.
      $env_keys = [];
      if (!empty($stdio_env)) {
        foreach ($stdio_env as $key => $value) {
          // Only add actual environment variable keys, not form elements.
          if (is_string($key) && !in_array($key, ['actions'])) {
            $env_keys[] = $key;
          }
        }
      }

      for ($i = 0; $i < $num_env_vars; $i++) {
        $env_key = $env_keys[$i] ?? '';
        $env_value = isset($env_keys[$i]) ? $stdio_env[$env_keys[$i]] : '';

        $form['transport']['settings']['stdio_env'][$i] = [
          '#type' => 'container',
          '#attributes' => ['class' => ['container-inline']],
        ];

        $form['transport']['settings']['stdio_env'][$i]['name'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Variable Name'),
          '#default_value' => $env_key,
          '#size' => 25,
          '#placeholder' => 'YOUR_ENV_VAR_NAME',
        ];

        $form['transport']['settings']['stdio_env'][$i]['value_type'] = [
          '#type' => 'select',
          '#title' => $this->t('Type'),
          '#options' => [
            'plain' => $this->t('Plain text'),
            'key' => $this->t('Key'),
          ],
          '#default_value' => (is_array($env_value) && isset($env_value['type'])) ? $env_value['type'] : 'plain',
        ];

        $form['transport']['settings']['stdio_env'][$i]['value'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Value'),
          '#default_value' => is_array($env_value) ? ($env_value['value'] ?? '') : $env_value,
          '#size' => 40,
          '#placeholder' => 'your-value-here',
          '#states' => [
            'visible' => [
              ':input[name="stdio_env[' . $i . '][value_type]"]' => ['value' => 'plain'],
            ],
          ],
        ];

        $form['transport']['settings']['stdio_env'][$i]['key_id'] = [
          '#type' => 'key_select',
          '#title' => $this->t('Key'),
          '#default_value' => is_array($env_value) ? ($env_value['key_id'] ?? '') : '',
          '#states' => [
            'visible' => [
              ':input[name="stdio_env[' . $i . '][value_type]"]' => ['value' => 'key'],
            ],
          ],
        ];
      }

      $form['transport']['settings']['stdio_env']['actions'] = [
        '#type' => 'actions',
      ];

      $form['transport']['settings']['stdio_env']['actions']['add'] = [
        '#type' => 'submit',
        '#value' => $this->t('Add another variable'),
        '#submit' => ['::addEnvVar'],
        '#ajax' => [
          'callback' => '::envVarCallback',
          'wrapper' => 'stdio-env-wrapper',
        ],
        '#limit_validation_errors' => [],
      ];
    }

    $tools_collection = $entity->getTools();
    if (count($tools_collection) > 0) {
      $form['tools_table'] = [
        '#type' => 'table',
        '#header' => [
          'enabled' => $this->t('Enable'),
          'name' => $this->t('Tool Name'),
          'description' => $this->t('Description'),
          'operation' => $this->t('Operation Type'),
        ],
        '#empty' => $this->t('No tools available. Save the form to fetch tools from the server.'),
      ];

      foreach ($tools_collection as $tool) {
        $tool_name = $tool->name();
        $form['tools_table'][$tool_name]['enabled'] = [
          '#type' => 'checkbox',
          '#default_value' => $tool->enabled(),
        ];

        $form['tools_table'][$tool_name]['name'] = [
          '#plain_text' => $tool_name,
        ];

        $form['tools_table'][$tool_name]['description'] = [
          '#plain_text' => $tool->description(),
        ];

        // Operation type selection.
        $operation_options = [];
        foreach (ToolOperation::cases() as $operation) {
          $operation_options[$operation->value] = $operation->getLabel();
        }

        $default_operation = $tool->getOperation()->value ?? ToolOperation::Read->value;

        $form['tools_table'][$tool_name]['operation'] = [
          '#type' => 'select',
          '#options' => $operation_options,
          '#default_value' => $default_operation,
          '#states' => [
            'visible' => [
              ':input[name="tools_table[' . $tool_name . '][enabled]"]' => ['checked' => TRUE],
            ],
          ],
        ];
      }
    }

    if (count($tools_collection) === 0 && !$entity->isNew()) {
      $form['tools_info'] = [
        '#markup' => '<p>' . $this->t('No tools available. Save the form to fetch tools from the server.') . '</p>',
      ];
    }

    $form['fetch_tools'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Fetch tools on save'),
      '#default_value' => TRUE,
      '#description' => $this->t('Attempt to connect to the server and fetch available tools when saving. Uncheck this if you want to configure the connection first without testing it.'),
    ];

    // Test connection section.
    $form['test_connection_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Test Connection'),
      '#open' => FALSE,
      '#attributes' => ['class' => ['mcp-test-connection']],
    ];

    $form['test_connection_section']['description'] = [
      '#markup' => '<p>' . $this->t('Test the connection to verify your configuration before saving.') . '</p>',
    ];

    // Test connection button.
    $form['test_connection_section']['test_connection'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Connection'),
      '#ajax' => [
        'callback' => '::testConnectionCallback',
        'wrapper' => 'test-connection-result',
        'event' => 'click',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Testing connection...'),
        ],
      ],
      '#attributes' => [
        'class' => ['button--primary'],
      ],
    ];

    // Container for test results.
    $form['test_connection_section']['test_result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'test-connection-result'],
    ];

    return $form;
  }

  /**
   * AJAX callback for transport type selection.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array<string, mixed>
   *   The form element to return.
   */
  public function transportTypeCallback(array &$form, FormStateInterface $form_state): array {
    return $form['transport']['settings'];
  }

  /**
   * Submit handler to add another environment variable field.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function addEnvVar(array &$form, FormStateInterface $form_state): void {
    $num_env_vars = $form_state->get('num_env_vars');
    $form_state->set('num_env_vars', $num_env_vars + 1);
    $form_state->setRebuild();
  }

  /**
   * AJAX callback for adding environment variables.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array<string, mixed>
   *   The form element to return.
   */
  public function envVarCallback(array &$form, FormStateInterface $form_state): array {
    return $form['transport']['settings']['stdio_env'];
  }

  /**
   * AJAX callback for testing connection.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array<string, mixed>
   *   The form element to return.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function testConnectionCallback(array &$form, FormStateInterface $form_state): array {
    $user_input = $form_state->getUserInput();
    $transport_type = $user_input['transport_type'] ?? 'http';

    $validation_errors = $this->validateTransportConfiguration($user_input, $transport_type);

    if (!empty($validation_errors)) {
      $form['test_connection_section']['test_result']['#theme'] = 'status_messages';
      $form['test_connection_section']['test_result']['#message_list'] = [
        'error' => [
          $this->t('Please fix the following errors before testing:'),
          [
            '#theme' => 'item_list',
            '#items' => $validation_errors,
          ],
        ],
      ];
      return $form['test_connection_section']['test_result'];
    }

    /** @var \Drupal\mcp_client\Entity\McpServer $tmp_mcp_server_entity */
    $tmp_mcp_server_entity = $this->entityTypeManager
      ->getStorage('mcp_server')
      ->create(['id' => 'temp_test']);

    $this->populateEntityFromUserInput($tmp_mcp_server_entity, $user_input, $transport_type);

    // Test the connection.
    try {
      $client = $this->clientFactory->createFromEntity($tmp_mcp_server_entity);
      $tools = $client->listTools();
      $tool_names = array_map(static fn($tool) => $tool['name'], array_slice($tools, 0, 5));
      $tool_list = !empty($tool_names) ? ' Available tools: ' . implode(', ', $tool_names) . (count($tools) > 5 ? '...' : '') : '';

      $form['test_connection_section']['test_result'] = [
        '#type' => 'container',
        'message' => [
          '#type' => 'markup',
          '#markup' => Markup::create('<div class="messages messages--status">✓ Connection successful! Found ' . count($tools) . ' tools.' . $tool_list . '</div>'),
        ],
      ];
    }
    catch (\Exception $e) {
      $form['test_connection_section']['test_result'] = [
        '#type' => 'container',
        'message' => [
          '#type' => 'markup',
          '#markup' => Markup::create('<div class="messages messages--error">✗ Connection failed: ' . $e->getMessage() . '</div>'),
        ],
      ];
    }
    return $form['test_connection_section']['test_result'];
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @phpstan-param array<string, mixed> $form
   * @param-out array<string, mixed> $form
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $transport_type = $form_state->getValue('transport_type');
    if ($transport_type === 'http') {
      $http_headers_input = $form_state->getValue('http_headers') ?? [];
      $form_state->setValue('http_headers', $this->parseHttpHeaders($http_headers_input));
    }

    if ($transport_type === 'stdio') {
      $stdio_env_input = $form_state->getValue('stdio_env') ?? [];
      $form_state->setValue('stdio_env', $this->parseStdioEnvironmentVariables($stdio_env_input));
    }

    // @phpstan-ignore-next-line paramOut.type
    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<string, mixed> $form
   * @phpstan-param \Drupal\Core\Form\FormStateInterface $form_state
   * @phpstan-return int
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function save(array $form, FormStateInterface $form_state): int {
    /** @var \Drupal\mcp_client\Entity\McpServer $entity */
    $entity = $this->entity;

    // Extract and save transport settings.
    $transport_type = $form_state->getValue('transport_type');

    // Get the original transport type from the original entity if it exists.
    $old_transport_type = NULL;
    if (!$entity->isNew()) {
      /** @var \Drupal\mcp_client\Entity\McpServer|null $original */
      $original = $this->entityTypeManager
        ->getStorage('mcp_server')
        ->loadUnchanged($entity->id());
      $old_transport_type = $original?->getTransportType();
    }

    // Check if transport type changed - if so, clear tools.
    if ($old_transport_type !== NULL && $old_transport_type !== $transport_type) {
      $entity->setTools(new ToolCollection());
      $this->messenger()->addWarning(
        $this->t('Transport type changed from @old to @new. Tools have been cleared and will be reloaded.', [
          '@old' => $old_transport_type,
          '@new' => $transport_type,
        ])
      );
    }

    $entity->setTransportType($transport_type);
    // Update tool enabled/operation states from form.
    $this->updateToolsFromForm($entity, $form_state);

    // Save transport-specific settings.
    if ($transport_type === 'http') {
      $endpoint = $form_state->getValue('endpoint');
      $timeout = $form_state->getValue('timeout');
      $http_headers = $form_state->getValue('http_headers');
      $entity
        ->setEndpoint($endpoint ?: '')
        ->setTimeout((int) $timeout ?: 30)
        ->setHttpHeaders($http_headers ?: [])
          // Clear STDIO-specific fields.
        ->setStdioCommand(NULL)
        ->setStdioCwd(NULL)
        ->setStdioEnv([]);
    }
    elseif ($transport_type === 'stdio') {
      $stdio_command = $form_state->getValue('stdio_command');
      $stdio_cwd = $form_state->getValue('stdio_cwd');
      $stdio_env = $form_state->getValue('stdio_env');
      $entity
        ->setStdioCommand($stdio_command ?: '')
        ->setStdioCwd($stdio_cwd ?: '')
        ->setStdioEnv($stdio_env ?: [])
          // Clear HTTP-specific fields.
        ->setEndpoint('')
        ->setHttpHeaders([]);
    }

    // Try to fetch tools from the server.
    $fetch_tools = (bool) $form_state->getValue('fetch_tools');
    if ($fetch_tools) {
      try {
        $this->messenger()->addStatus($this->t('Attempting to connect to MCP server at @endpoint...', [
          '@endpoint' => $entity->get('endpoint'),
        ]));

        // Get existing tools to preserve their enabled/operation states.
        $existing_tools = $entity->getTools();

        $client = $this->clientFactory->createFromEntity($entity);
        $tools_data = $client->listTools();
        if (count($tools_data) > 0) {
          $tools = [];
          foreach ($tools_data as $tool_data) {
            $tool_name = $tool_data['name'];
            $existing_tool = $existing_tools->get($tool_name);

            // Preserve enabled/operation from existing tool, or use defaults.
            $enabled = $existing_tool ? $existing_tool->enabled() : TRUE;
            $operation = $existing_tool ? $existing_tool->getOperation() : $this->inferOperationFromMetadata($tool_data);

            $tools[] = Tool::fromArray([
              'name' => $tool_name,
              'description' => $tool_data['description'] ?? '',
              // Following MCP protocol standard (camelCase).
              'inputSchema' => $tool_data['inputSchema'] ?? [],
              'enabled' => $enabled,
              'locked' => FALSE,
              'operation' => $operation?->value,
            ]);
          }
          $entity->setTools(new ToolCollection(...$tools));
          $this->messenger()->addStatus($this->t('Successfully fetched @count tools from the server.', [
            '@count' => count($tools),
          ]));
        }
        else {
          $this->messenger()->addWarning($this->t('Connected to server but no tools were found.'));
        }
      }
      catch (\Exception $e) {
        // Log the error but continue with save.
        $this->messenger()->addWarning(
          $this->t('Could not fetch tools from server: @message. You can configure tools later by editing the server.', [
            '@message' => $e->getMessage(),
          ])
        );
        $this->logger->error('Failed to fetch tools: @message. Trace: @trace', [
          '@message' => $e->getMessage(),
          '@trace' => $e->getTraceAsString(),
        ]);
      }
    }
    else {
      $this->messenger()->addWarning($this->t('Tool fetching skipped. You can fetch tools later by editing this server and enabling "Fetch tools on save".'));
    }

    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $entity->label()];
    $message = match($result) {
      \SAVED_NEW => $this->t('Created new MCP Server %label.', $message_args),
      \SAVED_UPDATED => $this->t('Updated MCP Server %label.', $message_args),
      default => $this->t('Saved MCP Server %label.', $message_args),
    };
    $this->messenger()->addStatus($message);
    $form_state->setRedirectUrl($entity->toUrl('collection'));
    // Clear the plugin cache for AI function calls.
    $this->pluginCacheClearer->clearCachedDefinitions();

    return $result;
  }

  /**
   * Updates tools from form data.
   *
   * @param \Drupal\mcp_client\Entity\McpServer $entity
   *   The entity.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  private function updateToolsFromForm(McpServer $entity, FormStateInterface $form_state): void {
    $tools_table = $form_state->getValue('tools_table');
    if (empty($tools_table)) {
      return;
    }

    $updated_tools = [];
    foreach ($entity->getTools() as $tool) {
      $tool_name = $tool->name();
      if (isset($tools_table[$tool_name])) {
        $enabled = !empty($tools_table[$tool_name]['enabled']);
        $operation_value = $tools_table[$tool_name]['operation'] ?? $tool->getOperation()->value ?? 'read';
        $operation = ToolOperation::from($operation_value);
        $updated_tools[] = $tool->toBuilder()
          ->setEnabled($enabled)
          ->setOperation($operation)
          ->build();
      }
      else {
        $updated_tools[] = $tool;
      }
    }

    $entity->setTools(new ToolCollection(...$updated_tools));
  }

  /**
   * Parse HTTP headers from form input.
   *
   * @param array<string, mixed> $input
   *   The form input array.
   *
   * @return array<string, mixed>
   *   The parsed HTTP headers.
   */
  private function parseHttpHeaders(array $input): array {
    $http_headers = [];

    if (!empty($input['Authorization_key'])) {
      $http_headers['Authorization_key'] = $input['Authorization_key'];
      $http_headers['Authorization_prefix'] = $input['Authorization_prefix'] ?? 'Bearer ';
    }

    return $http_headers;
  }

  /**
   * Parse STDIO environment variables from form input.
   *
   * @param array<string, mixed> $input
   *   The form input array.
   *
   * @return array<string, mixed>
   *   The parsed environment variables.
   */
  private function parseStdioEnvironmentVariables(array $input): array {
    $stdio_env = [];

    if (!empty($input)) {
      foreach ($input as $key => $env_var) {
        if (!is_numeric($key) || empty($env_var['name'])) {
          continue;
        }

        $var_name = trim($env_var['name']);
        if (empty($var_name)) {
          continue;
        }

        if (isset($env_var['value_type']) && $env_var['value_type'] === 'key' && !empty($env_var['key_id'])) {
          $stdio_env[$var_name] = [
            'type' => 'key',
            'key_id' => $env_var['key_id'],
          ];
        }
        else {
          $stdio_env[$var_name] = [
            'type' => 'plain',
            'value' => $env_var['value'] ?? '',
          ];
        }
      }
    }

    return $stdio_env;
  }

  /**
   * Validates transport configuration.
   *
   * @param array<string, mixed> $input
   *   User input array.
   * @param string $transport_type
   *   The transport type.
   *
   * @return array<\Drupal\Core\StringTranslation\TranslatableMarkup>
   *   Validation errors, if any.
   */
  private function validateTransportConfiguration(array $input, string $transport_type): array {
    $validation_errors = [];

    if ($transport_type === 'http') {
      $endpoint = $input['endpoint'] ?? '';
      if (empty($endpoint)) {
        $validation_errors[] = $this->t('Endpoint URL is required for HTTP transport.');
      }
    }
    elseif ($transport_type === 'stdio') {
      $stdio_command = $input['stdio_command'] ?? '';
      if (empty($stdio_command)) {
        $validation_errors[] = $this->t('Command is required for STDIO transport.');
      }
    }

    return $validation_errors;
  }

  /**
   * Populates entity from user input.
   *
   * @param \Drupal\mcp_client\Entity\McpServer $entity
   *   The entity to populate.
   * @param array<string, mixed> $input
   *   User input array.
   * @param string $transport_type
   *   The transport type.
   */
  private function populateEntityFromUserInput(McpServer $entity, array $input, string $transport_type): void {
    $entity->set('transport_type', $transport_type);

    if ($transport_type === 'http') {
      $endpoint = $input['endpoint'] ?? '';
      $timeout = $input['timeout'] ?? 30;
      $http_headers_input = $input['http_headers'] ?? [];

      $entity->set('endpoint', $endpoint);
      $entity->set('timeout', $timeout);
      $entity->set('http_headers', $this->parseHttpHeaders($http_headers_input));
    }
    elseif ($transport_type === 'stdio') {
      $stdio_command = $input['stdio_command'] ?? '';
      $stdio_cwd = $input['stdio_cwd'] ?? '';
      $stdio_env_input = $input['stdio_env'] ?? [];

      $entity->set('stdio_command', $stdio_command);
      $entity->set('stdio_cwd', $stdio_cwd);
      $entity->set('stdio_env', $this->parseStdioEnvironmentVariables($stdio_env_input));
    }
  }

  /**
   * Infers the operation type from tool metadata.
   *
   * @param array<string, mixed> $tool_info
   *   The tool information including name and description.
   *
   * @return \Drupal\tool\Tool\ToolOperation
   *   The inferred operation.
   */
  protected function inferOperationFromMetadata(array $tool_info): ToolOperation {
    $name = strtolower($tool_info['name'] ?? '');
    $description = strtolower($tool_info['description'] ?? '');
    $combined = $name . ' ' . $description;

    // Read operations: get, fetch, list, read, show, view, retrieve, query.
    if (preg_match('/\b(get|fetch|list|read|show|view|retrieve|query|search|find)\b/', $combined)) {
      return ToolOperation::Read;
    }

    // Write operations: create, update, delete, save, write, modify, set,
    // remove.
    if (preg_match('/\b(create|update|delete|save|write|modify|set|remove|edit|insert)\b/', $combined)) {
      return ToolOperation::Write;
    }

    // Transform operations: convert, transform, parse, format, encode, decode.
    if (preg_match('/\b(convert|transform|parse|format|encode|decode|translate|process)\b/', $combined)) {
      return ToolOperation::Transform;
    }

    // Explain operations: explain, describe, info, help, schema, definition.
    if (preg_match('/\b(explain|describe|info|help|schema|definition|documentation)\b/', $combined)) {
      return ToolOperation::Explain;
    }

    // Default to Trigger for actions that don't fit other categories.
    return ToolOperation::Trigger;
  }

}
