<?php

declare(strict_types=1);

namespace Drupal\mcp_client;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\mcp_client\ValueObject\ToolInterface;
use Drupal\mcp_client\ValueObject\ToolCollectionInterface;

/**
 * Provides an interface defining a MCP Server entity type.
 */
interface McpServerInterface extends ConfigEntityInterface {

  /**
   * Gets the server description.
   *
   * @return string
   *   The description.
   */
  public function getDescription(): string;

  /**
   * Sets the server description.
   *
   * @param string $description
   *   The description.
   *
   * @return $this
   */
  public function setDescription(string $description): self;

  /**
   * Gets the server endpoint URL.
   *
   * @return string
   *   The endpoint URL.
   */
  public function getEndpoint(): string;

  /**
   * Sets the server endpoint URL.
   *
   * @param string $endpoint
   *   The endpoint URL.
   *
   * @return $this
   */
  public function setEndpoint(string $endpoint): self;

  /**
   * Gets the transport type.
   *
   * @return string
   *   The transport type (http|stdio).
   */
  public function getTransportType(): string;

  /**
   * Sets the transport type.
   *
   * @param string $transport_type
   *   The transport type (http|stdio).
   *
   * @return $this
   */
  public function setTransportType(string $transport_type): self;

  /**
   * Gets the STDIO command.
   *
   * @return string|null
   *   The STDIO command or null.
   */
  public function getStdioCommand(): ?string;

  /**
   * Sets the STDIO command.
   *
   * @param string|null $stdio_command
   *   The STDIO command.
   *
   * @return $this
   */
  public function setStdioCommand(?string $stdio_command): self;

  /**
   * Gets the STDIO environment variables.
   *
   * @return array<string, string>
   *   Array of environment variables.
   */
  public function getStdioEnv(): array;

  /**
   * Sets the STDIO environment variables.
   *
   * @param array<string, string> $stdio_env
   *   Array of environment variables.
   *
   * @return self
   *   The MCP Server instance.
   */
  public function setStdioEnv(array $stdio_env): self;

  /**
   * Gets the STDIO working directory.
   *
   * @return string|null
   *   The working directory or null.
   */
  public function getStdioCwd(): ?string;

  /**
   * Sets the STDIO working directory.
   *
   * @param string|null $stdio_cwd
   *   The working directory.
   *
   * @return self
   *   The MCP Server instance.
   */
  public function setStdioCwd(?string $stdio_cwd): self;

  /**
   * Gets the HTTP timeout in seconds.
   *
   * @return int
   *   The timeout in seconds.
   */
  public function getTimeout(): int;

  /**
   * Sets the HTTP timeout in seconds.
   *
   * @param int $timeout
   *   The timeout in seconds.
   *
   * @return self
   *   The MCP Server instance.
   */
  public function setTimeout(int $timeout): self;

  /**
   * Gets the HTTP headers.
   *
   * @return array<string, string>
   *   Array of HTTP headers.
   */
  public function getHttpHeaders(): array;

  /**
   * Sets the HTTP headers.
   *
   * @param array<string, string> $http_headers
   *   Array of HTTP headers.
   *
   * @return self
   *   The MCP Server instance.
   */
  public function setHttpHeaders(array $http_headers): self;

  /**
   * Gets the tool collection.
   *
   * @return \Drupal\mcp_client\ValueObject\ToolCollectionInterface
   *   The tool collection.
   */
  public function getTools(): ToolCollectionInterface;

  /**
   * Sets the tool collection.
   *
   * @param \Drupal\mcp_client\ValueObject\ToolCollectionInterface $tools
   *   The tool collection.
   *
   * @return self
   *   The MCP Server instance.
   */
  public function setTools(ToolCollectionInterface $tools): self;

  /**
   * Gets a specific tool by name.
   *
   * @param string $name
   *   The tool name.
   *
   * @return \Drupal\mcp_client\ValueObject\ToolInterface|null
   *   The tool or null if not found.
   */
  public function getTool(string $name): ?ToolInterface;

  /**
   * Checks if a tool exists.
   *
   * @param string $name
   *   The tool name.
   *
   * @return bool
   *   True if the tool exists.
   */
  public function hasTool(string $name): bool;

}
