<?php

declare(strict_types=1);

namespace Drupal\mcp_client\ValueObject;

/**
 * Immutable value object representing a tool's input schema.
 *
 * Represents a JSON Schema for tool input validation.
 */
final class InputSchema implements InputSchemaInterface {

  /**
   * Constructs an InputSchema.
   *
   * @param string $type
   *   The schema type (e.g., 'object').
   * @param array<string, mixed> $properties
   *   The schema properties.
   * @param array<int, string> $required
   *   Required property names.
   * @param array<string, mixed> $additionalProperties
   *   Additional schema properties.
   */
  public function __construct(
    public readonly string $type = 'object',
    public readonly array $properties = [],
    public readonly array $required = [],
    public readonly array $additionalProperties = [],
  ) {}

  /**
   * Creates an InputSchema from an array.
   *
   * @param array<string, mixed> $data
   *   Schema data.
   *
   * @return self
   *   New InputSchema instance.
   */
  public static function fromArray(array $data): self {
    return new self(
      $data['type'] ?? 'object',
      $data['properties'] ?? [],
      $data['required'] ?? [],
      array_diff_key($data, array_flip(['type', 'properties', 'required'])),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(): array {
    $schema = [
      'type' => $this->type,
      'properties' => $this->properties,
    ];

    if (!empty($this->required)) {
      $schema['required'] = $this->required;
    }

    return array_merge($schema, $this->additionalProperties);
  }

  /**
   * {@inheritdoc}
   */
  public function getProperty(string $name): ?array {
    return $this->properties[$name] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getProperties(): array {
    return $this->properties;
  }

  /**
   * {@inheritdoc}
   */
  public function isRequired(string $name): bool {
    return in_array($name, $this->required, TRUE);
  }

}
