<?php

declare(strict_types=1);

namespace Drupal\mcp_client\ValueObject;

/**
 * Immutable value object representing a locked tool definition snapshot.
 *
 * Captures the tool's metadata at the time it was locked.
 */
final class LockedDefinition implements LockedDefinitionInterface {

  /**
   * Constructs a LockedDefinition.
   *
   * @param string $name
   *   The tool name.
   * @param string $description
   *   The tool description.
   * @param InputSchemaInterface $inputSchema
   *   The tool input schema (following MCP protocol standard).
   */
  public function __construct(
    public string $name,
    public string $description,
    public InputSchemaInterface $inputSchema,
  ) {}

  /**
   * Creates a LockedDefinition from an array.
   *
   * @param array<string, mixed> $data
   *   Locked definition data. Supports both inputSchema (MCP standard,
   *   camelCase) and input_schema (Drupal convention, snake_case) for
   *   backward compatibility.
   *
   * @return self
   *   New LockedDefinition instance.
   */
  public static function fromArray(array $data): self {
    // Support both inputSchema (MCP standard) and input_schema (legacy).
    $inputSchemaData = $data['inputSchema'] ?? $data['input_schema'] ?? [];
    return new self(
      $data['name'],
      $data['description'],
      InputSchema::fromArray($inputSchemaData),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(): array {
    return [
      'name' => $this->name,
      'description' => $this->description,
      // Following MCP protocol standard (camelCase).
      'inputSchema' => $this->inputSchema->toArray(),
    ];
  }

}
