<?php

declare(strict_types=1);

namespace Drupal\mcp_client\ValueObject;

use Drupal\tool\Tool\ToolOperation;

/**
 * Immutable value object representing an MCP tool.
 *
 * Contains tool metadata, configuration, and optional locked definition.
 */
final class Tool implements ToolInterface {

  /**
   * Constructs a Tool.
   *
   * @param string $name
   *   The tool name.
   * @param string $description
   *   The tool description.
   * @param InputSchemaInterface $inputSchema
   *   The tool input schema (following MCP protocol standard).
   * @param bool $enabled
   *   Whether the tool is enabled.
   * @param bool $locked
   *   Whether the tool is locked.
   * @param \Drupal\tool\Tool\ToolOperation $operation
   *   The tool operations.
   * @param LockedDefinitionInterface|null $lockedDefinition
   *   The locked definition (following MCP protocol standard).
   */
  public function __construct(
    public readonly string $name,
    public readonly string $description,
    public readonly InputSchemaInterface $inputSchema,
    public readonly bool $enabled = FALSE,
    public readonly bool $locked = FALSE,
    public readonly ?ToolOperation $operation = NULL,
    public readonly ?LockedDefinitionInterface $lockedDefinition = NULL,
  ) {
    if ($this->locked && $this->lockedDefinition === NULL) {
      throw new \InvalidArgumentException('Locked tools must have a locked_definition');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function name(): string {
    return $this->name;
  }

  /**
   * {@inheritdoc}
   */
  public function description(): string {
    return $this->description;
  }

  /**
   * {@inheritdoc}
   */
  public function getInputSchema(): InputSchemaInterface {
    return $this->inputSchema;
  }

  /**
   * {@inheritdoc}
   */
  public function enabled(): bool {
    return $this->enabled;
  }

  /**
   * {@inheritdoc}
   */
  public function locked(): bool {
    return $this->locked;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperation(): ?ToolOperation {
    return $this->operation;
  }

  /**
   * {@inheritdoc}
   */
  public function getLockedDefinition(): ?LockedDefinitionInterface {
    return $this->lockedDefinition;
  }

  /**
   * Creates a builder from this Tool instance.
   *
   * @return \Drupal\mcp_client\ValueObject\ToolBuilder
   *   A new builder initialized with this tool's values.
   */
  public function toBuilder(): ToolBuilder {
    return ToolBuilder::fromTool($this);
  }

  /**
   * Creates a Tool from an array using ToolBuilder.
   *
   * @param array<string, mixed> $data
   *   Tool data array.
   *
   * @return self
   *   New Tool instance.
   */
  public static function fromArray(array $data): self {
    return ToolBuilder::fromArray($data)->build();
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(): array {
    return [
      'name' => $this->name,
      'description' => $this->description,
      // Following MCP protocol standard (camelCase).
      'inputSchema' => $this->inputSchema->toArray(),
      'enabled' => $this->enabled,
      'locked' => $this->locked,
      'operation' => $this->operation?->value,
      // Following MCP protocol standard (camelCase).
      'lockedDefinition' => $this->lockedDefinition?->toArray(),
    ];
  }

}
