<?php

declare(strict_types=1);

namespace Drupal\mcp_client\ValueObject;

use Drupal\tool\Tool\ToolOperation;

/**
 * Builder for creating and modifying Tool instances.
 *
 * Provides a mutable, fluent interface for constructing immutable Tool objects
 * without creating intermediate objects. More memory-efficient than
 * wither methods.
 *
 * @example
 * ```php
 * $tool = ToolBuilder::create('my_tool', 'My tool description', $schema)
 *   ->setEnabled(true)
 *   ->setOperation(ToolOperation::READ)
 *   ->build();
 *
 * $modified = ToolBuilder::fromTool($existingTool)
 *   ->setEnabled(false)
 *   ->setOperation(ToolOperation::WRITE)
 *   ->build();
 * ```
 */
final class ToolBuilder {

  /**
   * The tool name.
   *
   * @var string
   */
  private string $name;

  /**
   * The tool description.
   *
   * @var string
   */
  private string $description;

  /**
   * The tool input schema.
   *
   * @var \Drupal\mcp_client\ValueObject\InputSchemaInterface
   */
  private InputSchemaInterface $inputSchema;

  /**
   * Whether the tool is enabled.
   *
   * @var bool
   */
  private bool $enabled = FALSE;

  /**
   * Whether the tool is locked.
   *
   * @var bool
   */
  private bool $locked = FALSE;

  /**
   * The tool operation.
   *
   * @var \Drupal\tool\Tool\ToolOperation|null
   */
  private ?ToolOperation $operation = NULL;

  /**
   * The locked definition.
   *
   * @var \Drupal\mcp_client\ValueObject\LockedDefinitionInterface|null
   */
  private ?LockedDefinitionInterface $lockedDefinition = NULL;

  /**
   * Private constructor to enforce factory methods.
   *
   * @param string $name
   *   The tool name.
   * @param string $description
   *   The tool description.
   * @param \Drupal\mcp_client\ValueObject\InputSchemaInterface $inputSchema
   *   The tool input schema.
   */
  private function __construct(
    string $name,
    string $description,
    InputSchemaInterface $inputSchema,
  ) {
    $this->name = $name;
    $this->description = $description;
    $this->inputSchema = $inputSchema;
  }

  /**
   * Creates a new builder with required fields.
   *
   * @param string $name
   *   The tool name.
   * @param string $description
   *   The tool description.
   * @param \Drupal\mcp_client\ValueObject\InputSchemaInterface $inputSchema
   *   The tool input schema.
   *
   * @return self
   *   A new builder instance.
   */
  public static function create(
    string $name,
    string $description,
    InputSchemaInterface $inputSchema,
  ): self {
    return new self($name, $description, $inputSchema);
  }

  /**
   * Creates a builder from an existing Tool.
   *
   * @param \Drupal\mcp_client\ValueObject\Tool $tool
   *   The tool to copy values from.
   *
   * @return self
   *   A new builder with the tool's values.
   */
  public static function fromTool(Tool $tool): self {
    $builder = new self($tool->name, $tool->description, $tool->inputSchema);
    $builder->enabled = $tool->enabled;
    $builder->locked = $tool->locked;
    $builder->operation = $tool->operation;
    $builder->lockedDefinition = $tool->lockedDefinition;
    return $builder;
  }

  /**
   * Creates a builder from an array.
   *
   * @param array<string, mixed> $data
   *   Tool data array.
   *
   * @return self
   *   A new builder with the array values.
   *
   * @throws \InvalidArgumentException
   *   If required fields are missing.
   */
  public static function fromArray(array $data): self {
    $name = $data['name'] ?? NULL;
    $description = $data['description'] ?? NULL;
    $inputSchemaData = $data['inputSchema'] ?? $data['input_schema'] ?? NULL;

    if ($name === NULL || $name === '') {
      throw new \InvalidArgumentException('Tool name is required in array data');
    }
    if ($description === NULL || $description === '') {
      throw new \InvalidArgumentException('Tool description is required in array data');
    }
    if ($inputSchemaData === NULL) {
      throw new \InvalidArgumentException('Tool input schema is required in array data');
    }

    $inputSchema = InputSchema::fromArray($inputSchemaData);
    $builder = new self($name, $description, $inputSchema);

    $builder->enabled = $data['enabled'] ?? TRUE;
    $builder->locked = $data['locked'] ?? FALSE;
    $builder->operation = isset($data['operation']) ? ToolOperation::tryFrom($data['operation']) : NULL;

    $lockedDefData = $data['lockedDefinition'] ?? $data['locked_definition'] ?? NULL;
    $builder->lockedDefinition = $lockedDefData ? LockedDefinition::fromArray($lockedDefData) : NULL;

    return $builder;
  }

  /**
   * Sets the tool name.
   */
  public function setName(string $name): self {
    $this->name = $name;
    return $this;
  }

  /**
   * Sets the tool description.
   */
  public function setDescription(string $description): self {
    $this->description = $description;
    return $this;
  }

  /**
   * Sets the input schema.
   */
  public function setInputSchema(InputSchemaInterface $inputSchema): self {
    $this->inputSchema = $inputSchema;
    return $this;
  }

  /**
   * Sets the enabled state.
   */
  public function setEnabled(bool $enabled): self {
    $this->enabled = $enabled;
    return $this;
  }

  /**
   * Sets the locked state.
   */
  public function setLocked(bool $locked): self {
    $this->locked = $locked;
    return $this;
  }

  /**
   * Sets the tool operation.
   */
  public function setOperation(?ToolOperation $operation): self {
    $this->operation = $operation;
    return $this;
  }

  /**
   * Sets the locked definition.
   */
  public function setLockedDefinition(?LockedDefinitionInterface $lockedDefinition): self {
    $this->lockedDefinition = $lockedDefinition;
    return $this;
  }

  /**
   * Builds the immutable Tool instance.
   *
   * @return \Drupal\mcp_client\ValueObject\Tool
   *   The constructed Tool.
   *
   * @throws \InvalidArgumentException
   *   If validation fails (e.g., locked tool without definition).
   */
  public function build(): Tool {
    return new Tool(
      $this->name,
      $this->description,
      $this->inputSchema,
      $this->enabled,
      $this->locked,
      $this->operation,
      $this->lockedDefinition
    );
  }

}
