<?php

declare(strict_types=1);

namespace Drupal\mcp_client\ValueObject;

/**
 * Collection of MCP tools.
 *
 * Provides iteration, counting, and management of tools.
 */
final class ToolCollection implements ToolCollectionInterface {

  /**
   * Tools in the collection.
   *
   * @var array<string, ToolInterface>
   */
  private array $tools = [];

  /**
   * Constructs a ToolCollection.
   *
   * @param ToolInterface ...$tools
   *   Variable number of tools to add to the collection.
   */
  public function __construct(ToolInterface ...$tools) {
    foreach ($tools as $tool) {
      $this->add($tool);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name): ?ToolInterface {
    return $this->tools[$name] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function add(ToolInterface $tool): void {
    $this->tools[$tool->name()] = $tool;
  }

  /**
   * {@inheritdoc}
   */
  public function remove(string $name): void {
    unset($this->tools[$name]);
  }

  /**
   * {@inheritdoc}
   */
  public function has(string $name): bool {
    return isset($this->tools[$name]);
  }

  /**
   * {@inheritdoc}
   */
  public function all(): array {
    return array_values($this->tools);
  }

  /**
   * Creates a ToolCollection from an array.
   *
   * @param array<int, array<string, mixed>> $data
   *   Array of tool data.
   *
   * @return self
   *   New ToolCollection instance.
   */
  public static function fromArray(array $data): self {
    $tools = array_map(static fn(array $toolData) => Tool::fromArray($toolData), $data);
    return new self(...$tools);
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(): array {
    return array_map(static fn(ToolInterface $tool) => $tool->toArray(), array_values($this->tools));
  }

  /**
   * {@inheritdoc}
   */
  public function getIterator(): \Traversable {
    return new \ArrayIterator($this->tools);
  }

  /**
   * {@inheritdoc}
   */
  public function count(): int {
    return count($this->tools);
  }

}
