---
id: 1
summary: "Integrate Tool API module with MCP Server to expose Tool API tools as MCP tools via configuration entities"
created: 2025-11-09
---

# Plan: Tool API to MCP Tools Integration

## Original Work Order

> I want to discover use the tools from the Tool API module (https://www.drupal.org/project/tool) to be MCP tools.
>
> Since the tools are provided by other modules and we cannot add the mcp/sdk annotations to them, we'll need some other way to mark tools as "MCP tools". I think the best way to do this is by using a configuration entity. Each configuration entity contains a required field identifying the tool, and then it can have other fields to add the MCP metadata. Additionally the MCP tool can be enabled and disabled using the status field in the configuration entity.
>
> Creating a new configuration entity type shoud be done using drush

## Plan Clarifications

| Question | Answer |
|----------|--------|
| Should the Tool API module be installed as a dependency? | Required dependency |
| What fields should the configuration entity have? | Tool ID (required), MCP name override, Description override, Status field (boolean, defaults to true) |
| Should there be a UI for managing these entities? | Full CRUD UI |
| How should the system discover available Tool API tools? | Tool manager service |

## Executive Summary

This plan establishes a bridge between Drupal's Tool API module and the MCP Server module, enabling any Tool API tool to be exposed as an MCP tool without requiring code modifications to the tool providers. The integration uses Drupal configuration entities as a registration mechanism, where each entity represents a Tool API tool that should be exposed to MCP clients.

The approach leverages Tool API's existing typed input/output definitions and tool manager service for discovery, while allowing site administrators to selectively enable tools, customize their MCP-facing names and descriptions, and manage the integration through a dedicated admin UI. This design maintains separation of concerns: Tool API modules define business logic with typed interfaces, while MCP tool configurations control AI assistant exposure.

Key benefits include: no need to modify existing Tool API tools, centralized management of MCP tool exposure, ability to customize metadata for AI context, and seamless integration with the existing MCP Server attribute-based discovery system.

## Context

### Current State

The mcp_server module currently uses PHP 8 attributes (`#[McpTool]`, `#[McpResource]`, etc.) to discover MCP capabilities from Drupal services. Developers annotate methods directly with these attributes and tag services with `mcp.tool`, `mcp.resource`, etc. This works well for custom code but cannot be applied to Tool API tools, which are:

- Defined by third-party modules without MCP knowledge
- Already have typed input/output schemas via ContextDefinition
- Cannot be annotated with MCP attributes without forking

The Tool API module (drupal/tool) provides a modern framework for defining reusable, typed tools with explicit input/output contracts. It includes auto-generated forms, JSON schema export for AI integration, and a tool manager service for discovery. However, there's no automatic way to expose these tools as MCP capabilities.

### Target State

After implementation, administrators will be able to:

1. Install Tool API as a dependency of mcp_server
2. Browse available Tool API tools through an admin UI
3. Create MCP tool configurations that reference Tool API tools
4. Customize MCP-specific metadata (name, description)
5. Enable/disable individual tools for MCP exposure
6. Have Tool API tools automatically appear as MCP tools to AI assistants

The system will automatically discover Tool API tools via the tool manager service, convert their typed schemas to MCP parameter definitions, and route MCP tool calls to the appropriate Tool API tool execution.

### Background

The Tool API module was designed with AI integration in mind, including a `tool_ai_connector` submodule and JSON schema export capabilities. This makes it an ideal candidate for MCP integration. The challenge is that Tool API tools are discovered and registered differently than MCP capabilities:

- Tool API uses plugin-style discovery or service-based registration
- MCP Server uses attribute-based discovery on tagged services
- Configuration entities provide a bridge: they're discoverable, manageable, and can map between the two systems

Previous consideration was given to using annotations directly on Tool API tools, but this would require modifying contrib/third-party code, violating the principle of clean separation and making updates difficult.

```mermaid
graph TB
    A[Tool API Module] -->|Provides| B[Typed Tools]
    B -->|Discovered by| C[Tool Manager Service]
    D[Config Entity] -->|References| B
    D -->|Provides MCP Metadata| E[MCP Server]
    C -->|Tool Discovery| D
    E -->|Executes| B
    F[Admin UI] -->|Creates/Manages| D
    G[AI Assistant] -->|Calls MCP Tool| E
```

## Technical Implementation Approach

### Component 1: Configuration Entity Type
**Objective**: Create a configuration entity to store the mapping between Tool API tools and MCP metadata

Using `drush generate config-entity`, create an `mcp_tool_config` configuration entity with:

- **Machine name** (entity ID): Used as config storage key
- **Tool ID** (string, required): Reference to Tool API tool (e.g., 'my_module:send_email')
- **MCP name** (string, optional): Override tool name for MCP exposure (defaults to tool ID)
- **Description** (text, optional): Override description for AI context
- **Status** (boolean): Enable/disable flag (defaults to TRUE, uses ConfigEntityBase's built-in status field)

The entity will extend `ConfigEntityBase` and implement `ConfigEntityInterface`. Schema definitions will be added to `config/schema/mcp_server.schema.yml`.

### Component 2: Tool API Dependency Integration
**Objective**: Add Tool API as a required dependency and create service integration layer

Update `composer.json` to require `drupal/tool:^1.0` and add it to `mcp_server.info.yml` dependencies. Create a service wrapper (`ToolApiDiscovery`) that:

- Injects the Tool API tool manager service
- Provides methods to list all available tools
- Converts Tool API tool definitions to MCP-compatible format
- Handles tool execution by delegating to Tool API

This service will be tagged and injected into the MCP server building process.

### Component 3: MCP Tool Bridge Service
**Objective**: Create a service that converts configuration entities into MCP tool registrations

Implement `McpToolApiBridge` service that:

- Loads all enabled `mcp_tool_config` entities
- For each entity, retrieves the corresponding Tool API tool definition
- Converts Tool API input schemas (ContextDefinition) to MCP parameter schemas
- Registers methods with `#[McpTool]` attributes dynamically
- Routes MCP tool calls to Tool API tool execution
- Handles parameter validation and type conversion

This service will be tagged with `mcp.tool` to integrate with the existing MCP discovery system.

```mermaid
sequenceDiagram
    participant AI as AI Assistant
    participant MCP as MCP Server
    participant Bridge as MCP Tool API Bridge
    participant Config as Config Entity Storage
    participant Tool as Tool API Manager

    AI->>MCP: List available tools
    MCP->>Bridge: Discover MCP tools
    Bridge->>Config: Load enabled configs
    Config-->>Bridge: Return config entities
    Bridge->>Tool: Get tool definitions
    Tool-->>Bridge: Return tool metadata
    Bridge-->>MCP: Return MCP tool list
    MCP-->>AI: Available tools

    AI->>MCP: Call tool (with params)
    MCP->>Bridge: Execute tool
    Bridge->>Config: Validate tool enabled
    Bridge->>Tool: Execute tool
    Tool-->>Bridge: Return result
    Bridge-->>MCP: Format result
    MCP-->>AI: Tool result
```

### Component 4: Admin UI
**Objective**: Provide a full CRUD interface for managing MCP tool configurations

Create admin routes and controllers for:

1. **List page** (`/admin/config/services/mcp-server/tools`):
   - Table showing all MCP tool configurations
   - Columns: Tool ID, MCP Name, Status, Operations
   - Filter by status (enabled/disabled)
   - Bulk operations for enable/disable

2. **Add form** (`/admin/config/services/mcp-server/tools/add`):
   - Autocomplete field for Tool ID (populated from Tool API manager)
   - Shows tool description and input/output schema preview
   - Fields for MCP name override and description override
   - Status checkbox (enabled by default)

3. **Edit form** (`/admin/config/services/mcp-server/tools/{id}/edit`):
   - Same fields as add form
   - Shows original Tool API definition for reference

4. **Delete confirmation** (`/admin/config/services/mcp-server/tools/{id}/delete`):
   - Standard Drupal delete confirmation form

Integrate with existing MCP Server admin menu structure. Add permissions: `administer mcp tool configurations`.

### Component 5: Service Provider Updates
**Objective**: Integrate the bridge service into MCP Server's service discovery

Update `McpServerServiceProvider` to:

- Detect services tagged with `mcp.tool_api.bridge`
- Ensure Tool API tool discovery happens during container compilation
- Register dynamic MCP tool methods from configuration entities
- Handle cache invalidation when configurations change

### Component 6: Testing and Documentation
**Objective**: Ensure the integration works correctly and is well-documented

Create:

- Unit tests for Tool API to MCP schema conversion
- Kernel tests for configuration entity CRUD operations
- Functional tests for admin UI
- Integration tests for end-to-end MCP tool execution
- Update README.md with Tool API integration instructions
- Add code examples for common use cases

## Risk Considerations and Mitigation Strategies

### Technical Risks

- **Tool API Schema Incompatibility**: Tool API's ContextDefinition types may not map cleanly to MCP parameter schemas
    - **Mitigation**: Create a comprehensive type mapping layer with fallback handling; document unsupported types; add validation that warns about incompatible schemas

- **Dynamic Tool Registration**: Registering MCP tools dynamically from configuration entities may conflict with attribute-based discovery
    - **Mitigation**: Use the service locator pattern already in place; ensure bridge service is properly tagged; test with both static (attribute) and dynamic (config) tools

- **Performance Impact**: Loading all configurations and Tool API definitions on every MCP request could be slow
    - **Mitigation**: Implement caching at multiple levels (compiled tool list, schema conversions); use lazy loading for tool execution; benchmark with realistic tool counts

### Implementation Risks

- **Drush Generate Output**: Code generated by `drush generate config-entity` may need significant customization
    - **Mitigation**: Review and refactor generated code; ensure it follows mcp_server coding standards; add proper type hints and documentation

- **Tool API Version Compatibility**: Different Tool API versions may have breaking changes
    - **Mitigation**: Pin to specific Tool API version range in composer.json; add version detection logic; document supported versions

- **Configuration Export/Import**: Config entities must work properly with config management
    - **Mitigation**: Test full config export/import cycle; ensure schema definitions are complete; verify dependencies are tracked correctly

### Integration Risks

- **Permission Model**: Tool API tools may have their own access checks that conflict with MCP permissions
    - **Mitigation**: Respect Tool API access checks; document that MCP exposure doesn't bypass Drupal permissions; add configuration option to require additional MCP-specific permission

- **Error Handling**: Tool API exceptions may not be appropriate for MCP error responses
    - **Mitigation**: Wrap Tool API execution in try-catch; convert Tool API errors to MCP error format; log detailed errors while returning sanitized messages to AI

## Success Criteria

### Primary Success Criteria

1. **Tool API tools can be exposed as MCP tools**: Create a configuration entity for any Tool API tool and have it appear in MCP tool listings
2. **Admin UI is functional**: Add, edit, delete, and list MCP tool configurations through the admin interface
3. **MCP tool calls execute correctly**: AI assistants can successfully call Tool API tools via MCP and receive properly formatted results
4. **Schema conversion works**: Tool API input/output definitions are correctly converted to MCP parameter schemas
5. **Enable/disable works**: Status field correctly controls whether tools appear in MCP listings

### Quality Assurance Metrics

1. **Test coverage**: All components have unit, kernel, or functional tests with 80%+ coverage
2. **Code standards**: All code passes `phpcs --standard=Drupal,DrupalPractice`
3. **Static analysis**: Code passes `phpstan analyse` with no errors
4. **Documentation completeness**: README updated with clear examples, API documented with proper docblocks
5. **Performance acceptable**: MCP tool discovery adds <100ms overhead, tool execution adds <50ms overhead
6. **Configuration management**: Config entities properly export/import with config:export and config:import

## Resource Requirements

### Development Skills

- Drupal configuration entity development (Drupal 10/11)
- Drupal form API and admin UI creation
- Service container and dependency injection patterns
- Understanding of MCP protocol and Tool API architecture
- PHP 8.3+ features (attributes, typed properties, constructor promotion)
- Drush command usage and code generation

### Technical Infrastructure

- Drush 13.x for code generation (`drush generate config-entity`)
- PHPUnit for testing (already configured in project)
- PHPStan for static analysis (already configured)
- Tool API module (drupal/tool:^1.0)
- MCP Inspector for manual testing (`npx @modelcontextprotocol/inspector`)

### External Dependencies

- Tool API module: Must be installed and configured
- At least one module providing Tool API tools for testing (could create test module)
- MCP client (Claude Desktop, or custom) for end-to-end validation

## Implementation Order

The implementation should proceed in this sequence to manage complexity and enable incremental testing:

1. **Add Tool API dependency**: Update composer.json and info file first to ensure the foundation is in place
2. **Generate configuration entity**: Use Drush to generate boilerplate, then customize the entity class and schema
3. **Create Tool API discovery service**: Build the abstraction layer for Tool API integration before the bridge
4. **Implement MCP bridge service**: Create the core conversion logic that connects the two systems
5. **Build admin UI**: Add user-facing management interface once the backend is functional
6. **Update service provider**: Integrate bridge service into MCP Server's discovery mechanism
7. **Add tests and documentation**: Validate everything works and document the integration

This order ensures each component can be tested independently before integration, and dependencies are established before dependent components are built.

## Notes

- The configuration entity approach provides flexibility to add more fields later (e.g., parameter mapping rules, rate limiting, custom permissions)
- Consider adding a Drush command to bulk-import all available Tool API tools as configurations
- Future enhancement: Support for Tool API resource providers (if they exist) to expose as MCP resources
- The Tool API module's JSON schema export feature could be leveraged for MCP schema generation
- Monitor Tool API module development for new features that could enhance integration

## Task Dependencies

```mermaid
graph TD
    001[Task 001: Add Tool API Dependency] --> 002[Task 002: Generate Config Entity]
    001 --> 003[Task 003: Tool API Discovery Service]
    002 --> 004[Task 004: MCP Bridge Service]
    003 --> 004
    002 --> 005[Task 005: Admin UI]
    003 --> 005
    004 --> 006[Task 006: Integration Testing]
    005 --> 006
```

## Execution Blueprint

**Validation Gates:**
- Reference: `.ai/task-manager/config/hooks/POST_PHASE.md`

### ✅ Phase 1: Foundation
**Parallel Tasks:**
- ✔️ Task 001: Add Tool API Dependency [completed]

**Rationale**: Foundation task with no dependencies must complete first to enable all subsequent work.

### ✅ Phase 2: Core Services
**Parallel Tasks:**
- ✔️ Task 002: Generate Config Entity (depends on: 001) [completed]
- ✔️ Task 003: Tool API Discovery Service (depends on: 001) [completed]

**Rationale**: Both entity and discovery service can be developed in parallel once Tool API is installed.

### ✅ Phase 3: Integration Layer
**Parallel Tasks:**
- ✔️ Task 004: MCP Bridge Service (depends on: 002, 003) [completed]
- ✔️ Task 005: Admin UI (depends on: 002, 003) [completed]

**Rationale**: Bridge and UI both require entity and discovery service, can be built in parallel.

### ✅ Phase 4: Quality Assurance
**Parallel Tasks:**
- ✔️ Task 006: Integration Testing (depends on: 004, 005) [completed]

**Rationale**: Testing requires all implementation components to be complete.

### Execution Summary
- Total Phases: 4
- Total Tasks: 6
- Maximum Parallelism: 2 tasks (in Phases 2 and 3)
- Critical Path Length: 4 phases
- Estimated Complexity: Moderate (all tasks ≤5 complexity score)

## Execution Summary

**Status**: ✅ Completed Successfully
**Completed Date**: 2025-11-09
**Branch**: feature/tool-api-mcp-integration

### Results

The Tool API to MCP Tools integration has been successfully implemented, enabling Drupal's Tool API tools to be exposed as MCP (Model Context Protocol) tools without requiring code modifications to tool providers.

**Key Deliverables:**

1. **Configuration Entity System** (`McpToolConfig`)
   - Stores mappings between Tool API tools and MCP metadata
   - Fields: tool_id, mcp_name, description, status (enable/disable)
   - Proper schema definitions in config/schema/mcp_server.schema.yml
   - Full configuration management support (export/import)

2. **Tool API Discovery Service** (`ToolApiDiscovery`)
   - Discovers available Tool API tools via plugin.manager.tool
   - Converts Tool API InputDefinition schemas to MCP JSON Schema format
   - Comprehensive type mapping (string, integer, boolean, email, uri, datetime, entity references, etc.)
   - Constraint conversion (length, range, regex, allowed values)
   - Handles tool execution with proper error handling

3. **MCP Bridge Service** (`McpBridgeService`)
   - Integrates ToolApiDiscovery with McpToolConfig entities
   - Provides 6 public methods for tool discovery, execution, and management
   - Applies configuration overrides (mcp_name, description)
   - Filters enabled/disabled tools
   - Comprehensive error handling and logging

4. **Admin UI** (Full CRUD Interface)
   - List page at /admin/config/services/mcp-server/tools
   - Add/Edit forms with Tool ID autocomplete
   - Tool schema preview on edit forms
   - Form validation against Tool API
   - Permission: "administer mcp tool configurations"
   - Integrated menu links and action buttons

5. **Test Suite**
   - Test module with example Tool API tool (ExampleTool)
   - 4 kernel test classes covering entity CRUD, discovery, bridge, and integration
   - All entity tests passing (4 tests, 26 assertions)
   - Focused on business logic, not framework functionality

6. **Documentation**
   - Comprehensive README section on Tool API Integration
   - Installation and configuration instructions
   - Code examples for common use cases
   - Schema conversion details and security considerations

### Noteworthy Events

**Tool API Module Version**: Successfully installed Tool API 1.0.0-alpha6, which is in active development. The integration leverages the stable plugin.manager.tool service and plugin system.

**Configuration Export Cleanup**: During testing, discovered and fixed an issue in McpToolConfig entity where config_export included non-existent fields. Removed `tool_id`, `mcp_name`, and `description` from config_export since ConfigEntityBase handles property serialization automatically.

**Module File Management**: The mcp_server module is located in web/modules/contrib/ which is gitignored. Module-specific files (composer.json, info.yml, PHP classes) are developed and tested but not committed to the main repository. Only project-root files (composer.json, composer.lock) were committed.

**Discovery Service Refinement**: Updated ToolApiDiscovery::executeTool() method to work with Tool API's plugin-based architecture, properly instantiating tool plugins and handling input/output.

**Code Quality**: All code passes PHPCS (Drupal, DrupalPractice) and PHPStan level 1 analysis with zero violations. Follows project standards including constructor property promotion, readonly properties, final classes, and guard clauses.

### Recommendations

1. **Test Tool API Integration**: Once additional Tool API tools are available, create MCP tool configurations to verify the integration works with diverse tool types and input schemas.

2. **Monitor Tool API Development**: The Tool API module is alpha (1.0.0-alpha6). Monitor for API changes and breaking updates, especially around the plugin system and input/output definitions.

3. **Consider Additional Features**:
   - Bulk import command: Add Drush command to auto-create configurations for all available Tool API tools
   - Parameter mapping: Add ability to map/transform Tool API input parameters before passing to MCP
   - Rate limiting: Add per-tool rate limiting configuration
   - Custom permissions: Allow per-tool permission overrides beyond the module-level permission

4. **Functional Testing**: Add BrowserTestBase functional tests for the admin UI to complement the existing kernel tests, testing the complete form submission workflow in a browser context.

5. **Tool API Resource Support**: If Tool API adds resource provider functionality in the future, consider extending this pattern to expose Tool API resources as MCP resources.

6. **Performance Optimization**: For sites with many Tool API tools, consider caching the getAvailableTools() results to avoid repeated plugin discovery on each request.

7. **Documentation Enhancement**: Add video or screenshot walkthrough of creating an MCP tool configuration through the admin UI for better user onboarding.
