---
id: 2
group: "foundation"
dependencies: [1]
status: "completed"
created: "2025-11-09"
skills:
  - "drupal-backend"
  - "drush"
---
# Generate and Customize Configuration Entity

## Objective
Use Drush to generate the mcp_tool_config configuration entity boilerplate, then customize it with required fields and proper schema

## Skills Required
- drupal-backend: Drupal configuration entity development
- drush: Code generation with drush generate

## Acceptance Criteria
- [ ] Configuration entity generated via `drush generate config-entity`
- [ ] Entity class customized with tool_id, mcp_name, and description properties
- [ ] Schema definition added to config/schema/mcp_server.schema.yml
- [ ] Entity implements ConfigEntityInterface and extends ConfigEntityBase
- [ ] Status field configured (uses built-in from ConfigEntityBase)
- [ ] Code follows Drupal coding standards and mcp_server patterns

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Entity machine name: `mcp_tool_config`
- Entity label: "MCP Tool Configuration"
- Fields:
  - `id` (machine name) - inherited from ConfigEntityBase
  - `label` - inherited from ConfigEntityBase
  - `tool_id` (string, required) - Reference to Tool API tool
  - `mcp_name` (string, nullable) - Override for MCP tool name
  - `description` (text, nullable) - Override for description
  - `status` (boolean) - inherited from ConfigEntityBase, defaults to TRUE
- Storage: Configuration storage (default for config entities)
- Location: src/Entity/McpToolConfig.php

## Input Dependencies
- Task 1: Tool API module must be installed to reference Tool API concepts in code

## Output Artifacts
- src/Entity/McpToolConfig.php - Configuration entity class
- Updated config/schema/mcp_server.schema.yml - Schema definitions
- Entity is discoverable by Drupal's entity system

## Implementation Notes
<details>
<summary>Detailed Steps</summary>

### Step 1: Generate Entity via Drush

Run from /var/www/html:
```bash
vendor/bin/drush generate config-entity
```

Answer the prompts:
- Module machine name: `mcp_server`
- Entity type ID: `mcp_tool_config`
- Entity type label: `MCP Tool Configuration`
- Make the entity type revisionable?: No
- Add a "Changed" field?: No
- Generate entity classes?: Yes

### Step 2: Customize Generated Entity Class

Edit `src/Entity/McpToolConfig.php`:

1. Add custom properties using promoted constructor parameters (follow mcp_server patterns):
```php
public function __construct(
  array $values,
  private readonly string $tool_id = '',
  private readonly ?string $mcp_name = null,
  private readonly ?string $description = null,
) {
  parent::__construct($values, 'mcp_tool_config');
}
```

2. Add getter methods:
```php
public function getToolId(): string {
  return $this->tool_id;
}

public function getMcpName(): ?string {
  return $this->mcp_name ?? $this->tool_id;
}

public function getDescription(): ?string {
  return $this->description;
}
```

3. Ensure annotation has proper config_export:
```php
 *   config_export = {
 *     "id",
 *     "label",
 *     "tool_id",
 *     "mcp_name",
 *     "description",
 *     "status",
 *   },
```

### Step 3: Update Schema Definition

Add to `config/schema/mcp_server.schema.yml`:
```yaml
mcp_server.mcp_tool_config.*:
  type: config_entity
  label: 'MCP Tool Configuration'
  mapping:
    id:
      type: string
      label: 'Machine name'
    label:
      type: label
      label: 'Label'
    tool_id:
      type: string
      label: 'Tool API Tool ID'
    mcp_name:
      type: string
      label: 'MCP Tool Name Override'
      nullable: true
    description:
      type: text
      label: 'Description Override'
      nullable: true
    status:
      type: boolean
      label: 'Enabled'
```

### Step 4: Apply Code Quality

- Add proper type hints throughout
- Use constructor property promotion
- Make properties `private readonly` where appropriate
- Add comprehensive docblocks
- Remove trailing spaces
- Add newline at end of file

### Step 5: Rebuild Cache

```bash
cd /var/www/html && vendor/bin/drush cache:rebuild
```

### Step 6: Verify Entity Discovery

```bash
cd /var/www/html && vendor/bin/drush eval "print_r(\\Drupal::entityTypeManager()->getDefinition('mcp_tool_config'));"
```

**Important**: Follow the refactored code patterns from McpServerController (constructor promotion, readonly properties)
</details>
