---
id: 3
group: "integration"
dependencies: [1]
status: "completed"
created: "2025-11-09"
skills:
  - "drupal-backend"
---
# Create Tool API Discovery Service

## Objective
Build a service that discovers Tool API tools, converts their schemas to MCP format, and handles tool execution

## Skills Required
- drupal-backend: Drupal service development, dependency injection, type conversion

## Acceptance Criteria
- [x] ToolApiDiscovery service class created with proper dependency injection
- [x] Service registered in mcp_server.services.yml
- [x] Injects Tool API tool manager service
- [x] Provides method to list all available Tool API tools
- [x] Converts Tool API ContextDefinition schemas to MCP parameter format
- [x] Handles tool execution by delegating to Tool API
- [x] Includes comprehensive error handling and type validation
- [x] Follows mcp_server coding patterns (constructor promotion, readonly properties)

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Class: `Drupal\mcp_server\ToolApiDiscovery`
- Location: src/ToolApiDiscovery.php
- Inject: Tool API tool manager service (discover from Tool API module)
- Methods:
  - `getAvailableTools(): array` - Returns list of tool IDs and metadata
  - `getToolDefinition(string $toolId): ?array` - Gets full tool definition
  - `convertToMcpSchema(array $toolDefinition): array` - Converts schemas
  - `executeTool(string $toolId, array $parameters): mixed` - Executes tool
- Error handling: Wrap Tool API calls in try-catch, convert exceptions to MCP-friendly format

## Input Dependencies
- Task 1: Tool API module must be installed to inject tool manager service

## Output Artifacts
- src/ToolApiDiscovery.php - Discovery service class
- Service registered and available for injection into other services
- Abstraction layer for Tool API integration

## Implementation Notes
<details>
<summary>Detailed Steps</summary>

### Step 1: Research Tool API Service

First, discover what service Tool API provides for tool discovery:
```bash
cd /var/www/html && vendor/bin/drush eval "print_r(array_keys(\\Drupal::getContainer()->getServiceIds()));" | grep -i tool
```

Or inspect Tool API module's services.yml file if installed.

### Step 2: Create Service Class

Create `src/ToolApiDiscovery.php`:

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server;

use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * Service for discovering and integrating Tool API tools.
 */
final class ToolApiDiscovery {

  /**
   * Constructs a ToolApiDiscovery service.
   */
  public function __construct(
    private readonly object $toolManager,  // Update type based on actual Tool API service
    private readonly LoggerChannelInterface $logger,
  ) {}

  /**
   * Gets all available Tool API tools.
   *
   * @return array<string, array>
   *   Array of tool definitions keyed by tool ID.
   */
  public function getAvailableTools(): array {
    try {
      // TODO: Call Tool API manager to get tools
      // Example: return $this->toolManager->getDefinitions();
      return [];
    }
    catch (\\Exception $e) {
      $this->logger->error('Failed to retrieve Tool API tools: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Gets a specific tool definition.
   *
   * @param string $toolId
   *   The tool ID.
   *
   * @return array|null
   *   Tool definition or NULL if not found.
   */
  public function getToolDefinition(string $toolId): ?array {
    try {
      // TODO: Get single tool definition from Tool API
      return null;
    }
    catch (\\Exception $e) {
      $this->logger->error('Failed to retrieve tool @tool: @message', [
        '@tool' => $toolId,
        '@message' => $e->getMessage(),
      ]);
      return null;
    }
  }

  /**
   * Converts Tool API schema to MCP parameter schema.
   *
   * @param array $toolDefinition
   *   Tool API tool definition.
   *
   * @return array
   *   MCP-compatible parameter schema.
   */
  public function convertToMcpSchema(array $toolDefinition): array {
    // TODO: Map ContextDefinition types to MCP types
    // Common mappings:
    // - string -> string
    // - integer -> integer
    // - boolean -> boolean
    // - entity_reference -> string (entity ID)
    return [];
  }

  /**
   * Executes a Tool API tool.
   *
   * @param string $toolId
   *   The tool ID.
   * @param array $parameters
   *   Tool parameters.
   *
   * @return mixed
   *   Tool execution result.
   */
  public function executeTool(string $toolId, array $parameters): mixed {
    try {
      // TODO: Execute tool via Tool API manager
      // $result = $this->toolManager->execute($toolId, $parameters);
      return [];
    }
    catch (\\Exception $e) {
      $this->logger->error('Tool execution failed for @tool: @message', [
        '@tool' => $toolId,
        '@message' => $e->getMessage(),
      ]);
      throw new \\RuntimeException(
        "Tool execution failed: {$e->getMessage()}",
        0,
        $e
      );
    }
  }

}
```

### Step 3: Register Service

Add to `mcp_server.services.yml`:
```yaml
  mcp_server.tool_api_discovery:
    class: Drupal\mcp_server\ToolApiDiscovery
    arguments:
      - '@tool.manager'  # Update with actual Tool API service ID
      - '@logger.channel.mcp_server'
```

### Step 4: Implement Schema Conversion Logic

Research Tool API's ContextDefinition format and create comprehensive type mapping:
- Handle complex types (entity references, typed data)
- Support nested schemas if Tool API provides them
- Add validation for unsupported types
- Log warnings for type mismatches

### Step 5: Test Discovery

After implementing, test via Drush:
```bash
cd /var/www/html && vendor/bin/drush eval "\\Drupal::service('mcp_server.tool_api_discovery')->getAvailableTools();"
```

**Note**: You'll need to research Tool API's actual service names and API by:
1. Reading Tool API module code
2. Checking tool.services.yml
3. Looking at Tool API documentation
4. Examining example tool implementations
</details>
