---
id: 5
group: "ui"
dependencies: [2, 3]
status: "completed"
created: "2025-11-09"
skills:
  - "drupal-backend"
---
# Build Admin UI for MCP Tool Configurations

## Objective
Create a full CRUD admin interface for managing mcp_tool_config entities with Tool API tool autocomplete and schema preview

## Skills Required
- drupal-backend: Drupal Form API, entity forms, routing, controllers, permissions

## Acceptance Criteria
- [ ] List page created at /admin/config/services/mcp-server/tools
- [ ] Add form with Tool ID autocomplete field
- [ ] Edit form for modifying configurations
- [ ] Delete confirmation form
- [ ] Routes defined in mcp_server.routing.yml
- [ ] Permission `administer mcp tool configurations` added
- [ ] Forms validate Tool ID exists in Tool API
- [ ] Tool schema preview shown on add/edit forms
- [ ] Status field (enable/disable) functional
- [ ] Integrated with existing MCP Server admin menu

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Routes in mcp_server.routing.yml:
  - `entity.mcp_tool_config.collection` - List page
  - `entity.mcp_tool_config.add_form` - Add form
  - `entity.mcp_tool_config.edit_form` - Edit form
  - `entity.mcp_tool_config.delete_form` - Delete form
- Controllers/Forms in src/Entity/ or src/Form/
- Use Drupal's entity form classes as base
- Inject ToolApiDiscovery service for autocomplete
- Add menu link to existing MCP Server admin section

## Input Dependencies
- Task 2: mcp_tool_config entity must exist to create forms
- Task 3: ToolApiDiscovery service for tool lookup and validation

## Output Artifacts
- Routes in mcp_server.routing.yml
- Entity list builder class (optional, use default if sufficient)
- Entity form class extending ConfigEntityForm
- Permission defined in mcp_server.permissions.yml
- Menu links in mcp_server.links.menu.yml
- Functional admin UI accessible to authorized users

## Implementation Notes
<details>
<summary>Detailed Steps</summary>

### Step 1: Define Routes

Add to `mcp_server.routing.yml`:
```yaml
entity.mcp_tool_config.collection:
  path: '/admin/config/services/mcp-server/tools'
  defaults:
    _entity_list: 'mcp_tool_config'
    _title: 'MCP Tool Configurations'
  requirements:
    _permission: 'administer mcp tool configurations'

entity.mcp_tool_config.add_form:
  path: '/admin/config/services/mcp-server/tools/add'
  defaults:
    _entity_form: 'mcp_tool_config.add'
    _title: 'Add MCP Tool Configuration'
  requirements:
    _permission: 'administer mcp tool configurations'

entity.mcp_tool_config.edit_form:
  path: '/admin/config/services/mcp-server/tools/{mcp_tool_config}/edit'
  defaults:
    _entity_form: 'mcp_tool_config.edit'
    _title: 'Edit MCP Tool Configuration'
  requirements:
    _permission: 'administer mcp tool configurations'

entity.mcp_tool_config.delete_form:
  path: '/admin/config/services/mcp-server/tools/{mcp_tool_config}/delete'
  defaults:
    _entity_form: 'mcp_tool_config.delete'
    _title: 'Delete MCP Tool Configuration'
  requirements:
    _permission: 'administer mcp tool configurations'
```

### Step 2: Create Permission

Create or update `mcp_server.permissions.yml`:
```yaml
administer mcp tool configurations:
  title: 'Administer MCP tool configurations'
  description: 'Manage which Tool API tools are exposed as MCP tools'
  restrict access: true
```

### Step 3: Create Entity Form Class

If not generated by drush, create `src/Form/McpToolConfigForm.php`:

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\mcp_server\ToolApiDiscovery;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for MCP Tool Configuration entities.
 */
final class McpToolConfigForm extends EntityForm {

  /**
   * Constructs an McpToolConfigForm.
   */
  public function __construct(
    private readonly ToolApiDiscovery $toolApiDiscovery,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('mcp_server.tool_api_discovery')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);
    $entity = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => [$this, 'exist'],
      ],
      '#disabled' => !$entity->isNew(),
    ];

    // Tool ID autocomplete
    $form['tool_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tool API Tool ID'),
      '#default_value' => $entity->get('tool_id'),
      '#required' => TRUE,
      '#description' => $this->t('The ID of the Tool API tool to expose (e.g., module_name:tool_name)'),
      '#autocomplete_route_name' => 'mcp_server.tool_autocomplete',
    ];

    // MCP name override
    $form['mcp_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('MCP Tool Name'),
      '#default_value' => $entity->get('mcp_name'),
      '#description' => $this->t('Override the tool name for MCP. Leave empty to use the Tool ID.'),
    ];

    // Description override
    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $entity->get('description'),
      '#description' => $this->t('Override the tool description for AI context.'),
    ];

    // Status
    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $entity->status(),
    ];

    // Tool schema preview (if editing existing)
    if (!$entity->isNew() && $entity->get('tool_id')) {
      $toolDef = $this->toolApiDiscovery->getToolDefinition($entity->get('tool_id'));
      if ($toolDef) {
        $form['schema_preview'] = [
          '#type' => 'details',
          '#title' => $this->t('Tool Schema'),
          '#open' => FALSE,
          'content' => [
            '#markup' => '<pre>' . print_r($toolDef, TRUE) . '</pre>',
          ],
        ];
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $toolId = $form_state->getValue('tool_id');
    if ($toolId) {
      $toolDef = $this->toolApiDiscovery->getToolDefinition($toolId);
      if (!$toolDef) {
        $form_state->setErrorByName('tool_id', $this->t('Tool ID "@tool" not found in Tool API.', [
          '@tool' => $toolId,
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);
    $entity = $this->entity;

    $message_args = ['%label' => $entity->label()];
    $message = $result === SAVED_NEW
      ? $this->t('Created new MCP tool configuration %label.', $message_args)
      : $this->t('Updated MCP tool configuration %label.', $message_args);
    $this->messenger()->addStatus($message);

    $form_state->setRedirectUrl($entity->toUrl('collection'));
    return $result;
  }

  /**
   * Checks if an entity with the given ID exists.
   */
  public function exist(string $id): bool {
    $entity = $this->entityTypeManager
      ->getStorage('mcp_tool_config')
      ->getQuery()
      ->condition('id', $id)
      ->accessCheck(FALSE)
      ->execute();
    return (bool) $entity;
  }

}
```

### Step 4: Create Autocomplete Route and Controller

Add route to `mcp_server.routing.yml`:
```yaml
mcp_server.tool_autocomplete:
  path: '/mcp-server/tool-autocomplete'
  defaults:
    _controller: '\\Drupal\\mcp_server\\Controller\\ToolAutocompleteController::handleAutocomplete'
  requirements:
    _permission: 'administer mcp tool configurations'
```

Create `src/Controller/ToolAutocompleteController.php` to provide autocomplete suggestions from ToolApiDiscovery.

### Step 5: Add Menu Links

Create or update `mcp_server.links.menu.yml`:
```yaml
mcp_server.tool_configs:
  title: 'MCP Tool Configurations'
  description: 'Manage Tool API tools exposed as MCP tools'
  parent: system.admin_config_services
  route_name: entity.mcp_tool_config.collection
  weight: 10
```

### Step 6: Configure Entity Annotation

Ensure the McpToolConfig entity class has proper annotations for list_builder, form handlers, and route templates.

### Step 7: Clear Cache and Test

```bash
cd /var/www/html && vendor/bin/drush cache:rebuild
```

Navigate to /admin/config/services/mcp-server/tools and verify the UI works.

**Note**: You may need to create a custom list builder if the default entity list doesn't meet requirements (e.g., for status filtering or bulk operations).
</details>
