---
id: 4
group: "enforcement"
dependencies: [1, 3]
status: "completed"
created: "2025-11-09"
completed: "2025-11-09"
skills:
  - drupal-backend
  - dependency-injection
---
# Implement authentication enforcement in McpBridgeService

## Objective
Add authentication checking logic to McpBridgeService::executeMcpTool() that enforces authentication requirements based on the tool's configured mode.

## Skills Required
- Drupal service architecture
- Dependency injection
- CurrentUser service integration

## Acceptance Criteria
- [ ] Inject CurrentUser service into McpBridgeService constructor
- [ ] Update service definition in mcp_server.services.yml
- [ ] Add authentication check before tool execution in executeMcpTool()
- [ ] Implement fail-fast pattern (disabled → fast path)
- [ ] Throw AuthenticationRequiredException for required mode + anonymous user
- [ ] Preserve existing functionality for optional and disabled modes

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
<details>
<summary>Implementation Details</summary>

**File Locations:**
- Service: `/var/www/html/web/modules/contrib/mcp_server/src/McpBridgeService.php:186`
- Service config: `/var/www/html/web/modules/contrib/mcp_server/mcp_server.services.yml`

**Constructor Injection:**
```php
/**
 * The current user service.
 *
 * @var \Drupal\Core\Session\AccountProxyInterface
 */
protected $currentUser;

public function __construct(
  // ... existing parameters
  AccountProxyInterface $current_user
) {
  // ... existing assignments
  $this->currentUser = $current_user;
}
```

**Service Definition Update:**
```yaml
mcp_server.mcp_bridge:
  class: Drupal\mcp_server\McpBridgeService
  arguments:
    - '@entity_type.manager'
    - '@plugin.manager.tool'
    - '@current_user'  # Add this
```

**Authentication Check in executeMcpTool():**
```php
public function executeMcpTool(string $tool_name, array $arguments = []): mixed {
  // Load tool config
  $tool_config = $this->entityTypeManager
    ->getStorage('mcp_tool_config')
    ->load($tool_name);

  if (!$tool_config) {
    throw new \InvalidArgumentException("Unknown tool: {$tool_name}");
  }

  // Fast path for disabled mode
  if ($tool_config->getAuthenticationMode() === 'disabled') {
    return $this->executeToolApiTool($tool_name, $arguments);
  }

  // Check authentication for required mode
  if ($tool_config->isAuthenticationRequired() && $this->currentUser->isAnonymous()) {
    throw new AuthenticationRequiredException(
      $tool_name,
      $tool_config->getAuthenticationMode()
    );
  }

  // Proceed with tool execution (required + authenticated, or optional)
  return $this->executeToolApiTool($tool_name, $arguments);
}
```

**Add use statements:**
```php
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\mcp_server\Exception\AuthenticationRequiredException;
```
</details>

## Input Dependencies
- Task 1: McpToolConfig with getAuthenticationMode() and isAuthenticationRequired()
- Task 3: AuthenticationRequiredException class

## Output Artifacts
- McpBridgeService with authentication enforcement
- Service properly injecting CurrentUser
- Authentication logic ready for controller integration
