---
id: 5
group: "enforcement"
dependencies: [3, 4]
status: "completed"
created: "2025-11-09"
completed: "2025-11-09"
skills:
  - drupal-backend
  - json-rpc
---
# Add authentication error handling to McpServerController

## Objective
Update McpServerController to catch AuthenticationRequiredException and return proper JSON-RPC error responses with HTTP 401 status and WWW-Authenticate header.

## Skills Required
- Drupal controller patterns
- JSON-RPC error formatting
- HTTP response headers

## Acceptance Criteria
- [ ] Catch AuthenticationRequiredException in controller
- [ ] Return JSON-RPC error with code -32001
- [ ] Include tool name and authentication_mode in error data
- [ ] Set HTTP status to 401 Unauthorized
- [ ] Add WWW-Authenticate: Bearer realm="mcp_server" header
- [ ] Maintain existing error handling for other exceptions

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
<details>
<summary>Implementation Details</summary>

**File Location:**
Find and update the controller that calls McpBridgeService::executeMcpTool() (likely McpServerController or similar in `/var/www/html/web/modules/contrib/mcp_server/src/Controller/`)

**Error Handling Pattern:**
```php
use Drupal\mcp_server\Exception\AuthenticationRequiredException;
use Symfony\Component\HttpFoundation\JsonResponse;

// In the method that executes tools:
try {
  $result = $this->mcpBridgeService->executeMcpTool($tool_name, $arguments);
  // ... return success response
} catch (AuthenticationRequiredException $e) {
  $response = new JsonResponse([
    'jsonrpc' => '2.0',
    'error' => [
      'code' => -32001,
      'message' => 'Authentication required',
      'data' => [
        'tool' => $e->getToolName(),
        'authentication_mode' => $e->getAuthenticationMode(),
      ],
    ],
    'id' => $request_id ?? NULL,
  ], 401);

  $response->headers->set('WWW-Authenticate', 'Bearer realm="mcp_server"');

  return $response;
} catch (\Exception $e) {
  // ... existing error handling
}
```

**Error Code Documentation:**
Add a comment documenting that -32001 is reserved for authentication failures, and -32002 is reserved for future authorization failures (PRD 2).
</details>

## Input Dependencies
- Task 3: AuthenticationRequiredException class
- Task 4: McpBridgeService throwing the exception

## Output Artifacts
- Controller with proper authentication error handling
- JSON-RPC compliant error responses
- OAuth 2.0 compliant WWW-Authenticate headers
