---
id: 5
group: "admin-interface"
dependencies: [1, 2]
status: "completed"
created: "2025-11-10"
skills:
  - drupal-backend
  - php
---
# Add Scope Selection UI to McpToolConfigForm

## Objective
Implement admin UI for selecting OAuth scopes in McpToolConfigForm with conditional display based on authentication_mode, loading available scopes from oauth2_scope entities.

## Skills Required
- **drupal-backend**: Drupal Form API, conditional display with #states, entity storage
- **php**: Form building methods and array manipulation

## Acceptance Criteria
- [ ] Scopes field added to McpToolConfigForm as checkboxes
- [ ] getAvailableScopes() method loads oauth2_scope entities
- [ ] Form options show scope label and description
- [ ] Conditional display using #states (visible when mode is required/optional)
- [ ] Form submission filters unchecked values and saves selected scopes
- [ ] Error handling for scope loading failures
- [ ] Help text explains scope behavior

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- File: `src/Form/McpToolConfigForm.php`
- Form element type: checkboxes
- Dependencies: EntityTypeManager for loading oauth2_scope entities
- Conditional logic: #states with authentication_mode field

## Input Dependencies
- Simple OAuth module with oauth2_scope entities (Task 1)
- McpToolConfig entity with scopes field and methods (Task 2)

## Output Artifacts
- Admin UI for scope selection
- Integration with entity save/load workflow
- User-friendly scope management interface

## Implementation Notes
<details>
<summary>Detailed Implementation Steps</summary>

1. **Add scopes field to buildForm()**:
   ```php
   $form['scopes'] = [
     '#type' => 'checkboxes',
     '#title' => $this->t('Required OAuth Scopes'),
     '#description' => $this->t('Select OAuth scopes required to execute this tool. Leave empty to allow any authenticated user. Only applicable when authentication mode is "required" or "optional".'),
     '#options' => $this->getAvailableScopes(),
     '#default_value' => $entity->getScopes(),
     '#states' => [
       'visible' => [
         ':input[name="authentication_mode"]' => [
           ['value' => 'required'],
           ['value' => 'optional'],
         ],
       ],
     ],
   ];
   ```

2. **Implement getAvailableScopes() helper**:
   ```php
   protected function getAvailableScopes(): array {
     try {
       $scope_storage = $this->entityTypeManager->getStorage('oauth2_scope');
       $scopes = $scope_storage->loadMultiple();

       $options = [];
       foreach ($scopes as $scope) {
         $options[$scope->id()] = sprintf(
           '%s - %s',
           $scope->label(),
           $scope->get('description')
         );
       }

       return $options;
     }
     catch (\Exception $e) {
       $this->logger('mcp_server')->error(
         'Failed to load OAuth scopes: @message',
         ['@message' => $e->getMessage()]
       );
       return [];
     }
   }
   ```

3. **Update submitForm() to save scopes**:
   ```php
   // Filter out unchecked values (checkboxes return 0 for unchecked)
   $scopes = array_filter($form_state->getValue('scopes'));
   $entity->setScopes(array_keys($scopes));
   ```

4. **Conditional display behavior**:
   - Field visible when authentication_mode = 'required' OR 'optional'
   - Field hidden when authentication_mode = 'disabled'
   - Uses Drupal's #states Form API for client-side toggling

5. **Error handling**:
   - If oauth2_scope storage fails to load, log error and return empty array
   - Empty options array results in no checkboxes shown (graceful degradation)
   - Form still saves successfully even if scope loading fails

6. **Testing verification**:
   - Visit /admin/config/services/mcp-server/[tool]/edit
   - Change authentication_mode and verify field visibility
   - Select scopes, save, reload, verify persistence
   - Export config and verify scopes appear correctly
</details>
