---
id: 6
group: "scope-validation"
dependencies: [2, 3, 4]
status: "completed"
created: "2025-11-10"
skills:
  - drupal-backend
  - php
---
# Integrate Scope Validation into McpBridgeService

## Objective
Extend McpBridgeService to validate OAuth scopes during tool execution, enforcing scope requirements for required mode and logging warnings for optional mode.

## Skills Required
- **drupal-backend**: Drupal service modification, dependency injection updates
- **php**: Control flow logic and exception handling

## Acceptance Criteria
- [ ] OAuthScopeValidator injected into McpBridgeService constructor
- [ ] executeMcpTool() updated with scope validation logic
- [ ] validateToolScopes() helper method created
- [ ] Required mode throws InsufficientScopeException when scopes missing
- [ ] Optional mode logs warnings when scopes missing but allows execution
- [ ] Disabled mode skips scope validation entirely
- [ ] Empty scopes array allows any authenticated user

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- File: `src/McpBridgeService.php`
- New dependency: OAuthScopeValidator service
- Methods: executeMcpTool() (modify), validateToolScopes() (create)
- Exception throwing: InsufficientScopeException

## Input Dependencies
- McpToolConfig entity with scopes field (Task 2)
- OAuthScopeValidator service (Task 3)
- InsufficientScopeException class (Task 4)

## Output Artifacts
- Scope enforcement in tool execution flow
- Integration with existing authentication from PRD 1
- Complete authorization system (authentication + scope validation)

## Implementation Notes
<details>
<summary>Detailed Implementation Steps</summary>

1. **Add OAuthScopeValidator to constructor**:
   ```php
   public function __construct(
     private readonly ToolApiDiscovery $toolApiDiscovery,
     private readonly EntityTypeManagerInterface $entityTypeManager,
     private readonly LoggerChannelInterface $logger,
     private readonly OAuthScopeValidator $scopeValidator,
     private readonly CurrentUserInterface $currentUser,
   ) {}
   ```

2. **Update executeMcpTool() method**:
   Follow the implementation from plan's Component 6:
   - Load tool config
   - Check authentication_mode
   - Disabled: skip all checks
   - Required: check isAnonymous, validate scopes with throw on failure
   - Optional: check isAnonymous, validate scopes without throw
   - Execute tool

3. **Create validateToolScopes() helper**:
   ```php
   protected function validateToolScopes(
     McpToolConfig $config,
     string $mcpName,
     bool $throwOnFailure,
   ): void {
     $required_scopes = $config->getScopes();

     // No restrictions if empty
     if (empty($required_scopes)) {
       return;
     }

     $token_scopes = $this->scopeValidator->extractTokenScopes();
     $validation = $this->scopeValidator->validateScopes($required_scopes, $token_scopes);

     if (!$validation->isValid) {
       if ($throwOnFailure) {
         throw new InsufficientScopeException(...);
       } else {
         $this->logger->warning(...);
       }
     }
   }
   ```

4. **Flow logic by authentication mode**:
   - **disabled**: No authentication or scope checks at all
   - **required + anonymous**: Throw AuthenticationRequiredException (401) - from PRD 1
   - **required + authenticated**: Validate scopes, throw InsufficientScopeException (403) if missing
   - **optional + anonymous**: Execute tool immediately (from PRD 1)
   - **optional + authenticated**: Validate scopes, log warning if missing, execute anyway

5. **Empty scopes array behavior**:
   - If config.getScopes() returns [], validateToolScopes returns early
   - This means: authenticated but no scope restrictions
   - Different from: no scopes in token (which would fail validation)

6. **Error message format for warnings**:
   ```
   'User lacks required scopes for tool @tool but optional auth mode allows execution.
    Required: @required, Missing: @missing, Current: @current'
   ```

7. **Integration points**:
   - Calls into PRD 1's authentication check (isAnonymous)
   - Throws PRD 1's AuthenticationRequiredException for unauthenticated
   - Throws new InsufficientScopeException for insufficient scopes
   - Uses loadToolConfig() from PRD 1
</details>
