---
id: 7
group: "http-transport"
dependencies: [4, 6]
status: "completed"
created: "2025-11-10"
skills:
  - drupal-backend
  - php
---
# Add InsufficientScopeException Handling to Controllers

## Objective
Update McpServerController and stdio transport to catch InsufficientScopeException and return properly formatted 403 error responses with scope details.

## Skills Required
- **drupal-backend**: Drupal controller development, JSON-RPC error formatting
- **php**: Exception handling and HTTP response construction

## Acceptance Criteria
- [ ] McpServerController catches InsufficientScopeException
- [ ] 403 status code returned for scope errors
- [ ] JSON-RPC error response includes scope details from getErrorData()
- [ ] WWW-Authenticate header set to 'Bearer error="insufficient_scope"'
- [ ] STDIO transport handles exception consistently
- [ ] Error response matches JSON-RPC 2.0 specification

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- File: `src/Controller/McpServerController.php`
- File: `src/Commands/McpServerCommands.php` (if STDIO transport exists)
- Error code: -32003 (custom code for insufficient scope)
- HTTP status: 403 Forbidden
- Header: WWW-Authenticate with insufficient_scope error

## Input Dependencies
- InsufficientScopeException class (Task 4)
- McpBridgeService throwing exception (Task 6)

## Output Artifacts
- HTTP controller with scope error handling
- STDIO transport with scope error handling
- Complete error response workflow for scope validation failures

## Implementation Notes
<details>
<summary>Detailed Implementation Steps</summary>

1. **Add catch block to McpServerController**:
   ```php
   catch (InsufficientScopeException $e) {
     return new JsonResponse([
       'jsonrpc' => '2.0',
       'error' => [
         'code' => -32003,
         'message' => 'Insufficient OAuth scopes',
         'data' => $e->getErrorData(),
       ],
       'id' => $request_id,
     ], 403, [
       'WWW-Authenticate' => 'Bearer error="insufficient_scope"',
     ]);
   }
   ```

2. **Error code rationale**:
   - -32000 to -32099 reserved for implementation-defined server errors
   - -32003 chosen for insufficient scope errors
   - Distinct from -32001 (authentication required from PRD 1)

3. **WWW-Authenticate header**:
   - OAuth 2.0 specification requires this header for 401/403 responses
   - Format: `Bearer error="insufficient_scope"`
   - Indicates to client why authorization failed

4. **JSON-RPC response structure**:
   ```json
   {
     "jsonrpc": "2.0",
     "error": {
       "code": -32003,
       "message": "Insufficient OAuth scopes",
       "data": {
         "tool": "widget.create",
         "required_scopes": ["widget:write", "widget:create"],
         "missing_scopes": ["widget:create"],
         "current_scopes": ["widget:write"]
       }
     },
     "id": <original request id>
   }
   ```

5. **STDIO transport handling**:
   - Similar catch block in McpServerCommands
   - Format error response for STDIO (line-delimited JSON)
   - Ensure consistent error structure across transports

6. **Error handling order**:
   - Place InsufficientScopeException catch AFTER AuthenticationRequiredException
   - More specific exceptions should be caught first
   - Generic Exception catch should remain last

7. **Testing verification**:
   - Make request with insufficient scopes
   - Verify 403 status code
   - Verify WWW-Authenticate header present
   - Verify error data contains required_scopes, missing_scopes, current_scopes
   - Verify request_id matches original request
</details>
