---
id: 4
summary: "Implement OAuth metadata discovery for MCP Server to enable automatic client discovery of authentication requirements and supported scopes"
created: 2025-11-09
---

# Plan: OAuth Metadata Discovery & Client Integration

## Original Work Order

> @.ai/task-manager/scratch/authentication-prd-3-discovery.md
>
> Implement PRD 3: MCP Server OAuth Metadata & Discovery

## Plan Clarifications

| Question | Answer |
|----------|--------|
| Should I create a plan that assumes PRD 1 & 2 are implemented first? | Yes - Plan PRD 3 only (assumes PRD 1 & 2 done) |
| Is simple_oauth_server_metadata module installed? | Yes - it's part of the simple_oauth_21 package |

## Executive Summary

This plan implements OAuth metadata discovery to complete the MCP Server authentication infrastructure. It enables MCP clients to automatically discover authentication requirements, supported OAuth scopes, and authorization server endpoints through standards-compliant RFC 9728 metadata. The implementation aggregates OAuth scopes from all configured MCP tools and exposes them via well-known endpoints, while adding per-tool authentication metadata to the tools/list response.

The approach leverages Drupal's event system to extend the existing simple_oauth_server_metadata module's RFC 9728 implementation, ensuring compliance with OAuth standards while providing MCP-specific extensions. Aggressive caching ensures discovery metadata is performant, with selective invalidation when tool configurations change.

Key benefits include: automatic client configuration, reduced manual setup, standards compliance, efficient caching, and enhanced developer experience for MCP client integration.

## Context

###  Current State

After implementing PRD 1 (authentication infrastructure) and PRD 2 (scope management), the MCP Server has:
- McpToolConfig entity with `authentication_mode` field (required/optional/disabled)
- McpToolConfig entity with `scopes` field (array of scope IDs per tool)
- OAuth token validation via Simple OAuth module
- Per-tool authentication enforcement

However, MCP clients cannot:
- Discover which scopes are supported by the server
- Learn authentication requirements before making requests
- Programmatically find OAuth endpoints (authorization_endpoint, token_endpoint)
- Understand per-tool authentication and scope requirements

The simple_oauth_server_metadata module provides RFC 9728 endpoints (/.well-known/oauth-protected-resource and /.well-known/oauth-authorization-server) but only includes base OAuth metadata - it doesn't know about MCP tool scopes.

### Target State

After implementation:

**For MCP Clients:**
- Can GET `/.well-known/oauth-protected-resource` to discover supported scopes
- Can GET `/.well-known/oauth-authorization-server` to find OAuth endpoints
- Can parse tools/list response to see per-tool authentication requirements
- Can automatically request correct scopes during OAuth flow
- Can cache discovery metadata for performance

**For Site Administrators:**
- Clients automatically discover OAuth configuration
- Minimal manual client configuration needed
- Clear visibility into which scopes are exposed

**Technical Capabilities:**
- Scope aggregation service collects all unique scopes from enabled tools
- Event subscriber extends RFC 9728 metadata with MCP scopes
- Tool metadata includes authorization field with mode and required scopes
- Aggressive caching with selective invalidation on config changes

### Background

The Model Context Protocol specification doesn't define a discovery mechanism for authentication, so we're extending it with OAuth 2.0 standards. RFC 9728 (OAuth 2.0 Protected Resource Metadata) provides a standardized way for clients to discover:
- Which resource server they're accessing
- Which authorization servers protect it
- Which scopes are supported
- How to submit bearer tokens

By combining RFC 9728 with custom tool metadata, we enable both protocol-level discovery (standard OAuth) and tool-level discovery (MCP-specific).

The simple_oauth_server_metadata module (part of simple_oauth_21 package) provides the foundation by implementing RFC 9728 endpoints and an event system for extending metadata. We subscribe to its events to inject MCP-specific scope data.

## Technical Implementation Approach

```mermaid
graph TB
    A[MCP Client] -->|1. GET /.well-known/oauth-protected-resource| B[simple_oauth_server_metadata]
    B -->|2. Dispatch BUILD event| C[ResourceMetadataSubscriber]
    C -->|3. Get scopes| D[OAuthScopeDiscoveryService]
    D -->|4. Load configs| E[McpToolConfig Storage]
    E -->|5. Return enabled configs| D
    D -->|6. Aggregate & cache scopes| F[Cache Backend]
    D -->|7. Return scopes| C
    C -->|8. Add to metadata| B
    B -->|9. Return JSON| A

    G[McpToolConfig Save] -->|Invalidate| F

    A -->|GET /mcp/tools/list| H[ToolApiDiscovery]
    H -->|Load config| E
    H -->|Add authorization field| A
```

### Component 1: OAuth Scope Discovery Service
**Objective**: Aggregate and cache unique OAuth scopes from all enabled MCP tool configurations

**Service Class**: `Drupal\mcp_server\Service\OAuthScopeDiscoveryService`

**Responsibilities**:
1. Load all McpToolConfig entities filtered by `status === TRUE`
2. Extract scopes array from each entity using `getScopes()` method
3. Merge all scopes into flat array
4. Deduplicate using `array_unique()`
5. Sort alphabetically for consistent output
6. Cache result with `mcp_server:discovery` cache tag
7. Return array of unique scope IDs

**Caching Strategy**:
- Cache ID: `mcp_server:scopes_supported`
- Cache tags: `['mcp_server:discovery']`
- Cache lifetime: `Cache::PERMANENT` (invalidate via tags)
- Cache backend: Default Drupal cache

**Service Definition** (`mcp_server.services.yml`):
```yaml
mcp_server.oauth_scope_discovery:
  class: Drupal\mcp_server\Service\OAuthScopeDiscoveryService
  arguments:
    - '@entity_type.manager'
    - '@cache.default'
```

**Key Method**:
```php
public function getScopesSupported(): array {
  // Check cache first
  // Load enabled configs
  // Extract and merge scopes
  // Deduplicate and sort
  // Cache result
  // Return scopes
}
```

### Component 2: Resource Metadata Event Subscriber
**Objective**: Extend RFC 9728 Protected Resource Metadata with MCP tool scopes

**Service Class**: `Drupal\mcp_server\EventSubscriber\ResourceMetadataSubscriber`

**Event Subscription**:
- Event: `ResourceMetadataEvents::BUILD` (from simple_oauth_server_metadata)
- Priority: 0 (default)
- Method: `onBuild(ResourceMetadataEvent $event)`

**Implementation Logic**:
1. Call `OAuthScopeDiscoveryService::getScopesSupported()`
2. If scopes array is not empty:
   - Get existing `scopes_supported` from event metadata (if any)
   - Merge MCP scopes with existing scopes
   - Deduplicate merged array with `array_unique()`
   - Set `$event->metadata['scopes_supported']` to merged array
3. If scopes array is empty, do nothing (don't add empty field)

**Service Definition** (`mcp_server.services.yml`):
```yaml
mcp_server.resource_metadata_subscriber:
  class: Drupal\mcp_server\EventSubscriber\ResourceMetadataSubscriber
  arguments:
    - '@mcp_server.oauth_scope_discovery'
  tags:
    - { name: event_subscriber }
```

**Result**: `GET /.well-known/oauth-protected-resource` returns:
```json
{
  "resource": "https://example.com",
  "authorization_servers": ["https://example.com"],
  "bearer_methods_supported": ["header"],
  "scopes_supported": [
    "content:read",
    "content:write",
    "widget:delete"
  ],
  "resource_documentation": "https://example.com/admin/config/services/mcp-server"
}
```

### Component 3: Tool Metadata Authorization Extension
**Objective**: Add per-tool authentication metadata to tools/list endpoint

**Target Service**: `Drupal\mcp_server\ToolApiDiscovery` (existing service)

**Modification Approach**:
Extend the `getToolDefinition()` method to include authorization metadata when building tool responses.

**Logic**:
1. Build base tool metadata (name, description, inputSchema)
2. Load McpToolConfig by tool_id
3. If config exists AND `requiresAuthentication()` returns TRUE:
   - Add `authorization` field to metadata with:
     - `method`: "oauth2"
     - `mode`: value from `getAuthenticationMode()` (required/optional)
     - `requiredScopes`: array from `getScopes()`
4. If config doesn't exist OR authentication_mode is 'disabled', omit authorization field entirely

**Enhanced Tool Response**:
```json
{
  "name": "widget.create",
  "description": "Create a new widget",
  "inputSchema": {...},
  "authorization": {
    "method": "oauth2",
    "mode": "required",
    "requiredScopes": ["widget:write", "widget:create"]
  }
}
```

**Integration Point**: Modify ToolApiDiscovery service, no new service needed.

### Component 4: Cache Invalidation Hooks
**Objective**: Invalidate discovery cache when tool configurations change

**Entity Hooks in McpToolConfig**:

**Hook: postSave()**
```php
public function postSave(EntityStorageInterface $storage) {
  parent::postSave($storage);
  Cache::invalidateTags(['mcp_server:discovery']);
}
```

**Hook: postDelete()**
```php
public static function postDelete(EntityStorageInterface $storage, array $entities) {
  parent::postDelete($storage, $entities);
  Cache::invalidateTags(['mcp_server:discovery']);
}
```

**Invalidation Triggers**:
- McpToolConfig entity created
- McpToolConfig entity updated (any field)
- McpToolConfig entity deleted
- Manual cache clear (`drush cache:rebuild`)

**Affected Caches**:
- OAuthScopeDiscoveryService cached scopes
- tools/list responses (if cached)
- Any other data tagged with `mcp_server:discovery`

## Risk Considerations and Mitigation Strategies

### Technical Risks

- **Performance Impact of Scope Aggregation**: Loading all McpToolConfig entities and extracting scopes could be slow with many tools
    - **Mitigation**: Aggressive caching with `Cache::PERMANENT` ensures aggregation only happens on cache miss; cache tags enable selective invalidation; benchmark with 100+ tools to verify performance

- **Cache Staleness**: Cached metadata could become stale if invalidation fails
    - **Mitigation**: Proper cache tag invalidation in entity hooks; cache tags correctly configured in service; manual cache clear as fallback; comprehensive testing of invalidation scenarios

- **RFC 9728 Compliance**: Incorrect metadata format could break standards-compliant clients
    - **Mitigation**: Follow RFC 9728 specification exactly; use event system to extend, not override; merge with existing scopes_supported if present; validate against RFC requirements

### Implementation Risks

- **Event Subscriber Not Called**: If event subscription fails, scopes won't be added to metadata
    - **Mitigation**: Use Drupal's event_subscriber tag correctly; test event subscription in kernel test; verify subscriber is registered in container; add logging to confirm subscriber execution

- **Dependency on PRD 1 & 2**: Missing authentication_mode or scopes fields will cause fatal errors
    - **Mitigation**: Verify PRD 1 & 2 implementation before starting; add defensive checks for field existence; document dependencies clearly; test with actual PRD 1 & 2 implementation

### Integration Risks

- **simple_oauth_server_metadata API Changes**: Module updates could break event subscription
    - **Mitigation**: Pin module version in composer.json; monitor module for updates; add version detection if needed; document supported versions

- **Tool API Discovery Conflicts**: Modifying ToolApiDiscovery could conflict with other extensions
    - **Mitigation**: Use conditional field addition (only when auth is configured); avoid modifying core tool discovery logic; test with multiple tool configurations

## Success Criteria

### Primary Success Criteria

1. **Protected Resource Metadata includes MCP scopes**: `GET /.well-known/oauth-protected-resource` returns `scopes_supported` field with all unique scopes from enabled tools
2. **Scope aggregation is cached**: `getScopesSupported()` hits cache on repeat calls, performs < 1ms on cached requests
3. **Cache invalidates correctly**: Saving/deleting McpToolConfig entity invalidates discovery cache within same request
4. **Tool metadata includes authorization**: tools/list response includes `authorization` field for tools with authentication_mode !== 'disabled'
5. **MCP clients can discover requirements**: External client can parse metadata and configure OAuth correctly

### Quality Assurance Metrics

1. **Test coverage**: 100% coverage for OAuthScopeDiscoveryService and ResourceMetadataSubscriber
2. **Performance**: Scope aggregation adds < 5ms overhead to metadata endpoint (uncached), < 1ms (cached)
3. **RFC 9728 compliance**: Metadata response validates against RFC 9728 JSON schema
4. **Cache efficiency**: Cache hit rate > 95% in production scenarios
5. **Code standards**: All code passes PHPCS Drupal/DrupalPractice and PHPStan level 1

## Resource Requirements

### Development Skills

- Drupal service development and dependency injection
- Drupal event subscriber implementation
- Drupal cache API and cache tag invalidation
- Entity hooks (postSave, postDelete)
- RFC 9728 OAuth Protected Resource Metadata specification
- simple_oauth_server_metadata module API
- PHPUnit testing (unit, kernel, functional)

### Technical Infrastructure

- simple_oauth_server_metadata module (part of simple_oauth_21 package)
- Drupal cache backend (default or Redis)
- PHPUnit for testing
- PHPCS and PHPStan for code quality
- MCP Inspector or custom client for testing discovery

### External Dependencies

- PRD 1 implementation (authentication_mode field, requiresAuthentication() method)
- PRD 2 implementation (scopes field, getScopes() method)
- simple_oauth_server_metadata module and its events
- RFC 9728 specification compliance

## Integration Strategy

This plan integrates with existing systems:

**With simple_oauth_server_metadata**:
- Subscribe to ResourceMetadataEvents::BUILD event
- Extend metadata, don't override
- Respect existing scopes_supported field

**With McpToolConfig Entity**:
- Read authentication_mode and scopes fields
- Add cache invalidation hooks
- Don't modify existing functionality

**With ToolApiDiscovery Service**:
- Extend tool building to add authorization field
- Conditional inclusion based on authentication mode
- Don't break existing tool discovery

**With Drupal Cache System**:
- Use standard cache API
- Use cache tags for selective invalidation
- Follow Drupal caching best practices

## Implementation Order

The implementation should proceed in this logical sequence:

1. **Create OAuthScopeDiscoveryService** - Foundation service for scope aggregation
2. **Add cache invalidation to McpToolConfig** - Ensure cache stays fresh when configs change
3. **Create ResourceMetadataSubscriber** - Extend RFC 9728 metadata with scopes
4. **Extend ToolApiDiscovery** - Add authorization metadata to tools/list
5. **Write comprehensive tests** - Unit, kernel, and functional tests
6. **Verify RFC 9728 compliance** - Test against specification requirements

This order ensures each component can be tested independently before integration, and cache infrastructure is in place before services that depend on it.

## Notes

**Assumptions**:
- PRD 1 and PRD 2 are fully implemented and tested before starting
- simple_oauth_server_metadata module is installed and configured
- OAuth authorization and token endpoints are functional
- MCP clients support RFC 9728 metadata discovery

**Future Enhancements** (out of scope):
- Scope descriptions in metadata
- Scope grouping/categorization
- Tool-level documentation URLs
- Admin UI for viewing aggregated scopes
- Scope usage analytics

**Testing Strategy**:
- Unit tests for scope aggregation logic
- Kernel tests for event subscription and cache invalidation
- Functional tests for metadata endpoints
- Integration tests for complete OAuth flow

**Documentation Needs**:
- Update README with OAuth discovery section
- Add examples of metadata responses
- Document cache invalidation strategy
- Provide client integration examples

## Task Dependencies

```mermaid
graph TD
    001[Task 001: OAuth Scope Discovery Service]
    002[Task 002: Cache Invalidation Hooks]
    003[Task 003: Resource Metadata Subscriber]
    004[Task 004: Extend Tool API Discovery]
    005[Task 005: Integration Tests]
    
    001 --> 003
    001 --> 005
    002 --> 005
    003 --> 005
    004 --> 005
```

## Execution Blueprint

**Validation Gates:**
- Reference: `.ai/task-manager/config/hooks/POST_PHASE.md`

### ✅ Phase 1: Foundation Infrastructure
**Parallel Tasks:**
- ✔️ Task 001: OAuth Scope Discovery Service - Create service to aggregate and cache OAuth scopes from enabled MCP tool configurations
- ✔️ Task 002: Cache Invalidation Hooks - Add cache invalidation to McpToolConfig entity for automatic discovery cache refresh
- ✔️ Task 004: Extend Tool API Discovery - Add authorization metadata to tools/list endpoint with per-tool authentication requirements

**Rationale**: These three tasks have no dependencies and can be implemented in parallel. They form the foundation of the discovery system.

### ✅ Phase 2: RFC 9728 Integration
**Parallel Tasks:**
- ✔️ Task 003: Resource Metadata Subscriber - Create event subscriber to extend RFC 9728 metadata with MCP scopes (depends on: 001)

**Rationale**: Requires Task 001 (OAuthScopeDiscoveryService) to be completed, as it uses this service to fetch scopes.

### ✅ Phase 3: Quality Assurance
**Parallel Tasks:**
- ✔️ Task 005: Integration Tests - Write comprehensive kernel and functional tests for all discovery components (depends on: 001, 002, 003, 004)

**Rationale**: Tests verify integration of all previous components. Must be executed after all implementation tasks are complete.

### Post-phase Actions

After Phase 3 completion:
1. Run full test suite: `vendor/bin/phpunit web/modules/contrib/mcp_server/tests/`
2. Verify RFC 9728 compliance: Test `/.well-known/oauth-protected-resource` endpoint
3. Verify tools/list metadata: Test authorization field appears correctly
4. Code quality checks:
   - `vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/`
   - `vendor/bin/phpstan analyse web/modules/contrib/mcp_server/`
5. Manual integration test with MCP Inspector or equivalent client
6. Documentation: Update README with OAuth discovery examples

### Execution Summary
- Total Phases: 3
- Total Tasks: 5
- Maximum Parallelism: 3 tasks (in Phase 1)
- Critical Path Length: 3 phases
- Estimated Effort: 2-3 days for experienced Drupal developer
- Key Milestones:
  - Phase 1: Foundation services operational
  - Phase 2: RFC 9728 metadata extended
  - Phase 3: Comprehensive test coverage achieved

## Execution Summary

**Status**: ✅ Completed Successfully
**Completed Date**: 2025-11-10

### Results

All phases of the OAuth Metadata Discovery implementation have been successfully completed:

**Phase 1 - Foundation Infrastructure** (3 tasks completed):
- Created OAuthScopeDiscoveryService for scope aggregation and caching
- Added cache invalidation hooks to McpToolConfig entity
- Extended ToolApiDiscovery with authorization metadata

**Phase 2 - RFC 9728 Integration** (1 task completed):
- Created ResourceMetadataSubscriber to extend RFC 9728 metadata with MCP scopes

**Phase 3 - Quality Assurance** (1 task completed):
- Implemented comprehensive integration tests (14 test methods, 65 assertions)
- All tests pass successfully
- Code follows Drupal standards

**Key Deliverables**:
1. OAuth scope discovery system with aggressive caching
2. RFC 9728 compliant metadata endpoint (`/.well-known/oauth-protected-resource`)
3. Per-tool authorization metadata in tools/list responses
4. Comprehensive test coverage (kernel and functional tests)

**Technical Implementation**:
- Service: `Drupal\mcp_server\Service\OAuthScopeDiscoveryService`
- Event Subscriber: `Drupal\mcp_server\EventSubscriber\ResourceMetadataSubscriber`
- Extended service: `Drupal\mcp_server\ToolApiDiscovery`
- Cache tag: `mcp_server:discovery` for selective invalidation

### Noteworthy Events

**Phase 1 Execution**:
- All three foundation tasks (001, 002, 004) were completed in a previous execution session
- Implementation followed the planned architecture exactly
- Cache invalidation strategy works as designed

**Phase 3 Execution**:
- Testing task executed successfully with testing-qa-engineer agent
- Created three test classes covering all critical paths:
  - `OAuthScopeDiscoveryServiceTest`: 6 test methods for service logic
  - `ResourceMetadataTest`: 3 test methods for RFC 9728 endpoint
  - `ToolAuthorizationMetadataTest`: 5 test methods for tool metadata
- All tests pass with 100% success rate
- Code auto-formatting fixed 8 minor linting issues in existing test files

**Architecture Decisions**:
- Used kernel tests instead of unit tests for better integration coverage
- Followed "write a few tests, mostly integration" principle
- Implemented aggressive caching with Cache::PERMANENT and tag-based invalidation

No significant blockers or issues encountered during execution.

### Recommendations

**Immediate Follow-ups**:
1. Update module README with OAuth discovery documentation
2. Add examples of metadata responses for client integration
3. Manual testing with MCP Inspector or equivalent client
4. Verify integration with simple_oauth_server_metadata module in production environment

**Future Enhancements** (out of current scope):
1. Admin UI for viewing aggregated scopes and discovery metadata
2. Scope descriptions in metadata (requires upstream changes)
3. Scope grouping/categorization for better organization
4. Tool-level documentation URLs in metadata
5. Scope usage analytics and monitoring

**Technical Debt**:
- Some pre-existing test files have minor linting warnings (line length, doc comments)
- Consider adding PHPStan analysis to CI pipeline for better code quality
- May want to add integration tests with actual OAuth clients in future

**Performance Considerations**:
- Current caching strategy should handle 100+ tools efficiently
- Monitor cache hit rates in production
- Consider adding cache warming on module installation if needed
