---
id: 1
group: "foundation"
dependencies: []
status: "completed"
created: "2025-11-10"
completed: "2025-11-10"
skills:
  - drupal-backend
  - php
---
# Create OAuth Scope Discovery Service

## Objective
Create OAuthScopeDiscoveryService to aggregate and cache unique OAuth scopes from all enabled MCP tool configurations.

## Skills Required
- **drupal-backend**: Drupal service creation, dependency injection, cache API
- **php**: Array manipulation, deduplication, sorting logic

## Acceptance Criteria
- [ ] OAuthScopeDiscoveryService class created in src/Service/
- [ ] Service registered in mcp_server.services.yml with dependencies
- [ ] getScopesSupported() method returns aggregated unique scopes
- [ ] Scopes cached with 'mcp_server:discovery' cache tag
- [ ] Only enabled McpToolConfig entities are processed
- [ ] Scopes sorted alphabetically for consistent output
- [ ] Empty array returned when no scopes exist

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- File: `src/Service/OAuthScopeDiscoveryService.php`
- Service ID: `mcp_server.oauth_scope_discovery`
- Dependencies: EntityTypeManagerInterface, CacheBackendInterface
- Cache ID: `mcp_server:scopes_supported`
- Cache tags: `['mcp_server:discovery']`
- Cache lifetime: `Cache::PERMANENT`

## Input Dependencies
None - this is a foundation task. Assumes PRD 1 & 2 are implemented (McpToolConfig entity with scopes field exists).

## Output Artifacts
- OAuthScopeDiscoveryService class
- Service definition in mcp_server.services.yml
- Cached scope data available for other components

## Implementation Notes
<details>
<summary>Detailed Implementation Steps</summary>

### 1. Create Service Class

**File**: `src/Service/OAuthScopeDiscoveryService.php`

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Service;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Service for discovering OAuth scopes from MCP tool configurations.
 */
final class OAuthScopeDiscoveryService {

  /**
   * Cache ID for scopes supported.
   */
  private const CACHE_ID = 'mcp_server:scopes_supported';

  /**
   * Constructs an OAuthScopeDiscoveryService.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly CacheBackendInterface $cache,
  ) {}

  /**
   * Gets all supported OAuth scopes from enabled tools.
   *
   * @return array<string>
   *   Array of unique scope IDs, sorted alphabetically.
   */
  public function getScopesSupported(): array {
    // Check cache first
    $cached = $this->cache->get(self::CACHE_ID);
    if ($cached !== FALSE) {
      return $cached->data;
    }

    // Load all enabled McpToolConfig entities
    $storage = $this->entityTypeManager->getStorage('mcp_tool_config');
    $configs = $storage->loadByProperties(['status' => TRUE]);

    // Extract and merge scopes from all configs
    $all_scopes = [];
    foreach ($configs as $config) {
      $scopes = $config->getScopes();
      if (!empty($scopes)) {
        $all_scopes = array_merge($all_scopes, $scopes);
      }
    }

    // Deduplicate and sort
    $unique_scopes = array_unique($all_scopes);
    sort($unique_scopes);

    // Cache the result
    $this->cache->set(
      self::CACHE_ID,
      $unique_scopes,
      Cache::PERMANENT,
      ['mcp_server:discovery']
    );

    return $unique_scopes;
  }

}
```

### 2. Register Service

**File**: `mcp_server.services.yml`

Add the following service definition:

```yaml
  mcp_server.oauth_scope_discovery:
    class: Drupal\mcp_server\Service\OAuthScopeDiscoveryService
    arguments:
      - '@entity_type.manager'
      - '@cache.default'
```

**Location**: Add this after the existing services, maintaining alphabetical order.

### 3. Key Implementation Details

**Caching Strategy**:
- Use `Cache::PERMANENT` because we'll invalidate via cache tags
- Tag with `['mcp_server:discovery']` for selective invalidation
- Cache check happens first for performance
- Empty array is a valid cached value

**Scope Aggregation Logic**:
1. Load only enabled configs: `loadByProperties(['status' => TRUE])`
2. Extract scopes using `$config->getScopes()` (from PRD 2)
3. Check for empty before merging (avoid merging empty arrays)
4. Use `array_merge()` to flatten all scopes into single array
5. Use `array_unique()` to remove duplicates
6. Use `sort()` for alphabetical ordering (consistent output)

**Error Handling**:
- If no configs exist, returns empty array
- If all configs have no scopes, returns empty array
- Cache backend failures fall through to recomputation

### 4. Testing Considerations

**Manual Testing**:
```bash
# Clear cache
vendor/bin/drush cache:rebuild

# Enable debug logging to see cache behavior
# Access /.well-known/oauth-protected-resource after implementing subscriber

# Verify service is registered
vendor/bin/drush devel:services | grep oauth_scope_discovery
```

**Unit Test Scenarios** (for future test task):
- Empty configs returns empty array
- Single config with scopes returns those scopes
- Multiple configs merge and deduplicate scopes
- Disabled configs are ignored
- Cache hit returns cached data
- Cache miss triggers recomputation

### 5. Integration Points

**Depends On**:
- McpToolConfig entity (PRD 1)
- `getScopes()` method (PRD 2)

**Used By**:
- ResourceMetadataSubscriber (Task 3)
- Future admin UI for scope management

### 6. Verification Steps

After implementation:
1. Rebuild cache: `vendor/bin/drush cache:rebuild`
2. Check service exists: `vendor/bin/drush devel:services mcp_server.oauth_scope_discovery`
3. Verify no syntax errors: `vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/Service/OAuthScopeDiscoveryService.php`
4. Code standards: `vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/src/Service/OAuthScopeDiscoveryService.php`

</details>
