---
id: 2
group: "foundation"
dependencies: []
status: "completed"
created: "2025-11-10"
completed: "2025-11-10"
skills:
  - drupal-backend
  - php
---
# Add Cache Invalidation to McpToolConfig Entity

## Objective
Add cache invalidation hooks to McpToolConfig entity to ensure discovery cache is cleared when tool configurations change.

## Skills Required
- **drupal-backend**: Drupal entity hooks, cache tag invalidation
- **php**: Method implementation, parent method calls

## Acceptance Criteria
- [x] postSave() hook implemented in McpToolConfig
- [x] postDelete() static method implemented in McpToolConfig
- [x] Both hooks invalidate 'mcp_server:discovery' cache tag
- [x] Parent hooks called correctly (super call pattern)
- [x] Cache invalidates on entity create, update, and delete
- [x] Manual cache clear still works as expected

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- File: `src/Entity/McpToolConfig.php` (existing entity class)
- Methods: `postSave()`, `postDelete()` (entity hooks)
- Cache tag: `'mcp_server:discovery'`
- Hook triggers: Create, update, delete operations

## Input Dependencies
None - modifying existing entity class from PRD 1.

## Output Artifacts
- Cache invalidation integrated into McpToolConfig entity
- Automatic cache refresh when configs change
- Ensures OAuthScopeDiscoveryService always has fresh data

## Implementation Notes
<details>
<summary>Detailed Implementation Steps</summary>

### 1. Add postSave() Hook

**File**: `src/Entity/McpToolConfig.php`

Add this method to the McpToolConfig class:

```php
  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE): void {
    parent::postSave($storage, $update);
    
    // Invalidate discovery cache when tool config changes
    \Drupal\Core\Cache\Cache::invalidateTags(['mcp_server:discovery']);
  }
```

**Location**: Add after existing methods, before the closing brace of the class.

**Key Points**:
- Call `parent::postSave()` FIRST to ensure base functionality runs
- Pass all parameters to parent method
- Use fully qualified class name for Cache class (or add use statement)
- Runs on both create ($update = FALSE) and update ($update = TRUE)

### 2. Add postDelete() Hook

**File**: `src/Entity/McpToolConfig.php`

Add this static method to the McpToolConfig class:

```php
  /**
   * {@inheritdoc}
   */
  public static function postDelete(EntityStorageInterface $storage, array $entities): void {
    parent::postDelete($storage, $entities);
    
    // Invalidate discovery cache when tool configs are deleted
    \Drupal\Core\Cache\Cache::invalidateTags(['mcp_server:discovery']);
  }
```

**Key Points**:
- Static method (required by entity hook signature)
- Call `parent::postDelete()` FIRST
- Invalidates cache even if multiple entities deleted at once
- All deleted entities trigger single cache invalidation

### 3. Add Use Statement (Optional but Recommended)

At the top of `src/Entity/McpToolConfig.php`, add:

```php
use Drupal\Core\Cache\Cache;
```

Then simplify the hooks to use:
```php
Cache::invalidateTags(['mcp_server:discovery']);
```

### 4. Implementation Context

**Existing Entity Structure**:
The McpToolConfig entity already exists from PRD 1 and has:
- `authentication_mode` field
- `scopes` field (from PRD 2)
- `requiresAuthentication()` method
- `getScopes()` method

**Hook Lifecycle**:
1. User saves McpToolConfig via admin UI → postSave() → cache invalidated
2. Code creates McpToolConfig programmatically → postSave() → cache invalidated
3. User deletes McpToolConfig → postDelete() → cache invalidated
4. Drush cache:rebuild → cache cleared anyway (no harm in tag invalidation)

**Cache Tag Scope**:
The `'mcp_server:discovery'` tag affects:
- OAuthScopeDiscoveryService cached scopes
- Any future discovery-related caches
- Does NOT affect tool list cache (different tag)
- Does NOT affect OAuth module caches

### 5. Testing Considerations

**Manual Testing**:
```bash
# Create a tool config
vendor/bin/drush config:set mcp_tool_config.example_tool scopes '["test:scope"]' -y

# Verify cache is cleared (check logs or debug)
vendor/bin/drush watchdog:show --severity=debug

# Update a tool config
# Delete a tool config
# Each should trigger cache invalidation
```

**Verification**:
1. Before: OAuthScopeDiscoveryService returns scopes [A, B]
2. Add new tool with scope C
3. After: OAuthScopeDiscoveryService returns scopes [A, B, C]
4. No manual cache clear needed

### 6. Edge Cases

**Multiple Entity Operations**:
- If 10 configs saved in loop, cache invalidated 10 times (no optimization needed - cache tags are cheap)
- postDelete() handles batch deletions (invalidates once regardless of entity count)

**Transaction Rollback**:
- If entity save fails and rolls back, cache still invalidated (acceptable trade-off)
- Cache will rebuild on next request with correct data

**Import/Export**:
- Config import of McpToolConfig triggers postSave()
- Cache invalidates correctly during config sync

### 7. Integration Points

**Affects**:
- OAuthScopeDiscoveryService (Task 1)
- ResourceMetadataSubscriber (Task 3)
- Any component using 'mcp_server:discovery' cache tag

**Does Not Affect**:
- Simple OAuth module caches (different tag namespace)
- Tool API discovery cache (different tag)
- Render caches (different context)

### 8. Verification Steps

After implementation:
1. Code standards: `vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/src/Entity/McpToolConfig.php`
2. Static analysis: `vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/Entity/McpToolConfig.php`
3. Functional test: Create/update/delete config and verify cache clears
4. Check no PHP errors: `vendor/bin/drush watchdog:show --severity=error`

### 9. Debugging Tips

**Verify cache invalidation is working**:
```php
// Add temporary debug logging in postSave()
\Drupal::logger('mcp_server')->debug('Cache invalidated for discovery');
```

**Check cache tag usage**:
```bash
# Install devel module if not already
vendor/bin/drush pm:enable devel -y

# Check cache bins
vendor/bin/drush devel:cache:get mcp_server:scopes_supported
```

</details>
