---
id: 3
group: "rfc-integration"
dependencies: [1]
status: "completed"
created: "2025-11-10"
completed: "2025-11-10"
skills:
  - drupal-backend
  - oauth-standards
---
# Create Resource Metadata Event Subscriber

## Objective
Create ResourceMetadataSubscriber to extend RFC 9728 Protected Resource Metadata with MCP tool scopes using the simple_oauth_server_metadata module's event system.

## Skills Required
- **drupal-backend**: Event subscriber implementation, service dependency injection
- **oauth-standards**: RFC 9728 compliance, OAuth metadata format

## Acceptance Criteria
- [x] ResourceMetadataSubscriber class created in src/EventSubscriber/
- [x] Service registered with event_subscriber tag
- [x] Subscribes to ResourceMetadataEvents::BUILD event
- [x] Adds scopes_supported to metadata response
- [x] Merges with existing scopes_supported if present
- [x] Only adds field when scopes array is not empty
- [x] Deduplicates merged scopes array

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- File: `src/EventSubscriber/ResourceMetadataSubscriber.php`
- Service ID: `mcp_server.resource_metadata_subscriber`
- Event: `ResourceMetadataEvents::BUILD` (from simple_oauth_server_metadata)
- Priority: 0 (default)
- Dependencies: OAuthScopeDiscoveryService

## Input Dependencies
- OAuthScopeDiscoveryService (Task 1)
- simple_oauth_server_metadata module events
- PRD 1 & 2 implementation (McpToolConfig with scopes)

## Output Artifacts
- ResourceMetadataSubscriber event subscriber
- Extended /.well-known/oauth-protected-resource endpoint
- RFC 9728 compliant metadata with MCP scopes

## Implementation Notes
<details>
<summary>Detailed Implementation Steps</summary>

### 1. Create Event Subscriber Class

**File**: `src/EventSubscriber/ResourceMetadataSubscriber.php`

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\EventSubscriber;

use Drupal\mcp_server\Service\OAuthScopeDiscoveryService;
use Drupal\simple_oauth_server_metadata\Event\ResourceMetadataEvent;
use Drupal\simple_oauth_server_metadata\Event\ResourceMetadataEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber to add MCP scopes to OAuth resource metadata.
 */
final class ResourceMetadataSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a ResourceMetadataSubscriber.
   */
  public function __construct(
    private readonly OAuthScopeDiscoveryService $scopeDiscovery,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ResourceMetadataEvents::BUILD => 'onBuild',
    ];
  }

  /**
   * Adds MCP scopes to resource metadata.
   *
   * @param \Drupal\simple_oauth_server_metadata\Event\ResourceMetadataEvent $event
   *   The resource metadata event.
   */
  public function onBuild(ResourceMetadataEvent $event): void {
    // Get MCP scopes from discovery service
    $mcp_scopes = $this->scopeDiscovery->getScopesSupported();

    // Only add scopes_supported if we have scopes
    if (empty($mcp_scopes)) {
      return;
    }

    // Get current metadata
    $metadata = $event->getMetadata();

    // Merge with existing scopes_supported if present
    $existing_scopes = $metadata['scopes_supported'] ?? [];
    $merged_scopes = array_merge($existing_scopes, $mcp_scopes);

    // Deduplicate and set
    $metadata['scopes_supported'] = array_values(array_unique($merged_scopes));

    // Update event metadata
    $event->setMetadata($metadata);
  }

}
```

### 2. Register Event Subscriber

**File**: `mcp_server.services.yml`

Add the following service definition:

```yaml
  mcp_server.resource_metadata_subscriber:
    class: Drupal\mcp_server\EventSubscriber\ResourceMetadataSubscriber
    arguments:
      - '@mcp_server.oauth_scope_discovery'
    tags:
      - { name: event_subscriber }
```

**Location**: Add after the oauth_scope_discovery service.

### 3. Key Implementation Details

**Event Subscription Pattern**:
- Implement `EventSubscriberInterface`
- Return event => method mapping in `getSubscribedEvents()`
- Priority 0 (default) is fine - no ordering requirements
- Event name constant from `ResourceMetadataEvents::BUILD`

**Metadata Merging Logic**:
1. Get MCP scopes from OAuthScopeDiscoveryService (cached, fast)
2. Early return if empty (don't pollute metadata with empty field)
3. Get existing metadata via `$event->getMetadata()`
4. Check for existing `scopes_supported` field
5. Merge arrays using `array_merge()` (preserves both sets)
6. Deduplicate with `array_unique()`
7. Use `array_values()` to reindex (JSON array, not object)
8. Set back via `$event->setMetadata()`

**RFC 9728 Compliance**:
- `scopes_supported` is optional field per RFC 9728 section 3
- Field is string array of scope identifiers
- Order doesn't matter per spec (but we sort for consistency)
- Merging preserves scopes from other modules
- Field must be valid JSON array (hence array_values)

**Expected Metadata Output**:
```json
{
  "resource": "https://example.com",
  "authorization_servers": ["https://example.com"],
  "bearer_methods_supported": ["header"],
  "scopes_supported": [
    "content:read",
    "content:write",
    "widget:delete"
  ],
  "resource_documentation": "https://example.com/admin/config/services/mcp-server"
}
```

### 4. Testing Considerations

**Manual Testing**:
```bash
# Clear cache
vendor/bin/drush cache:rebuild

# Test the endpoint
curl -s https://drupal-contrib.ddev.site/.well-known/oauth-protected-resource | jq

# Expected: scopes_supported field with MCP tool scopes
# Verify scopes match enabled tool configs
```

**Test Scenarios**:
- No enabled tools → no scopes_supported field
- Single tool with scopes → scopes appear in metadata
- Multiple tools → scopes merged and deduplicated
- Simple OAuth has scopes → both module scopes present
- Cache invalidation → new scopes appear after config change

### 5. Integration with simple_oauth_server_metadata

**Module Relationship**:
- simple_oauth_server_metadata provides base RFC 9728 implementation
- Fires `ResourceMetadataEvents::BUILD` event before returning response
- We subscribe to event to add MCP-specific data
- Don't override base metadata, only extend it

**Event Flow**:
1. Client: `GET /.well-known/oauth-protected-resource`
2. simple_oauth_server_metadata builds base metadata
3. Dispatches BUILD event with metadata array
4. Our subscriber adds/merges scopes_supported
5. simple_oauth_server_metadata returns final JSON

**Base Metadata Fields** (from simple_oauth_server_metadata):
- `resource`: Site base URL
- `authorization_servers`: Array with token endpoint URL
- `bearer_methods_supported`: ["header"]
- `resource_documentation`: Admin config URL
- `scopes_supported`: May or may not exist (we merge if exists)

### 6. Error Handling

**Event Not Firing**:
- Verify simple_oauth_server_metadata module enabled
- Check event subscriber tag in services.yml
- Clear cache after service changes
- Check module weight if needed

**Empty Scopes**:
- Early return prevents adding empty field
- RFC 9728 allows omitting scopes_supported
- Better than returning [] (indicates no scopes vs unknown scopes)

**Invalid Scopes Format**:
- OAuthScopeDiscoveryService always returns string array
- array_unique handles any duplicates
- array_values ensures JSON array encoding

### 7. Verification Steps

After implementation:
1. Rebuild cache: `vendor/bin/drush cache:rebuild`
2. Check service: `vendor/bin/drush devel:services mcp_server.resource_metadata_subscriber`
3. Test endpoint: `curl https://drupal-contrib.ddev.site/.well-known/oauth-protected-resource`
4. Verify JSON: Should include scopes_supported with tool scopes
5. Code standards: `vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/src/EventSubscriber/`
6. Static analysis: `vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/EventSubscriber/`

### 8. RFC 9728 Reference

**Relevant Sections**:
- Section 2: Metadata Request (GET /.well-known/oauth-protected-resource)
- Section 3: Metadata Response (JSON object with optional fields)
- Section 3.1: scopes_supported field (array of strings)

**Specification Quote**:
> scopes_supported: OPTIONAL. JSON array of strings containing scope values supported by the protected resource.

**Our Implementation**:
- Compliant: scopes_supported is array of strings
- Compliant: Field is optional (omitted when empty)
- Extension: Merges scopes from multiple sources (allowed by spec)

### 9. Debugging Tips

**Verify event subscription**:
```bash
# Check if subscriber is registered
vendor/bin/drush devel:services --tag=event_subscriber | grep resource_metadata
```

**Test event flow**:
```php
// Add temporary logging in onBuild()
\Drupal::logger('mcp_server')->debug('Building resource metadata with @count scopes', [
  '@count' => count($mcp_scopes),
]);
```

**Check metadata structure**:
```bash
# Pretty print JSON response
curl -s https://drupal-contrib.ddev.site/.well-known/oauth-protected-resource | jq .
```

</details>
