---
id: 4
group: "tool-metadata"
dependencies: []
status: "completed"
created: "2025-11-10"
completed: "2025-11-10"
skills:
  - drupal-backend
  - php
---
# Extend ToolApiDiscovery with Authorization Metadata

## Objective
Extend ToolApiDiscovery service to include per-tool authorization metadata in tools/list response, showing authentication requirements and required scopes.

## Skills Required
- **drupal-backend**: Modifying existing Drupal services, entity loading
- **php**: Conditional logic, array manipulation

## Acceptance Criteria
- [x] getToolDefinition() method modified to add authorization field
- [x] Authorization field includes method, mode, and requiredScopes
- [x] Field only added when authentication is not disabled
- [x] Field omitted entirely when authentication is disabled
- [x] Mode reflects "required" or "optional" from config
- [x] RequiredScopes array matches config scopes
- [x] Empty scopes array handled correctly (any authenticated user)

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- File: `src/ToolApiDiscovery.php` (existing service)
- Method: `getToolDefinition()` (modify existing)
- New field: `authorization` object with method, mode, requiredScopes
- Condition: Only add if authentication_mode !== 'disabled'

## Input Dependencies
- McpToolConfig entity with authentication_mode and scopes (PRD 1 & 2)
- ToolApiDiscovery service (existing from Plan 1)

## Output Artifacts
- Enhanced tools/list endpoint with authorization metadata
- Per-tool authentication requirements visible to MCP clients
- Standards-compliant tool metadata format

## Implementation Notes
<details>
<summary>Detailed Implementation Steps</summary>

### 1. Locate Target Method

**File**: `src/ToolApiDiscovery.php`

Find the `getToolDefinition()` method or the method that builds tool response data. This method likely:
- Builds tool metadata (name, description, inputSchema)
- Returns array or object representing a tool
- Called when building tools/list response

**Current Structure** (approximate):
```php
public function getToolDefinition(string $tool_id): array {
  // Load tool from Tool API module
  // Build basic metadata
  // Return tool array
}
```

### 2. Add Authorization Field Logic

Extend the method to add authorization metadata:

```php
public function getToolDefinition(string $tool_id): array {
  // ... existing tool building logic ...

  // Build base tool metadata
  $tool = [
    'name' => $mcp_name,
    'description' => $description,
    'inputSchema' => $input_schema,
  ];

  // Add authorization metadata if authentication is configured
  $config = $this->loadToolConfig($tool_id);
  if ($config && $config->requiresAuthentication()) {
    $tool['authorization'] = [
      'method' => 'oauth2',
      'mode' => $config->getAuthenticationMode(),
      'requiredScopes' => $config->getScopes(),
    ];
  }

  return $tool;
}
```

### 3. Key Implementation Details

**Conditional Inclusion**:
- Load McpToolConfig using existing helper method (likely `loadToolConfig()`)
- Check `$config->requiresAuthentication()` (returns FALSE for disabled mode)
- Only add authorization field when authentication is required or optional
- Completely omit field for disabled mode

**Authorization Field Structure**:
```php
[
  'method' => 'oauth2',           // Always 'oauth2' (only method we support)
  'mode' => 'required',            // 'required' or 'optional' from config
  'requiredScopes' => [            // Array of scope IDs
    'widget:write',
    'widget:create',
  ],
]
```

**Empty Scopes Behavior**:
- If `getScopes()` returns [], requiredScopes will be []
- This means: authenticated but no specific scope restrictions
- MCP client should still authenticate, but doesn't need specific scopes
- Different from disabled mode (no authorization field at all)

### 4. Expected Tool Response Examples

**Tool with Required Authentication**:
```json
{
  "name": "widget.create",
  "description": "Create a new widget",
  "inputSchema": {
    "type": "object",
    "properties": {...}
  },
  "authorization": {
    "method": "oauth2",
    "mode": "required",
    "requiredScopes": ["widget:write", "widget:create"]
  }
}
```

**Tool with Optional Authentication**:
```json
{
  "name": "public.search",
  "description": "Search public content",
  "inputSchema": {...},
  "authorization": {
    "method": "oauth2",
    "mode": "optional",
    "requiredScopes": ["content:read"]
  }
}
```

**Tool with Disabled Authentication**:
```json
{
  "name": "health.check",
  "description": "Health check endpoint",
  "inputSchema": {...}
}
```
Note: No authorization field at all.

**Tool with Authentication but No Scope Restrictions**:
```json
{
  "name": "user.profile",
  "description": "Get user profile",
  "inputSchema": {...},
  "authorization": {
    "method": "oauth2",
    "mode": "required",
    "requiredScopes": []
  }
}
```
Note: Empty array means any authenticated user can access.

### 5. Integration Points

**Method Signatures**:
- Verify `loadToolConfig()` exists (from PRD 1)
- Verify `requiresAuthentication()` method exists (from PRD 1)
- Verify `getAuthenticationMode()` method exists (from PRD 1)
- Verify `getScopes()` method exists (from PRD 2)

**Service Dependencies**:
- No new dependencies needed
- Uses existing entity type manager injection
- Reuses existing config loading logic

**Where This Runs**:
- Called from McpController::listTools()
- Response sent to MCP clients
- May be cached (depends on controller implementation)

### 6. Testing Considerations

**Manual Testing**:
```bash
# Clear cache
vendor/bin/drush cache:rebuild

# Test tools/list endpoint
curl -s -X POST https://drupal-contrib.ddev.site/mcp/tools/list \
  -H "Content-Type: application/json" \
  -d '{"jsonrpc": "2.0", "id": 1, "method": "tools/list"}' | jq

# Verify authorization field appears for configured tools
# Verify field omitted for disabled tools
```

**Test Scenarios**:
- Tool with required auth → authorization field with mode: "required"
- Tool with optional auth → authorization field with mode: "optional"
- Tool with disabled auth → no authorization field
- Tool with scopes → scopes in requiredScopes array
- Tool without scopes → empty requiredScopes array
- No config exists → no authorization field (backward compatible)

### 7. Backward Compatibility

**Existing Behavior**:
- Tools without McpToolConfig work as before
- No authorization field for unconfigured tools
- Doesn't break existing MCP clients (optional field)

**New Behavior**:
- Configured tools show authentication requirements
- MCP clients can read and adapt behavior
- Non-aware clients ignore authorization field

### 8. Error Handling

**Config Loading Failures**:
- If `loadToolConfig()` returns NULL, no authorization field added
- Tool still works (backward compatible)
- Logs warning if needed (depends on existing implementation)

**Invalid Config Values**:
- `requiresAuthentication()` handles edge cases
- `getAuthenticationMode()` returns valid value or default
- `getScopes()` returns array (empty if not set)

### 9. Verification Steps

After implementation:
1. Rebuild cache: `vendor/bin/drush cache:rebuild`
2. Test endpoint: `curl -X POST https://drupal-contrib.ddev.site/mcp/tools/list ...`
3. Verify JSON structure: Check authorization field format
4. Test all auth modes: required, optional, disabled
5. Code standards: `vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/src/ToolApiDiscovery.php`
6. Static analysis: `vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/ToolApiDiscovery.php`

### 10. MCP Client Integration

**How Clients Use This**:
1. Client calls tools/list
2. Reads authorization field for each tool
3. If mode: "required" and user not authenticated:
   - Initiates OAuth flow
   - Requests requiredScopes in token request
4. If mode: "optional":
   - Can call without auth (limited functionality)
   - Or authenticate for full access
5. If no authorization field:
   - Calls tool without authentication

**Example Client Logic**:
```javascript
// Parse tool response
const tool = await client.getTool('widget.create');

if (tool.authorization?.mode === 'required') {
  // Must authenticate
  const scopes = tool.authorization.requiredScopes;
  await client.authenticate(scopes);
}

// Now call tool
const result = await client.callTool('widget.create', args);
```

### 11. Debugging Tips

**Verify config loading**:
```php
// Add temporary debug logging
$config = $this->loadToolConfig($tool_id);
\Drupal::logger('mcp_server')->debug('Tool @tool config: @config', [
  '@tool' => $tool_id,
  '@config' => $config ? 'found' : 'not found',
]);
```

**Check field structure**:
```bash
# Validate JSON structure
curl -s -X POST ... | jq '.result.tools[] | select(.authorization)'
```

**Test mode values**:
```bash
# Check all possible modes
curl ... | jq '.result.tools[] | .authorization.mode' | sort -u
# Expected output: "required", "optional" (not "disabled")
```

</details>
