---
id: 5
group: "testing"
dependencies: [1, 2, 3, 4]
status: "completed"
created: "2025-11-10"
completed: "2025-11-10"
skills:
  - drupal-testing
  - php
---
# Write Comprehensive Integration Tests

## Objective
Create comprehensive integration tests for OAuth metadata discovery, focusing on kernel and functional tests following "write a few tests, mostly integration" principle.

## Skills Required
- **drupal-testing**: PHPUnit kernel and functional tests, test fixtures
- **php**: Test assertions, mock data creation

## Acceptance Criteria
- [x] OAuthScopeDiscoveryServiceTest (kernel test) created
- [x] ResourceMetadataTest (functional test) created
- [x] ToolAuthorizationMetadataTest (kernel test) created
- [x] Cache invalidation tested in kernel test
- [x] RFC 9728 endpoint tested in functional test
- [x] Tools/list authorization field tested in kernel test
- [x] All tests pass with vendor/bin/phpunit
- [x] Code coverage for critical paths

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Test suite: kernel and functional (not unit - follow integration approach)
- Files: tests/src/Kernel/, tests/src/Functional/
- Fixtures: Create test McpToolConfig entities
- Assertions: Scope aggregation, metadata format, cache behavior

## Input Dependencies
- All previous tasks (1-4) completed
- PRD 1 & 2 implementation (authentication and scopes)
- simple_oauth_server_metadata module

## Output Artifacts
- Kernel test for scope discovery and cache
- Functional test for RFC 9728 endpoint
- Functional test for tools/list authorization
- Comprehensive test coverage documentation

## Implementation Notes
<details>
<summary>Detailed Implementation Steps</summary>

### 1. Kernel Test: OAuth Scope Discovery Service

**File**: `tests/src/Kernel/OAuthScopeDiscoveryServiceTest.php`

```php
<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\mcp_server\Entity\McpToolConfig;

/**
 * Tests OAuth scope discovery service.
 *
 * @group mcp_server
 */
class OAuthScopeDiscoveryServiceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'mcp_server',
    'tool',
  ];

  /**
   * The scope discovery service.
   */
  protected $scopeDiscovery;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    
    $this->installEntitySchema('mcp_tool_config');
    $this->installEntitySchema('user');
    $this->installConfig(['mcp_server']);
    
    $this->scopeDiscovery = $this->container->get('mcp_server.oauth_scope_discovery');
  }

  /**
   * Tests empty scopes when no configs exist.
   */
  public function testEmptyScopes(): void {
    $scopes = $this->scopeDiscovery->getScopesSupported();
    $this->assertSame([], $scopes);
  }

  /**
   * Tests scope aggregation from multiple tools.
   */
  public function testScopeAggregation(): void {
    // Create first tool config
    McpToolConfig::create([
      'id' => 'tool1',
      'tool_id' => 'tool1',
      'status' => TRUE,
      'authentication_mode' => 'required',
      'scopes' => ['content:read', 'content:write'],
    ])->save();

    // Create second tool config
    McpToolConfig::create([
      'id' => 'tool2',
      'tool_id' => 'tool2',
      'status' => TRUE,
      'authentication_mode' => 'required',
      'scopes' => ['widget:delete', 'content:read'],
    ])->save();

    $scopes = $this->scopeDiscovery->getScopesSupported();
    
    // Should be deduplicated and sorted
    $expected = ['content:read', 'content:write', 'widget:delete'];
    $this->assertSame($expected, $scopes);
  }

  /**
   * Tests disabled tools are ignored.
   */
  public function testDisabledToolsIgnored(): void {
    // Enabled tool
    McpToolConfig::create([
      'id' => 'enabled',
      'tool_id' => 'enabled',
      'status' => TRUE,
      'scopes' => ['enabled:scope'],
    ])->save();

    // Disabled tool
    McpToolConfig::create([
      'id' => 'disabled',
      'tool_id' => 'disabled',
      'status' => FALSE,
      'scopes' => ['disabled:scope'],
    ])->save();

    $scopes = $this->scopeDiscovery->getScopesSupported();
    
    $this->assertContains('enabled:scope', $scopes);
    $this->assertNotContains('disabled:scope', $scopes);
  }

  /**
   * Tests cache invalidation on config save.
   */
  public function testCacheInvalidation(): void {
    // First call - populates cache
    $scopes1 = $this->scopeDiscovery->getScopesSupported();
    $this->assertSame([], $scopes1);

    // Create new config
    McpToolConfig::create([
      'id' => 'new_tool',
      'tool_id' => 'new_tool',
      'status' => TRUE,
      'scopes' => ['new:scope'],
    ])->save();

    // Second call - should reflect new config (cache invalidated)
    $scopes2 = $this->scopeDiscovery->getScopesSupported();
    $this->assertContains('new:scope', $scopes2);
  }

  /**
   * Tests cache invalidation on config delete.
   */
  public function testCacheInvalidationOnDelete(): void {
    $config = McpToolConfig::create([
      'id' => 'delete_me',
      'tool_id' => 'delete_me',
      'status' => TRUE,
      'scopes' => ['delete:scope'],
    ]);
    $config->save();

    // Verify scope exists
    $scopes1 = $this->scopeDiscovery->getScopesSupported();
    $this->assertContains('delete:scope', $scopes1);

    // Delete config
    $config->delete();

    // Verify scope removed
    $scopes2 = $this->scopeDiscovery->getScopesSupported();
    $this->assertNotContains('delete:scope', $scopes2);
  }

  /**
   * Tests empty scopes array handled correctly.
   */
  public function testEmptyScopesArray(): void {
    McpToolConfig::create([
      'id' => 'no_scopes',
      'tool_id' => 'no_scopes',
      'status' => TRUE,
      'scopes' => [],
    ])->save();

    $scopes = $this->scopeDiscovery->getScopesSupported();
    $this->assertSame([], $scopes);
  }

}
```

### 2. Functional Test: RFC 9728 Resource Metadata

**File**: `tests/src/Functional/ResourceMetadataTest.php`

```php
<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\mcp_server\Entity\McpToolConfig;

/**
 * Tests OAuth resource metadata endpoint.
 *
 * @group mcp_server
 */
class ResourceMetadataTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'mcp_server',
    'tool',
    'simple_oauth',
    'simple_oauth_server_metadata',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests metadata endpoint without scopes.
   */
  public function testMetadataWithoutScopes(): void {
    $response = $this->drupalGet('/.well-known/oauth-protected-resource');
    $this->assertSession()->statusCodeEquals(200);
    
    $metadata = json_decode($response, TRUE);
    $this->assertIsArray($metadata);
    $this->assertArrayHasKey('resource', $metadata);
    
    // No scopes should be present
    if (isset($metadata['scopes_supported'])) {
      $this->assertEmpty($metadata['scopes_supported']);
    }
  }

  /**
   * Tests metadata endpoint with MCP scopes.
   */
  public function testMetadataWithScopes(): void {
    // Create tool configs with scopes
    McpToolConfig::create([
      'id' => 'tool1',
      'tool_id' => 'tool1',
      'status' => TRUE,
      'authentication_mode' => 'required',
      'scopes' => ['content:read', 'content:write'],
    ])->save();

    McpToolConfig::create([
      'id' => 'tool2',
      'tool_id' => 'tool2',
      'status' => TRUE,
      'authentication_mode' => 'required',
      'scopes' => ['widget:delete'],
    ])->save();

    $response = $this->drupalGet('/.well-known/oauth-protected-resource');
    $this->assertSession()->statusCodeEquals(200);
    
    $metadata = json_decode($response, TRUE);
    $this->assertArrayHasKey('scopes_supported', $metadata);
    
    $scopes = $metadata['scopes_supported'];
    $this->assertContains('content:read', $scopes);
    $this->assertContains('content:write', $scopes);
    $this->assertContains('widget:delete', $scopes);
  }

  /**
   * Tests RFC 9728 compliance.
   */
  public function testRfc9728Compliance(): void {
    $response = $this->drupalGet('/.well-known/oauth-protected-resource');
    $metadata = json_decode($response, TRUE);
    
    // Required fields per RFC 9728
    $this->assertArrayHasKey('resource', $metadata);
    
    // Optional but expected fields
    if (isset($metadata['scopes_supported'])) {
      $this->assertIsArray($metadata['scopes_supported']);
      foreach ($metadata['scopes_supported'] as $scope) {
        $this->assertIsString($scope);
      }
    }
  }

}
```

### 3. Functional Test: Tool Authorization Metadata

**File**: `tests/src/Functional/ToolAuthorizationMetadataTest.php`

```php
<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\mcp_server\Entity\McpToolConfig;

/**
 * Tests tool authorization metadata in tools/list.
 *
 * @group mcp_server
 */
class ToolAuthorizationMetadataTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'mcp_server',
    'tool',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests authorization field for required auth.
   */
  public function testRequiredAuthorization(): void {
    // Create tool config with required auth
    McpToolConfig::create([
      'id' => 'required_tool',
      'tool_id' => 'required_tool',
      'status' => TRUE,
      'authentication_mode' => 'required',
      'scopes' => ['test:read', 'test:write'],
    ])->save();

    // Call tools/list
    $response = $this->drupalPostJson('/mcp/tools/list', [
      'jsonrpc' => '2.0',
      'id' => 1,
      'method' => 'tools/list',
    ]);

    $data = json_decode($response, TRUE);
    $tools = $data['result']['tools'] ?? [];
    
    // Find our tool
    $tool = $this->findTool($tools, 'required_tool');
    $this->assertNotNull($tool);
    
    // Verify authorization field
    $this->assertArrayHasKey('authorization', $tool);
    $this->assertSame('oauth2', $tool['authorization']['method']);
    $this->assertSame('required', $tool['authorization']['mode']);
    $this->assertContains('test:read', $tool['authorization']['requiredScopes']);
    $this->assertContains('test:write', $tool['authorization']['requiredScopes']);
  }

  /**
   * Tests no authorization field for disabled auth.
   */
  public function testDisabledAuthorization(): void {
    McpToolConfig::create([
      'id' => 'disabled_tool',
      'tool_id' => 'disabled_tool',
      'status' => TRUE,
      'authentication_mode' => 'disabled',
    ])->save();

    $response = $this->drupalPostJson('/mcp/tools/list', [
      'jsonrpc' => '2.0',
      'id' => 1,
      'method' => 'tools/list',
    ]);

    $data = json_decode($response, TRUE);
    $tools = $data['result']['tools'] ?? [];
    
    $tool = $this->findTool($tools, 'disabled_tool');
    $this->assertNotNull($tool);
    
    // Authorization field should not exist
    $this->assertArrayNotHasKey('authorization', $tool);
  }

  /**
   * Helper to find tool by name.
   */
  protected function findTool(array $tools, string $name): ?array {
    foreach ($tools as $tool) {
      if ($tool['name'] === $name) {
        return $tool;
      }
    }
    return NULL;
  }

}
```

### 4. Running Tests

**Run All Tests**:
```bash
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/
```

**Run Specific Test Suite**:
```bash
# Kernel tests only
vendor/bin/phpunit --testsuite=kernel web/modules/contrib/mcp_server/

# Functional tests only
vendor/bin/phpunit --testsuite=functional web/modules/contrib/mcp_server/
```

**Run Single Test Class**:
```bash
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/src/Kernel/OAuthScopeDiscoveryServiceTest.php
```

### 5. Test Coverage Strategy

**Following "Write a Few Tests, Mostly Integration"**:
- ✅ Kernel tests for service logic (OAuthScopeDiscoveryService)
- ✅ Functional tests for HTTP endpoints (metadata, tools/list)
- ❌ No unit tests (too granular, low value)
- ❌ No browser tests with JS (not needed for this feature)

**Critical Paths Covered**:
1. Scope aggregation and deduplication
2. Cache invalidation on entity save/delete
3. RFC 9728 metadata format
4. Authorization field in tool responses
5. Different authentication modes
6. Edge cases (empty scopes, disabled tools)

### 6. Test Data Fixtures

**Minimal Fixture Approach**:
- Create McpToolConfig entities in setUp() or individual tests
- Use simple, focused data (not complex scenarios)
- Clean up automatically (PHPUnit handles teardown)

**Example Fixture**:
```php
protected function createToolConfig(array $values = []): McpToolConfig {
  $defaults = [
    'id' => 'test_tool_' . rand(),
    'tool_id' => 'test_tool',
    'status' => TRUE,
    'authentication_mode' => 'required',
    'scopes' => [],
  ];
  
  return McpToolConfig::create($values + $defaults);
}
```

### 7. Verification Steps

After implementation:
1. Run tests: `vendor/bin/phpunit web/modules/contrib/mcp_server/tests/`
2. Check coverage: All critical paths have test coverage
3. Verify green: All tests pass
4. Code standards: `vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/tests/`
5. Check test output: Review `sites/simpletest/browser_output/` for failures

### 8. Debugging Failed Tests

**Common Issues**:
- Module dependencies not listed in $modules
- Entity schema not installed
- Config not installed
- Cache not cleared between tests

**Debug Steps**:
```bash
# Run with verbose output
vendor/bin/phpunit --verbose web/modules/contrib/mcp_server/tests/

# Run single test method
vendor/bin/phpunit --filter testScopeAggregation web/modules/contrib/mcp_server/tests/

# Check browser output for functional tests
ls sites/simpletest/browser_output/
```

</details>
