---
id: 3
group: "functional-consolidation"
dependencies: [2]
status: "completed"
created: 2025-11-10
completed: 2025-11-10
skills:
  - phpunit
  - drupal-backend
blockers:
  - Authentication logic bug: anonymous requests incorrectly pass authentication checks
  - Tool execution failing with error -32603 instead of -32001 for auth failures
---
# Consolidate Functional Tests into Single Site Build

## Objective
Merge all functional test scenarios into a single test class with one test method, eliminating 5 expensive Drupal site rebuilds while maintaining identical test coverage.

## Skills Required
- **phpunit**: Functional test consolidation patterns and helper method design
- **drupal-backend**: BrowserTestBase understanding and Drupal site setup cost awareness

## Acceptance Criteria
- [ ] Create new `tests/src/Functional/McpServerFunctionalTest.php` with single `testEndToEndWorkflows()` method
- [ ] Migrate all scenarios from `McpAuthenticationFunctionalTest` to private helper methods
- [ ] Migrate all scenarios from `ResourceMetadataTest` to private helper methods
- [ ] Verify all helper methods use descriptive assertion messages
- [ ] Delete `tests/src/Functional/McpAuthenticationFunctionalTest.php`
- [ ] Delete `tests/src/Functional/ResourceMetadataTest.php`
- [ ] Run functional tests and verify identical assertions pass
- [ ] Measure execution time reduction (should be ~80% faster)

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Drupal BrowserTestBase
- SimpleOauth testing infrastructure
- HTTP request/response testing via Guzzle
- Understanding of Drupal site setup costs (~10-30s per test method)

## Input Dependencies
- Task 2 completed (kernel tests consolidated)
- Clean baseline for functional tests

## Output Artifacts
- Single functional test class with consolidated scenarios
- ~80% faster functional test execution
- Identical test coverage via helper methods

## Implementation Notes

<details>
<summary>Detailed Implementation Instructions</summary>

**Critical Performance Principle:**
Each `public function test*()` method in BrowserTestBase triggers a FULL Drupal installation:
- Database schema creation
- Module installation
- Configuration import
- Routing table building
- **Cost: ~10-30 seconds PER test method**

By using ONE test method with helper methods, we pay this cost ONCE instead of 6 times.

**Phase 1: Create Consolidated Test Class**

Create `tests/src/Functional/McpServerFunctionalTest.php`:

```php
<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Functional;

use Drupal\Component\Serialization\Json;
use Drupal\consumers\Entity\Consumer;
use Drupal\mcp_server\Entity\McpToolConfig;
use Drupal\simple_oauth\Entity\Oauth2Token;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\simple_oauth\Functional\SimpleOauthTestTrait;

/**
 * Consolidated functional tests for MCP Server end-to-end workflows.
 *
 * All scenarios run in a single Drupal installation to avoid expensive
 * site rebuilds. Each scenario is a private helper method.
 *
 * @group mcp_server
 */
final class McpServerFunctionalTest extends BrowserTestBase {

  use SimpleOauthTestTrait;

  protected static $modules = [
    'mcp_server',
    'mcp_server_test',
    'tool',
    'simple_oauth',
    'simple_oauth_server_metadata',
    'consumers',
    'serialization',
    'user',
    'system',
  ];

  protected $defaultTheme = 'stark';

  protected Consumer $client;
  protected $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->setUpOAuthInfrastructure();
  }

  /**
   * Tests all end-to-end MCP Server workflows via HTTP.
   *
   * This consolidated test runs all functional scenarios in a single
   * Drupal installation to avoid expensive site rebuilds.
   */
  public function testEndToEndWorkflows(): void {
    // Authentication scenarios
    $this->assertAnonymousRequestToRequiredToolReturns401();
    $this->assertAuthenticatedRequestToRequiredToolSucceeds();
    $this->assertInvalidTokenReturns401();
    $this->assertAnonymousRequestToOptionalToolSucceeds();

    // OAuth metadata scenarios
    $this->assertMetadataEndpointWithoutScopes();
    $this->assertMetadataEndpointWithScopes();
    $this->assertMetadataRfc9728Compliance();
  }

  // Copy all helper methods from McpAuthenticationFunctionalTest and ResourceMetadataTest
  // Make them private methods (not test* methods)
}
```

**Phase 2: Extract Helper Methods**

From `McpAuthenticationFunctionalTest`, convert these test methods to private helpers:
- `testAnonymousRequestToRequiredModeTool()` → `assertAnonymousRequestToRequiredToolReturns401()`
- `testAuthenticatedRequestToRequiredModeTool()` → `assertAuthenticatedRequestToRequiredToolSucceeds()`
- `testInvalidTokenToRequiredModeTool()` → `assertInvalidTokenReturns401()`
- `testAnonymousRequestToOptionalModeTool()` → `assertAnonymousRequestToOptionalToolSucceeds()`

From `ResourceMetadataTest`, convert:
- `testMetadataWithoutScopes()` → `assertMetadataEndpointWithoutScopes()`
- `testMetadataWithScopes()` → `assertMetadataEndpointWithScopes()`
- `testRfc9728Compliance()` → `assertMetadataRfc9728Compliance()`

Also copy these helper methods as-is:
- `setUpOAuthInfrastructure()` (combine setUp logic from both files)
- `initializeMcpSession()`
- `createToolConfigs()`
- `createAccessToken()`
- `drupalPost()`

**Phase 3: Delete Old Files**
```bash
rm tests/src/Functional/McpAuthenticationFunctionalTest.php
rm tests/src/Functional/ResourceMetadataTest.php
```

**Phase 4: Verify Performance Improvement**
```bash
cd /var/www/html
time vendor/bin/phpunit web/modules/contrib/mcp_server/tests/src/Functional/
```

Before: ~6 site builds (6 × 15s = 90s)
After: ~1 site build (1 × 15s = 15s)
Expected improvement: ~80% faster

**Design Principles:**
- One public `test*()` method = one Drupal site build
- Private helper methods use assertions directly (no performance cost)
- Helpers are independent (unique entity IDs, no shared state mutations)
- Descriptive assertion messages identify failures clearly
</details>
