---
id: 6
summary: "Implement custom CallToolRequest handler to bypass SDK reflection and enable clean integration with dynamically discovered tools"
created: 2025-11-11
---

# Plan: Custom Handler for SDK Integration with Dynamic Tools

## Original Work Order

> I've completed a deep dive into the MCP SDK's architecture. Here's what I found:
>
> 1. Handler Selection Mechanism (Protocol.php:170-173)
>
> The SDK uses a first-match-wins pattern:
>
> foreach ($this->requestHandlers as $handler) {
>     if (!$handler->supports($request)) {
>         continue;
>     }
>     // First handler that supports the request wins
>     $handler->handle($request, $session);
>     break;
> }
>
> Each handler implements supports(Request $request): bool to declare what it can handle.
>
> 2. Critical Discovery: Handler Priority (Builder.php:432-443)
>
> Custom handlers are checked BEFORE SDK's default handlers:
>
> $requestHandlers = array_merge($this->requestHandlers, [
>     new Handler\Request\CallToolHandler(...),  // SDK's default
>     new Handler\Request\ListToolsHandler(...),
>     // ... other defaults
> ]);
>
> The array_merge() puts custom handlers (from addRequestHandler()) at the front of the array. This means we can intercept tool calls before the SDK's CallToolHandler!
>
> 3. Two Separate Systems
>
> The SDK has two independent systems:
>
> A. Tool Registry (for discovery):
> - Tools registered via addTool(handler, name, description, inputSchema)
> - Stored in Registry for metadata
> - Used by ListToolsHandler to respond to tools/list requests
> - The handler parameter is only used if CallToolHandler processes the request
>
> B. Request Handlers (for routing):
> - Handlers registered via addRequestHandler(handler)
> - First handler whose supports() returns true handles the request
> - Complete control over how requests are processed
>
> 4. The Clean Solution
>
> We can write a custom RequestHandlerInterface that:
>
> 1. Intercepts CallToolRequest before SDK's CallToolHandler sees it
> 2. Bypasses the ReferenceHandler reflection system entirely
> 3. Passes arguments directly to our bridge service as an associative array
> 4. No code generation or eval() needed!
>
> The flow would be:
> Request arrives → Protocol loops handlers → Our CustomCallToolHandler.supports() returns true
> → Our handler executes → Arguments passed as array to bridge → Done!
>
> For tool discovery (tools/list), we still use addTool() to register metadata, but we can pass a dummy handler (like a closure that returns null) since it will never be called - our custom handler intercepts first!
>
> 5. Issues with Current Implementation
>
> Looking at our existing CustomCallToolHandler:
>
> ❌ Wrong interface method - Has getMethod() instead of supports()
> ❌ Not registered - Never added via addRequestHandler()
> ❌ ToolHandlerFactory still used - File generation approach is still active
>
> The interface is:
> interface RequestHandlerInterface {
>     public function supports(Request $request): bool;
>     public function handle(Request $request, SessionInterface $session): Response|Error;
> }
>
> SDK's CallToolHandler does:
> public function supports(Request $request): bool {
>     return $request instanceof CallToolRequest;
> }
>
> Our custom handler should do the same to intercept all tool calls.
>
> 6. Recommended Approach
>
> 1. Fix CustomCallToolHandler:
>    - Remove getMethod() method
>    - Add supports() method that returns $request instanceof CallToolRequest
> 2. Update McpServerFactory:
>    - Instantiate CustomCallToolHandler with bridge and logger
>    - Call $builder->addRequestHandler($customHandler) BEFORE build()
>    - For addTool(), we can pass a simple dummy handler like: fn() => null
>    - Remove ToolHandlerFactory dependency
> 3. Clean up:
>    - Remove ToolHandlerFactory class
>    - Remove Handler/Generated directory
>    - Much simpler codebase!
>
> 7. Why This Works
>
> - No reflection: Arguments arrive as ['name' => 'John', 'email' => '...'] and we pass them directly to bridge
> - Clean extension point: Using SDK's designed extension mechanism (addRequestHandler())
> - No code generation: Everything is runtime with regular PHP objects
> - Maintains SDK compatibility: Tools still listed in discovery, handlers properly implement interface

## Executive Summary

This plan implements a clean, maintainable solution for integrating dynamically discovered Drupal tools with the MCP SDK's request handling system. Instead of fighting the SDK's reflection-based parameter matching through code generation or eval(), we leverage the SDK's official extension point (`addRequestHandler()`) to intercept tool execution requests before they reach the SDK's default CallToolHandler.

The approach eliminates all code generation complexity (ToolHandlerFactory, generated handler classes) by creating a single custom request handler that receives arguments as associative arrays and passes them directly to the bridge service. This solution is architecturally sound, uses SDK's designed extension mechanisms, and dramatically simplifies the codebase while maintaining full SDK compatibility for tool discovery.

**Critically, this plan includes comprehensive cleanup** to remove all traces of failed approaches, debug logging, and obsolete code. This cleanup phase is mandatory and ensures the codebase is production-ready, maintainable, and free of technical debt.

Key benefits include: zero runtime code generation, no reflection conflicts, clean separation between tool discovery (metadata) and tool execution (runtime), 90% reduction in handler-related code complexity, and removal of ~260 lines of obsolete/debug code.

## Context

### Current State

The module currently uses a complex code generation approach (ToolHandlerFactory) that dynamically creates PHP class files with parameter signatures matching each tool's input schema. This was necessary because:

1. The SDK's default CallToolHandler uses PHP reflection to map incoming JSON-RPC arguments to handler function parameters BY NAME
2. Our tools are discovered dynamically from database configuration entities, so we can't create static handler classes with known parameter signatures
3. The previous attempt to use a generic `function(array $arguments)` handler failed because reflection saw only an "arguments" parameter name, not the actual tool parameters like "name", "email", etc.

**Problems with current approach:**
- Generates PHP files at runtime in `temporary://mcp_handlers/`
- Requires file I/O for every unique tool signature
- Complex class generation logic with string templating
- Uses `\Drupal::service()` in generated code (service locator anti-pattern)
- Hard to debug (generated code not in version control)
- CustomCallToolHandler exists but has wrong interface (`getMethod()` instead of `supports()`)
- ToolHandlerFactory is still being used despite CustomCallToolHandler being created

### Target State

A clean implementation where:

1. **Single custom handler** (`CustomCallToolHandler`) intercepts all `CallToolRequest` instances
2. **No code generation** - handler is a regular PHP class with proper dependency injection
3. **Arguments passed directly** to bridge service as associative arrays (no reflection needed)
4. **Tool discovery separate from execution** - tools registered with dummy handlers for metadata, custom handler handles all execution
5. **Simplified service registration** - ToolHandlerFactory removed from service container
6. **Clean codebase** - Handler/Generated directory and all generation logic removed

### Background

**Investigation findings:**

1. **SDK Handler Priority**: Custom handlers registered via `addRequestHandler()` are checked BEFORE SDK's default handlers (array_merge puts them first)
2. **First-Match-Wins**: Protocol loops through handlers and uses first one whose `supports()` method returns true
3. **Two Independent Systems**: Tool Registry (for `tools/list` discovery) is separate from Request Handlers (for `tools/call` execution)
4. **Extension Point**: `RequestHandlerInterface` with `supports()` and `handle()` methods is the official SDK extension mechanism

**Solutions that didn't work:**

1. **Eval() approach** (attempted earlier): Generated closures dynamically using eval() - correctly criticized as "incredibly brittle"
2. **File generation approach** (current): Generates PHP class files with `__invoke()` methods - overly complex for the problem
3. **Generic array handler** (original bug): `function(array $arguments)` failed because reflection looks for parameter NAME "arguments", not a generic array parameter

## Technical Implementation Approach

### Component 1: Fix CustomCallToolHandler Interface

**Objective**: Correct the CustomCallToolHandler to properly implement RequestHandlerInterface so it can intercept CallToolRequest instances

The existing CustomCallToolHandler at `src/Handler/CustomCallToolHandler.php` has the correct `handle()` logic but wrong interface method. It has `getMethod()` which doesn't exist in `RequestHandlerInterface`.

**Required changes:**

1. Remove the `getMethod()` method entirely (lines 47-49)
2. Add `supports(Request $request): bool` method that returns `$request instanceof CallToolRequest`
3. Keep all existing `handle()` logic - it's already correct (receives arguments as array, passes to bridge, handles exceptions)

**Implementation details:**

```php
public function supports(Request $request): bool {
  return $request instanceof CallToolRequest;
}
```

This ensures our handler matches the exact same requests as SDK's CallToolHandler, but runs first due to handler priority.

### Component 2: Update McpServerFactory Registration

**Objective**: Register the custom handler with the SDK builder and simplify tool registration to use dummy handlers

**Current state (src/McpServerFactory.php:51-78):**
- Uses ToolHandlerFactory to generate handler classes
- Passes generated class string to `addTool()`
- ToolHandlerFactory dependency injected in constructor

**Target state:**
- Instantiate CustomCallToolHandler with bridge and logger dependencies
- Call `$builder->addRequestHandler($customHandler)` to register it
- For each tool, call `addTool()` with a dummy handler like `fn() => null` (never called since custom handler intercepts)
- Remove ToolHandlerFactory from constructor and service definition

**Implementation details:**

```php
public function create(): Server {
  $builder = Server::builder()
    ->setServerInfo('Drupal MCP Server', '1.0.0')
    ->setPaginationLimit(50)
    ->setLogger($this->logger)
    ->setEventDispatcher($this->eventDispatcher)
    ->setSession($this->sessionStore);

  // Register custom handler BEFORE registering tools
  $customHandler = new CustomCallToolHandler($this->mcpBridge, $this->logger);
  $builder->addRequestHandler($customHandler);

  // Register tools with dummy handlers (only for metadata/discovery)
  $enabled_tools = $this->mcpBridge->getEnabledTools();
  foreach ($enabled_tools as $tool_data) {
    $builder->addTool(
      handler: fn() => null,  // Dummy - never called
      name: $tool_data['id'],
      description: $tool_data['description'],
      inputSchema: $tool_data['inputSchema'],
    );
  }

  return $builder->build();
}
```

### Component 3: Service Definition Cleanup

**Objective**: Remove ToolHandlerFactory from service container since it's no longer needed

**Changes to mcp_server.services.yml:**

1. Remove `mcp_server.handler_factory` service definition (lines 16-22)
2. Remove `$handlerFactory` parameter from `mcp_server.server.factory` constructor arguments (line 34)
3. Keep all other services unchanged

### Component 4: Comprehensive Codebase Cleanup

**Objective**: Remove all traces of old implementations, failed approaches, and debugging code to ensure a clean, production-ready codebase

This cleanup phase is critical to remove technical debt and prevent confusion from residual code that was part of exploratory development.

#### Phase 4.1: Delete Obsolete Files

**Files to delete completely:**

1. **`src/Handler/ToolHandlerFactory.php`** - Entire code generation infrastructure (185 lines)
   - Generates PHP class files dynamically
   - Uses string templating for handler creation
   - Contains references to "eval() alternative"
   - No longer needed with custom handler approach

2. **`src/Trait/DebugLoggingTrait.php`** - Unused test debugging utility (78 lines)
   - Not referenced anywhere in codebase
   - Created during troubleshooting but never used
   - Contains reflection-based debug logging
   - Pure technical debt

3. **Generated handler cache directory** - Runtime cleanup
   - Clear `temporary://mcp_handlers/` if exists
   - Can be done defensively in update hook
   - Files are auto-generated, never committed

#### Phase 4.2: Remove Debug Logging Statements

**Production code should not contain troubleshooting log statements:**

1. **`src/McpBridgeService.php:237`** - Remove excessive debug logging
   ```php
   // REMOVE THIS:
   $this->logger->info('executeMcpTool called: tool=@tool, auth_mode=@mode, anonymous=@anon', [
     '@tool' => $mcpName,
     '@mode' => $authentication_mode,
     '@anon' => $is_anonymous ? 'YES' : 'NO',
   ]);
   ```
   - This is excessive debug logging added during authentication troubleshooting
   - Production systems should not log every tool execution with this detail
   - Legitimate ERROR and WARNING logging should remain

2. **`src/McpServerFactory.php:61`** - Remove server build logging
   ```php
   // REMOVE THIS:
   $this->logger->info('Building MCP server with @count enabled tools', ['@count' => count($enabled_tools)]);
   ```
   - Not actionable information for production
   - Server build happens on every request; clutters logs

3. **`src/McpServerFactory.php:63`** - Remove per-tool registration logging
   ```php
   // REMOVE THIS (inside foreach loop):
   $this->logger->info('Registering tool: @id', ['@id' => $tool_data['id']]);
   ```
   - Logs once per tool on every request
   - Creates noise without value in production

4. **`src/Handler/CustomCallToolHandler.php:60, 70`** - Remove debug logging from handler
   ```php
   // REMOVE BOTH OF THESE:
   $this->logger->debug('Executing tool via custom handler', [
     'name' => $tool_name,
     'arguments' => $arguments,
   ]);

   $this->logger->debug('Tool executed successfully', [
     'name' => $tool_name,
     'result_type' => gettype($result),
   ]);
   ```
   - Debug logging added during handler development
   - Called on every tool execution
   - Legitimate exception logging (lines 85, 94, 103) should remain

**Keep legitimate operational logging:**
- Error logging in catch blocks (provides diagnostic value)
- Security-related logging in Controller (authentication failures, scope violations)
- Warning logging for configuration issues

#### Phase 4.3: Remove Obsolete Comments and Documentation

**Update or remove comments referencing old approaches:**

1. **`src/McpServerFactory.php:65-66`** - Remove outdated comment
   ```php
   // REMOVE THESE LINES:
   // Generate handler class with correct parameter signature.
   // Uses file generation instead of eval() for better security and debugging.
   ```
   - References file generation approach that's being removed
   - Replace with comment about custom handler interception

2. **Update class docblock in `src/McpServerFactory.php:15-19`** - Simplify description
   - Current: "This factory creates fully configured MCP Server instances with all necessary tool handlers and settings applied."
   - Update to mention custom handler registration: "This factory creates fully configured MCP Server instances with custom request handler for dynamic tool execution."

#### Phase 4.4: Service Definition Cleanup

**Clean service container definitions:**

1. **`mcp_server.services.yml:15-22`** - Delete entire service definition
   ```yaml
   # DELETE THIS ENTIRE BLOCK:
   mcp_server.handler_factory:
     class: Drupal\mcp_server\Handler\ToolHandlerFactory
     arguments:
       - '@file_system'
       - '@mcp_server.bridge'
   ```

2. **`mcp_server.services.yml:34`** - Remove from factory arguments
   ```yaml
   # REMOVE THIS LINE:
   - '@mcp_server.handler_factory'
   ```

3. **Update comment in `mcp_server.services.yml:23-25`** - Revise factory description
   - Current: "This factory creates fully configured MCP Server instances with all necessary tool handlers and settings applied. Tools are provided dynamically via McpBridgeService based on configuration entities."
   - Update to: "This factory creates fully configured MCP Server instances with custom request handler that intercepts tool calls. Tools are registered for discovery only; execution is handled by CustomCallToolHandler."

#### Phase 4.5: Constructor Parameter Cleanup

**Remove ToolHandlerFactory from McpServerFactory constructor:**

1. **`src/McpServerFactory.php:37-43`** - Update constructor
   ```php
   public function __construct(
     private readonly McpBridgeService $mcpBridge,
     private readonly LoggerInterface $logger,
     private readonly EventDispatcherInterface $eventDispatcher,
     private readonly FileSessionStore $sessionStore,
     // REMOVE THIS LINE:
     private readonly ToolHandlerFactory $handlerFactory,
   ) {}
   ```

2. **`src/McpServerFactory.php:34-35`** - Update constructor docblock
   - Remove `@param \Drupal\mcp_server\Handler\ToolHandlerFactory $handlerFactory` line

#### Phase 4.6: Verification Checklist

After cleanup, verify:

- [ ] No references to `ToolHandlerFactory` in codebase (except this plan)
- [ ] No references to `DebugLoggingTrait` anywhere
- [ ] No `logger->info()` or `logger->debug()` in McpServerFactory
- [ ] No excessive debug logging in McpBridgeService or CustomCallToolHandler
- [ ] Service container definitions only reference services that exist
- [ ] PHPStan analysis passes (no references to deleted classes)
- [ ] Code sniffer passes (no trailing comments or unused imports)
- [ ] All imports are used (`use` statements at top of files)
- [ ] No TODOs, FIXMEs, or HACKs referencing old approaches

#### Phase 4.7: Update Hook for Safe Deployment

**Add defensive update hook to clear cached generated handlers:**

Create `mcp_server.install` update hook:

```php
/**
 * Remove obsolete handler factory cache and rebuild service container.
 */
function mcp_server_update_9001() {
  // Defensively clear any generated handler cache.
  try {
    $file_system = \Drupal::service('file_system');
    $cache_dir = 'temporary://mcp_handlers';
    if (file_exists($file_system->realpath($cache_dir))) {
      $file_system->deleteRecursive($cache_dir);
    }
  }
  catch (\Exception $e) {
    // Safe to ignore - directory may not exist.
  }

  // Service container will be rebuilt automatically.
  return t('Cleared obsolete MCP handler cache.');
}
```

This ensures clean migration for existing deployments.

## Risk Considerations and Mitigation Strategies

### Technical Risks

- **SDK behavior change in custom handler ordering**: If SDK changes `array_merge()` logic in Builder.php
    - **Mitigation**: This is SDK's documented extension point; unlikely to change. If it does, we can verify handler order in tests by checking that our handler is called, not SDK's default.

- **CallToolRequest interface changes**: If SDK changes CallToolRequest structure or adds new fields
    - **Mitigation**: Our handler uses generic Request interface and only checks `instanceof CallToolRequest`. Backward compatible with any CallToolRequest changes that maintain type hierarchy.

- **Arguments format changes**: If SDK changes how arguments are passed in CallToolRequest
    - **Mitigation**: We access `$request->arguments ?? []` which is defensive programming. Unit tests verify argument passing.

### Implementation Risks

- **Existing tests may fail**: Tests currently expect ToolHandlerFactory behavior
    - **Mitigation**: Review all tests that mock or interact with tool handlers. Update test expectations to reflect new architecture. Tests should actually be simpler since no generated code to mock.

- **Service container cache**: Old service definition may be cached
    - **Mitigation**: Run `drush cache:rebuild` after service definition changes. Document this requirement.

- **Breaking changes for existing deployments**: Sites using current implementation may have generated handlers cached
    - **Mitigation**: Generated handlers are in temporary:// which is safe to delete. Add update hook to clear ToolHandlerFactory cache if it exists (defensive).

## Success Criteria

### Primary Success Criteria

1. All existing functional and kernel tests pass with new handler architecture
2. Tool execution works correctly via HTTP transport with dynamically discovered tools
3. Zero PHP files generated at runtime (no writes to temporary://mcp_handlers/)
4. CustomCallToolHandler successfully intercepts all CallToolRequest instances
5. Tool discovery (`tools/list`) returns correct metadata for all enabled tools
6. Authentication and scope validation continue to work correctly

### Quality Assurance Metrics

1. Code coverage remains at or above current levels for handler-related code
2. Static analysis (PHPStan) passes with no new errors
3. Code sniffer (Drupal coding standards) passes for all modified files
4. No regression in error handling - exceptions properly caught and formatted
5. Execution time for tool calls improves (no file generation overhead)
6. Memory usage decreases (no caching of generated class strings)

### Cleanup Verification Metrics

1. **Zero code generation artifacts** - No PHP files created in temporary:// at runtime
2. **No obsolete classes** - ToolHandlerFactory and DebugLoggingTrait completely removed
3. **Clean logging** - No debug/info logging in McpServerFactory or CustomCallToolHandler (except exception handling)
4. **No orphaned imports** - All `use` statements reference classes actually used in the file
5. **Service container clean** - No references to deleted services in mcp_server.services.yml
6. **No stale comments** - All comments about "generated", "eval()", or "reflection" removed or updated
7. **Update hook present** - mcp_server_update_9001() exists and clears handler cache
8. **Search verification** - Grep for "ToolHandlerFactory", "DebugLoggingTrait", "handler_factory" returns zero results (except in this plan)

## Resource Requirements

### Development Skills

- Deep understanding of MCP SDK's Protocol and Builder architecture
- Drupal 11 dependency injection and service container
- PHP 8.3 features (constructor property promotion, readonly properties, type system)
- PHPUnit testing for request handlers
- Understanding of JSON-RPC request/response patterns

### Technical Infrastructure

- Drupal 11.1+ environment with MCP Server module installed
- MCP SDK (vendor/mcp/sdk) - current version with CallToolRequest support
- PHPUnit testing framework
- PHPStan for static analysis
- PHP CodeSniffer for code quality
- Access to functional test environment with HTTP transport

## Implementation Order

1. **Fix CustomCallToolHandler** - Correct the interface implementation (lowest risk, enables testing)
   - Remove `getMethod()` method
   - Add `supports()` method
   - Verify interface compliance

2. **Update McpServerFactory** - Register custom handler and simplify tool registration (core change)
   - Remove ToolHandlerFactory dependency from constructor
   - Instantiate and register CustomCallToolHandler
   - Replace generated handlers with `fn() => null` dummy handlers
   - Update class docblock

3. **Update service definitions** - Remove ToolHandlerFactory from container (dependency cleanup)
   - Delete `mcp_server.handler_factory` service
   - Remove handler factory from factory service arguments
   - Update service comments

4. **Run tests and fix failures** - Verify all tests pass, update any that need adjustment
   - Run full test suite
   - Update test expectations if needed
   - Ensure no regressions

5. **Comprehensive cleanup** - Remove ALL traces of old implementation
   - **Phase 4.1**: Delete ToolHandlerFactory.php and DebugLoggingTrait.php
   - **Phase 4.2**: Remove all debug logging statements (4 locations)
   - **Phase 4.3**: Clean up obsolete comments and documentation
   - **Phase 4.4**: Verify service definitions are clean
   - **Phase 4.5**: Remove unused imports from all modified files
   - **Phase 4.6**: Run verification checklist
   - **Phase 4.7**: Add update hook for safe deployment

6. **Quality assurance pass** - Final verification before commit
   - Run PHPStan analysis (must pass)
   - Run code sniffer (must pass)
   - Manual test of tool execution via HTTP
   - Verify no generated files created at runtime
   - Check logs for absence of debug noise

7. **Documentation update** - Ensure inline docs reflect new architecture
   - Verify all docblocks are accurate
   - Ensure comments explain WHY, not WHAT
   - Remove any references to old approaches

## Notes

- The dummy handler `fn() => null` in `addTool()` will never be called because our custom handler intercepts first. It exists purely to register tool metadata in the SDK's Registry for `tools/list` responses.
- This architecture leverages SDK's designed extension point (`addRequestHandler()`) rather than working around its reflection system, making it future-proof and maintainable.
- The approach eliminates ~200 lines of complex code generation logic (ToolHandlerFactory) in favor of a simple 3-line registration in McpServerFactory.
- No backwards compatibility concerns since generated handlers were temporary and never committed to version control.

### Critical: Cleanup is Not Optional

The comprehensive cleanup phase (Component 4) is **mandatory** and not a "nice-to-have". Here's why:

1. **Technical Debt Prevention**: Leaving old code creates confusion for future maintainers about which approach is current
2. **Security**: Debug logging exposes internal state and execution details in production logs
3. **Performance**: Unnecessary logging on every request adds overhead
4. **Maintainability**: Obsolete classes and comments make the codebase harder to understand
5. **Professional Standards**: Production code should be clean, focused, and free of experimental artifacts

**Do not skip the cleanup phase.** A thorough cleanup is the difference between a professional implementation and a "works but messy" hack. The time investment in cleanup (estimated 30-45 minutes) prevents hours of confusion later.

## Task Dependency Visualization

```mermaid
graph TD
    001[Task 01: Fix CustomCallToolHandler Interface] --> 002[Task 02: Register Custom Handler]
    002 --> 003[Task 03: Comprehensive Cleanup]
    003 --> 004[Task 04: Integration Verification]
```

## Execution Blueprint

**Validation Gates:**
- Reference: `.ai/task-manager/config/hooks/POST_PHASE.md`

### ✅ Phase 1: Interface Correction
**Parallel Tasks:**
- ✔️ Task 01: Fix CustomCallToolHandler Interface

**Objective**: Correct the CustomCallToolHandler to implement the proper RequestHandlerInterface, enabling it to be registered with the MCP SDK builder.

### ✅ Phase 2: Handler Registration
**Parallel Tasks:**
- ✔️ Task 02: Register Custom Handler (depends on: 01)

**Objective**: Update McpServerFactory to instantiate and register the custom handler, removing all code generation infrastructure from the service container.

### ✅ Phase 3: Codebase Cleanup
**Parallel Tasks:**
- ✔️ Task 03: Comprehensive Cleanup (depends on: 02)

**Objective**: Remove all obsolete code generation files, debug logging statements, and outdated comments to ensure a production-ready, maintainable codebase.

### Phase 4: Integration Verification
**Parallel Tasks:**
- Task 04: Integration Verification (depends on: 03)

**Objective**: Execute comprehensive test suite and manual verification to ensure the custom handler architecture works correctly across all scenarios.

### Post-phase Actions

After all phases complete:
1. Verify zero references to deleted classes in codebase
2. Confirm no PHP files generated in temporary:// directory
3. Review logs to ensure only legitimate errors/warnings present
4. Document architecture in module documentation (if needed)

### Execution Summary
- Total Phases: 4
- Total Tasks: 4
- Maximum Parallelism: 1 task (sequential execution required)
- Critical Path Length: 4 phases
- Estimated Time: 2-3 hours total
- Complexity: All tasks ≤4.4 (well within acceptable range)

---

**Note**: Manually archived on 2025-11-16
