---
id: 1
group: handler-implementation
dependencies: []
status: completed
created: 2025-11-11T00:00:00.000Z
skills:
  - php
  - drupal-backend
---
# Fix CustomCallToolHandler Interface Implementation

## Objective
Correct the CustomCallToolHandler to properly implement RequestHandlerInterface by replacing the invalid `getMethod()` method with the required `supports()` method.

## Skills Required
- PHP 8.3+ (interface implementation, instanceof operator)
- Drupal backend development (service patterns, dependency injection)

## Acceptance Criteria
- [ ] `getMethod()` method removed from CustomCallToolHandler
- [ ] `supports(Request $request): bool` method implemented
- [ ] Method returns `$request instanceof CallToolRequest`
- [ ] All existing `handle()` logic remains unchanged
- [ ] PHPStan analysis passes with no interface compliance errors
- [ ] Class docblock accurately describes the handler's purpose

## Technical Requirements

**File to modify**: `src/Handler/CustomCallToolHandler.php`

**Interface contract** (from MCP SDK):
```php
interface RequestHandlerInterface {
    public function supports(Request $request): bool;
    public function handle(Request $request, SessionInterface $session): Response|Error;
}
```

**Current issue**:
- Line 47-49: `getMethod()` method exists but is not part of RequestHandlerInterface
- This prevents the handler from being properly recognized by the SDK's Protocol loop

**Required implementation**:
```php
public function supports(Request $request): bool {
  return $request instanceof CallToolRequest;
}
```

This ensures the handler intercepts ALL CallToolRequest instances before SDK's default CallToolHandler can process them (due to handler priority in Builder::build()).

## Input Dependencies
None - this is the first task and has no dependencies.

## Output Artifacts
- Corrected `src/Handler/CustomCallToolHandler.php` implementing proper interface
- Handler ready to be registered with MCP Server Builder

## Implementation Notes

<details>
<summary>Detailed Implementation Steps</summary>

### Step 1: Remove Invalid Method
Delete lines 47-49 containing:
```php
public function getMethod(): string {
  return 'tools/call';
}
```

### Step 2: Add Required Method
Insert the `supports()` method after the constructor (around line 43):
```php
/**
 * {@inheritdoc}
 */
public function supports(Request $request): bool {
  return $request instanceof CallToolRequest;
}
```

### Step 3: Verify Existing Logic
Confirm that the `handle()` method (lines 54-112) remains unchanged:
- Correctly asserts `$request instanceof CallToolRequest`
- Extracts tool name and arguments from request
- Calls `$this->bridge->executeMcpTool()` with arguments as array
- Handles AuthenticationRequiredException and InsufficientScopeException
- Formats results correctly using `formatResult()` helper
- Returns proper Response or Error objects

### Step 4: Validation
Run static analysis to verify interface compliance:
```bash
cd /var/www/html && vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/Handler/CustomCallToolHandler.php
```

### Why This Works
The SDK's Protocol class (Protocol.php:170-173) loops through registered handlers and calls `supports()` on each:
```php
foreach ($this->requestHandlers as $handler) {
    if (!$handler->supports($request)) {
        continue;
    }
    $handler->handle($request, $session);
    break;  // First match wins!
}
```

Since custom handlers are added via `addRequestHandler()` BEFORE SDK's default handlers (Builder.php:432 uses `array_merge()` with custom handlers first), our handler will match and execute BEFORE the SDK's CallToolHandler ever sees the request.

### Testing Notes
After this change, the handler will be ready for registration in McpServerFactory (next task). The handler can be tested in isolation by:
1. Creating a mock CallToolRequest
2. Verifying `supports()` returns true
3. Calling `handle()` and checking the response format
</details>
