---
id: 2
group: handler-implementation
dependencies:
  - 1
status: completed
created: 2025-11-11T00:00:00.000Z
skills:
  - php
  - drupal-backend
---
# Register Custom Handler and Remove Code Generation

## Objective
Update McpServerFactory to instantiate and register CustomCallToolHandler with the SDK builder, replace generated handlers with dummy handlers for tool metadata, and remove ToolHandlerFactory dependency from the service container.

## Skills Required
- PHP 8.3+ (constructor property promotion, readonly properties, closures)
- Drupal backend development (service container, dependency injection, YAML service definitions)

## Acceptance Criteria
- [ ] CustomCallToolHandler instantiated in `McpServerFactory::create()`
- [ ] Custom handler registered via `$builder->addRequestHandler()`
- [ ] Tool registration uses `fn() => null` dummy handlers
- [ ] ToolHandlerFactory removed from constructor
- [ ] Service definition updated to remove handler_factory service
- [ ] Factory service arguments updated (no handler_factory reference)
- [ ] All imports cleaned up (no unused `use` statements)
- [ ] PHPStan analysis passes
- [ ] Drupal cache rebuild completes successfully

## Technical Requirements

**Files to modify**:
1. `src/McpServerFactory.php` - Update constructor, imports, and create() method
2. `mcp_server.services.yml` - Remove handler_factory service, update factory arguments

**Current state**:
- Factory uses ToolHandlerFactory to generate handler class strings
- Generated handlers are passed to `$builder->addTool()`
- Service container injects handler_factory into factory

**Target state**:
- Factory instantiates CustomCallToolHandler directly
- Registers handler BEFORE tool registration using `addRequestHandler()`
- Tools registered with `fn() => null` dummy handlers (only for metadata)
- No ToolHandlerFactory dependency

## Input Dependencies
- Task 1: CustomCallToolHandler must implement correct interface with `supports()` method

## Output Artifacts
- Modified `src/McpServerFactory.php` with custom handler registration
- Updated `mcp_server.services.yml` with cleaned service definitions
- Working server that intercepts tool calls before SDK's default handler

## Implementation Notes

<details>
<summary>Detailed Implementation Steps</summary>

### Step 1: Update McpServerFactory Constructor

**File**: `src/McpServerFactory.php`

Remove ToolHandlerFactory dependency:

```php
// REMOVE from constructor (line 34-35):
 * @param \Drupal\mcp_server\Handler\ToolHandlerFactory $handlerFactory
 *   The tool handler factory.

// REMOVE from constructor parameters (line 42):
private readonly ToolHandlerFactory $handlerFactory,
```

Remove ToolHandlerFactory import:
```php
// DELETE line 9:
use Drupal\mcp_server\Handler\ToolHandlerFactory;
```

Add CustomCallToolHandler import:
```php
// ADD after line 8:
use Drupal\mcp_server\Handler\CustomCallToolHandler;
```

### Step 2: Update create() Method

**File**: `src/McpServerFactory.php` (lines 51-78)

Replace the current implementation with:

```php
public function create(): Server {
  $builder = Server::builder()
    ->setServerInfo('Drupal MCP Server', '1.0.0')
    ->setPaginationLimit(50)
    ->setLogger($this->logger)
    ->setEventDispatcher($this->eventDispatcher)
    ->setSession($this->sessionStore);

  // Register custom handler to intercept tool calls.
  $customHandler = new CustomCallToolHandler($this->mcpBridge, $this->logger);
  $builder->addRequestHandler($customHandler);

  // Register tools with dummy handlers for metadata discovery only.
  $enabled_tools = $this->mcpBridge->getEnabledTools();
  foreach ($enabled_tools as $tool_data) {
    $builder->addTool(
      handler: fn() => null,  // Never called - custom handler intercepts
      name: $tool_data['id'],
      description: $tool_data['description'],
      inputSchema: $tool_data['inputSchema'],
    );
  }

  return $builder->build();
}
```

**Key changes**:
- Removed all `logger->info()` debug statements (lines 61, 63)
- Removed ToolHandlerFactory usage (lines 65-67)
- Instantiate CustomCallToolHandler with bridge and logger
- Register handler BEFORE tool loop using `addRequestHandler()`
- Use `fn() => null` as dummy handler in `addTool()`
- Removed outdated comments about file generation

### Step 3: Update Service Definitions

**File**: `mcp_server.services.yml`

Delete the handler_factory service (lines 16-22):
```yaml
# DELETE THIS ENTIRE BLOCK:
  mcp_server.handler_factory:
    class: Drupal\mcp_server\Handler\ToolHandlerFactory
    arguments:
      - '@file_system'
      - '@mcp_server.bridge'
```

Update factory service arguments (around line 34):
```yaml
# BEFORE:
      - '@mcp_server.handler_factory'

# AFTER:
# (just remove the line entirely)
```

The factory service definition should look like:
```yaml
  mcp_server.server.factory:
    class: Drupal\mcp_server\McpServerFactory
    arguments:
      - '@mcp_server.bridge'
      - '@logger.channel.mcp_server'
      - '@event_dispatcher'
      - '@mcp_server.session'
      # handler_factory removed - no longer needed
```

### Step 4: Update Class Docblock

**File**: `src/McpServerFactory.php` (lines 15-20)

Update the class docblock to reflect new architecture:
```php
/**
 * Factory for building configured MCP Server instances.
 *
 * This factory creates fully configured MCP Server instances with custom
 * request handler that intercepts tool calls. Tools are registered for
 * discovery only; execution is handled by CustomCallToolHandler.
 */
```

### Step 5: Validation

Run the following commands to verify:

```bash
# Static analysis
cd /var/www/html && vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/McpServerFactory.php

# Check for unused imports
cd /var/www/html && vendor/bin/phpcs --standard=Drupal web/modules/contrib/mcp_server/src/McpServerFactory.php

# Rebuild cache to load new service definitions
cd /var/www/html && vendor/bin/drush cache:rebuild
```

### Why This Works

**Handler Priority** (from MCP SDK Builder.php:432-443):
```php
$requestHandlers = array_merge($this->requestHandlers, [
    new Handler\Request\CallToolHandler(...),  // SDK's default
    new Handler\Request\ListToolsHandler(...),
    // ... other defaults
]);
```

The `array_merge()` puts custom handlers (from `addRequestHandler()`) at the **front** of the array. When Protocol loops through handlers:

1. Our CustomCallToolHandler.supports() is checked FIRST
2. Returns true for CallToolRequest
3. Our handler executes
4. Loop breaks - SDK's CallToolHandler never runs

**Dummy Handlers**:
The `fn() => null` handlers are registered purely for metadata in the SDK's Registry. When `tools/list` is called, ListToolsHandler reads the Registry and returns tool metadata. The handler parameter is stored but never invoked because our custom handler intercepts `tools/call` requests.

### Testing Strategy

After implementation:
1. Start the MCP server
2. Send a `tools/list` request - should return all enabled tools
3. Send a `tools/call` request - should execute via CustomCallToolHandler
4. Check logs - should see custom handler execution, not SDK's reflection errors
5. Verify no files created in `temporary://mcp_handlers/`
</details>
