---
id: 3
group: cleanup
dependencies:
  - 2
status: completed
created: 2025-11-11T00:00:00.000Z
skills:
  - php
  - drupal-backend
---
# Comprehensive Codebase Cleanup

## Objective
Remove all traces of obsolete code generation infrastructure, unused debugging utilities, excessive debug logging, and outdated comments to ensure a clean, production-ready codebase.

## Skills Required
- PHP 8.3+ (file operations, code analysis)
- Drupal backend development (update hooks, cache management, logging best practices)

## Acceptance Criteria
- [ ] ToolHandlerFactory.php deleted
- [ ] DebugLoggingTrait.php deleted
- [ ] All debug/info logging removed from McpBridgeService, CustomCallToolHandler
- [ ] Obsolete comments removed from all modified files
- [ ] Update hook added to clear handler cache
- [ ] No unused imports in any modified files
- [ ] Grep verification: no references to deleted classes (except this task)
- [ ] PHPStan analysis passes
- [ ] PHP CodeSniffer passes for all modified files

## Technical Requirements

**Files to delete**:
1. `src/Handler/ToolHandlerFactory.php` (185 lines)
2. `src/Trait/DebugLoggingTrait.php` (78 lines)

**Files to clean**:
1. `src/McpBridgeService.php` - Remove debug logging
2. `src/Handler/CustomCallToolHandler.php` - Remove debug logging
3. `mcp_server.install` - Add update hook

**Logging removal targets** (4 locations):
- McpBridgeService:237 - executeMcpTool debug logging
- CustomCallToolHandler:60 - tool execution start
- CustomCallToolHandler:70 - tool execution success

**Keep legitimate logging**:
- Error logging in catch blocks
- Security warnings (authentication failures, scope violations)
- Warning logging for configuration issues

## Input Dependencies
- Task 2: McpServerFactory must be updated and ToolHandlerFactory no longer referenced

## Output Artifacts
- Clean codebase with ~260 lines of obsolete code removed
- Production-ready logging (errors/warnings only, no debug noise)
- Update hook for safe deployment to existing sites
- Zero references to deleted classes in codebase

## Implementation Notes

<details>
<summary>Detailed Implementation Steps</summary>

### Phase 1: Delete Obsolete Files

```bash
# Delete code generation infrastructure
rm web/modules/contrib/mcp_server/src/Handler/ToolHandlerFactory.php

# Delete unused test debugging utility
rm web/modules/contrib/mcp_server/src/Trait/DebugLoggingTrait.php

# Defensively clear any generated handler cache (may not exist)
rm -rf web/sites/default/files/php/mcp_handlers/ 2>/dev/null || true
```

### Phase 2: Remove Debug Logging

**File**: `src/McpBridgeService.php`

Find and remove excessive info logging (around line 237):
```php
// REMOVE THIS ENTIRE BLOCK:
$this->logger->info('executeMcpTool called: tool=@tool, auth_mode=@mode, anonymous=@anon', [
  '@tool' => $mcpName,
  '@mode' => $authentication_mode,
  '@anon' => $is_anonymous ? 'YES' : 'NO',
]);
```

**File**: `src/Handler/CustomCallToolHandler.php`

Remove both debug logging statements but keep exception logging:

```php
// REMOVE line ~60:
$this->logger->debug('Executing tool via custom handler', [
  'name' => $tool_name,
  'arguments' => $arguments,
]);

// REMOVE line ~70:
$this->logger->debug('Tool executed successfully', [
  'name' => $tool_name,
  'result_type' => gettype($result),
]);

// KEEP exception logging at lines 85, 94, 103 - these provide diagnostic value
```

### Phase 3: Clean Up Unused Imports

After deleting ToolHandlerFactory, check McpServerFactory for unused imports:

**File**: `src/McpServerFactory.php`

Verify these imports are present and used:
- `CustomCallToolHandler` - used in create()
- `McpBridgeService` - used in constructor
- `Server` - used in create()
- `FileSessionStore` - used in constructor
- `EventDispatcherInterface` - used in constructor
- `LoggerInterface` - used in constructor

No imports should remain from deleted classes.

### Phase 4: Add Update Hook

**File**: `mcp_server.install` (create if doesn't exist)

Add defensive cleanup hook:

```php
<?php

declare(strict_types=1);

/**
 * @file
 * Install, update and uninstall functions for the MCP Server module.
 */

/**
 * Remove obsolete handler factory cache and rebuild service container.
 */
function mcp_server_update_9001(): string {
  // Defensively clear any generated handler cache.
  try {
    $file_system = \Drupal::service('file_system');
    $cache_dir = 'temporary://mcp_handlers';
    $real_path = $file_system->realpath($cache_dir);
    if ($real_path && file_exists($real_path)) {
      $file_system->deleteRecursive($cache_dir);
    }
  }
  catch (\Exception $e) {
    // Safe to ignore - directory may not exist.
  }

  // Service container will be rebuilt automatically after updates.
  return t('Cleared obsolete MCP handler cache.');
}
```

**Important**: Follow Drupal strict coding standards:
- Use `declare(strict_types=1);`
- Include file docblock
- Use proper type hints
- Return translated string from update hook

### Phase 5: Verification Checklist

Run these commands to verify cleanup:

```bash
# 1. Check for references to deleted classes (should return nothing except this task)
cd web/modules/contrib/mcp_server
grep -r "ToolHandlerFactory" --exclude-dir=.ai . || echo "Clean: no references"
grep -r "DebugLoggingTrait" . || echo "Clean: no references"

# 2. Verify no excessive logging remains
grep -n "logger->info.*tool" src/McpBridgeService.php src/McpServerFactory.php || echo "Clean: no debug logging"
grep -n "logger->debug" src/Handler/CustomCallToolHandler.php || echo "Clean: no debug logging"

# 3. Run static analysis
cd /var/www/html && vendor/bin/phpstan analyse web/modules/contrib/mcp_server/

# 4. Check coding standards
cd /var/www/html && vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/src/

# 5. Verify update hook syntax
cd /var/www/html && vendor/bin/phpcs --standard=Drupal web/modules/contrib/mcp_server/mcp_server.install

# 6. Test update hook execution
cd /var/www/html && vendor/bin/drush updatedb -y
```

### Phase 6: Update Comments

**File**: `src/Handler/CustomCallToolHandler.php`

Verify class docblock is accurate (lines 20-28):
```php
/**
 * Custom tool call handler that bypasses SDK's parameter name matching.
 *
 * This handler receives tool call requests and passes arguments directly
 * to the bridge service as an associative array, avoiding the need for
 * reflection-based parameter mapping or code generation.
 *
 * @implements RequestHandlerInterface<CallToolResult>
 */
```

This is already correct - no changes needed.

### What Logging to Keep

**Keep these patterns**:
- `$this->logger->error()` in catch blocks (diagnostic value)
- `$this->logger->warning()` for configuration issues
- Security-related logging in Controller (authentication, authorization)

**Remove these patterns**:
- `$this->logger->info()` on every request/tool execution
- `$this->logger->debug()` for routine operations
- Tool registration logging
- Server build logging

**Rationale**: Production logs should contain actionable information. Debug logging added during development creates noise and provides no operational value. Exception logging remains because it aids troubleshooting when things go wrong.

### Testing After Cleanup

1. **Service container**: Verify services resolve correctly after cache rebuild
2. **Tool execution**: Confirm tools execute without errors
3. **Log files**: Check that logs contain only errors/warnings, no debug noise
4. **File system**: Verify no generated handlers created at runtime
5. **Update hook**: Test hook runs without errors on fresh install

### Common Issues

**Issue**: PHPStan complains about unused imports
**Fix**: Remove any `use` statements for deleted classes

**Issue**: Update hook fails with file_system service error
**Fix**: Verify try-catch wraps the entire cleanup block

**Issue**: Tests fail after cleanup
**Fix**: Update mocks that referenced ToolHandlerFactory

**Issue**: CodeSniffer complains about line length in update hook
**Fix**: Break long lines at logical points (parameter lists, etc.)
</details>
