---
id: 4
group: testing
dependencies:
  - 3
status: completed
created: 2025-11-11T00:00:00.000Z
skills:
  - phpunit
  - drupal-backend
---
# Integration Testing and Verification

## Objective
Verify that the custom handler architecture works correctly by running the full test suite and performing integration testing to ensure tool discovery, execution, authentication, and error handling all function as expected.

## Skills Required
- PHPUnit testing (test execution, failure analysis, debugging)
- Drupal backend development (functional testing, kernel testing, cache management)

## Acceptance Criteria
- [ ] All existing kernel tests pass
- [ ] All existing functional tests pass
- [ ] Tool discovery (tools/list) returns correct metadata
- [ ] Tool execution (tools/call) works with dynamically discovered tools
- [ ] Authentication and scope validation continue to work
- [ ] Error handling properly catches and formats exceptions
- [ ] No PHP files generated in temporary:// during test runs
- [ ] Zero regressions in code coverage
- [ ] Manual HTTP transport test succeeds

## Technical Requirements

**Test execution strategy**:
1. Run kernel tests first (faster, catch basic issues)
2. Run functional tests (slower, verify HTTP transport)
3. Manual verification of key workflows
4. Performance check (no file generation overhead)

**Key test files to monitor**:
- `tests/src/Kernel/McpBridgeServiceTest.php` - Core bridge logic
- `tests/src/Kernel/McpToolConfigTest.php` - Tool configuration
- `tests/src/Functional/McpServerFunctionalTest.php` - HTTP transport

## Input Dependencies
- Task 3: Cleanup completed, all obsolete code removed
- Task 2: Custom handler registered in McpServerFactory
- Task 1: CustomCallToolHandler implements correct interface

## Output Artifacts
- Test execution report showing all passes
- Verification that no generated files exist after tests
- Confirmation that logs contain only legitimate errors/warnings
- Evidence that custom handler intercepts tool calls correctly

## Implementation Notes

<details>
<summary>Detailed Implementation Steps</summary>

### Meaningful Test Strategy Guidelines

**IMPORTANT**: Copy these guidelines when creating or modifying tests.

Your critical mantra for test generation is: "write a few tests, mostly integration".

**Definition of "Meaningful Tests":**
Tests that verify custom business logic, critical paths, and edge cases specific to the application. Focus on testing YOUR code, not the framework or library functionality.

**When TO Write Tests:**
- Custom business logic and algorithms
- Critical user workflows and data transformations
- Edge cases and error conditions for core functionality
- Integration points between different system components
- Complex validation logic or calculations

**When NOT to Write Tests:**
- Third-party library functionality (already tested upstream)
- Framework features (React hooks, Express middleware, etc.)
- Simple CRUD operations without custom logic
- Getter/setter methods or basic property access
- Configuration files or static data
- Obvious functionality that would break immediately if incorrect

**Test Task Creation Rules:**
- Combine related test scenarios into single tasks
- Focus on integration and critical path testing over unit test coverage
- Avoid creating separate tasks for testing each CRUD operation individually
- Question whether simple functions need dedicated test tasks

### Step 1: Environment Preparation

```bash
# Clear cache to ensure fresh service container
cd /var/www/html && vendor/bin/drush cache:rebuild

# Verify no leftover generated handlers exist
ls -la web/sites/default/files/php/mcp_handlers/ 2>/dev/null || echo "Clean: no generated files"
```

### Step 2: Run Kernel Tests

Execute kernel test suite (fast, no HTTP stack):

```bash
cd /var/www/html && vendor/bin/phpunit --testsuite=kernel web/modules/contrib/mcp_server/tests/
```

**Expected results**:
- All tests pass
- No warnings about missing services
- No file generation during test execution

**Key tests to verify**:
1. `McpBridgeServiceTest` - Tool execution through bridge
2. `McpToolConfigTest` - Configuration entity handling
3. `OAuthScopeIntegrationTest` - Scope discovery and validation

**Common issues**:
- **Mock failures**: If tests mock ToolHandlerFactory, update mocks to reflect new architecture
- **Service resolution**: If container can't resolve services, check service definitions
- **Argument passing**: Verify arguments passed as arrays, not individual parameters

### Step 3: Run Functional Tests

Execute functional test suite (slower, includes HTTP transport):

```bash
cd /var/www/html && vendor/bin/phpunit --testsuite=functional web/modules/contrib/mcp_server/tests/
```

**Expected results**:
- All tests pass
- JSON-RPC communication works correctly
- Tools execute via HTTP transport
- Authentication/authorization properly enforced

**Key test**: `McpServerFunctionalTest`
- Verifies end-to-end tool execution
- Tests HTTP transport with real JSON-RPC messages
- Validates authentication and scope checking

### Step 4: Manual Verification

**Verify tool discovery**:
1. Start MCP server or use functional test
2. Send `tools/list` JSON-RPC request
3. Confirm response contains all enabled tools with correct metadata

**Verify tool execution**:
1. Send `tools/call` request with valid tool name and arguments
2. Confirm custom handler intercepts (check logs)
3. Verify result format matches CallToolResult schema
4. Test with invalid arguments to verify error handling

**Check logs**:
```bash
# Should contain ONLY errors/warnings, no debug noise
tail -100 web/sites/default/files/logs/drupal.log | grep mcp_server
```

Expected log patterns:
- ✅ Error logs when tool execution fails
- ✅ Warning logs for missing configuration
- ✅ Info logs for authentication failures (security)
- ❌ Debug logs for routine operations
- ❌ Info logs for every tool execution
- ❌ Info logs for server build or tool registration

### Step 5: Verify No Code Generation

```bash
# Check temporary directory - should be empty or not exist
ls -la web/sites/default/files/php/mcp_handlers/ 2>/dev/null && echo "ERROR: Generated files found!" || echo "✓ Clean: No generated files"

# Verify ToolHandlerFactory is completely removed
grep -r "ToolHandlerFactory" web/modules/contrib/mcp_server/src/ web/modules/contrib/mcp_server/mcp_server.services.yml && echo "ERROR: References remain!" || echo "✓ Clean: No ToolHandlerFactory references"
```

### Step 6: Performance Check

Compare execution time before and after changes:

```bash
# Run with timing
cd /var/www/html && time vendor/bin/phpunit --testsuite=kernel web/modules/contrib/mcp_server/tests/
```

Expected improvement:
- Slight reduction in test execution time (no file I/O for handler generation)
- Lower memory usage (no caching of generated class strings)

### Step 7: Regression Testing

**Test existing functionality**:
1. **Authentication**: Tools requiring authentication should enforce it
2. **Scope validation**: Tools requiring OAuth scopes should check them
3. **Anonymous access**: Tools allowing anonymous should work without authentication
4. **Error handling**: Invalid tool names, missing arguments should return proper errors
5. **Tool configuration**: Enabling/disabling tools should work correctly

**Verify no breaking changes**:
- All configuration entities still work
- Tool API integration unchanged
- OAuth scope discovery functional
- Event subscribers still triggered

### Step 8: Test Update Hook

```bash
# Run update hook (should complete without errors)
cd /var/www/html && vendor/bin/drush updatedb -y

# Verify hook ran
cd /var/www/html && vendor/bin/drush updatedb --status | grep "mcp_server"
```

Expected: Update 9001 should show as completed.

### Step 9: Code Quality Verification

```bash
# Static analysis
cd /var/www/html && vendor/bin/phpstan analyse web/modules/contrib/mcp_server/

# Coding standards
cd /var/www/html && vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/

# Check test coverage (optional)
cd /var/www/html && vendor/bin/phpunit --coverage-text web/modules/contrib/mcp_server/tests/
```

All checks should pass with no new errors or warnings.

### Debugging Failed Tests

**Issue**: Tests fail with "Handler not found" error
**Analysis**: Custom handler not properly registered
**Fix**: Verify `addRequestHandler()` called in McpServerFactory::create()

**Issue**: Tests fail with "Invalid handler" error
**Analysis**: CustomCallToolHandler missing `supports()` method
**Fix**: Verify Task 1 completed correctly

**Issue**: Tests mock ToolHandlerFactory and fail
**Analysis**: Tests still reference old architecture
**Fix**: Update test setup to mock CustomCallToolHandler or bridge service directly

**Issue**: Tool arguments not passed correctly
**Analysis**: Arguments may be formatted incorrectly
**Fix**: Verify `$request->arguments ?? []` extracts associative array correctly

### Success Indicators

✅ **All tests pass** - Zero failures, zero errors
✅ **No generated files** - Empty temporary:// handler directory
✅ **Clean logs** - Only legitimate errors/warnings
✅ **Handler intercepts** - Custom handler processes all CallToolRequest
✅ **Performance stable** - No regression in execution time
✅ **Code quality** - PHPStan and CodeSniffer pass
✅ **Update hook works** - Hook 9001 completes successfully

### Final Verification Checklist

Before marking this task complete:

- [ ] Ran full kernel test suite - all pass
- [ ] Ran full functional test suite - all pass
- [ ] Verified no generated PHP files exist
- [ ] Checked logs - no excessive debug logging
- [ ] Confirmed CustomCallToolHandler intercepts tool calls
- [ ] Tested tool discovery - returns correct metadata
- [ ] Tested tool execution - works with dynamic arguments
- [ ] Verified authentication still enforced
- [ ] Verified OAuth scope validation works
- [ ] Ran PHPStan - passes with no new errors
- [ ] Ran CodeSniffer - passes for all modified files
- [ ] Executed update hook - completes successfully
- [ ] Manual HTTP test - tool execution works end-to-end

If all items check out, the integration is successful and the plan is complete.
</details>
