---
id: 7
summary: "Remove optional authentication mode and implement configuration settings to address architectural simplifications identified in code review"
created: 2025-11-13
---

# Plan: Architecture Simplifications

## Original Work Order
> Address the concerns in @ARCHITECTURE_REVIEW.md

## Executive Summary

This plan addresses two high-priority architectural concerns identified in the comprehensive code review: the "optional" authentication mode gold plating and the unused configuration schema. The implementation will remove the ambiguous "optional" authentication mode, streamline authentication logic, and properly implement configuration settings that are currently defined but not used. This work will reduce cognitive complexity, eliminate misleading documentation, and ensure configuration values are properly respected throughout the codebase.

The approach focuses on surgical improvements to existing code while maintaining backward compatibility is not a concern since the module is unreleased. All changes will include comprehensive test updates and documentation revisions to reflect the simplified architecture.

## Context

### Current State

The MCP Server module currently has two architectural issues that reduce code quality and create confusion:

1. **Optional Authentication Mode**: The authentication system supports three modes (required/optional/disabled), but the "optional" mode has unclear semantics - it logs warnings for authenticated users with insufficient scopes but executes anyway. This creates a false sense of security and adds 11 lines of branching logic without clear value.

2. **Unused Configuration Schema**: Four configuration values (`server_name`, `server_version`, `pagination_limit`, `session_ttl`) are defined in the schema and default config file, documented in README.md, but hardcoded in `McpServerFactory` instead of being read from configuration. This misleads users into thinking these values are configurable.

The codebase is otherwise well-architected with 91% test coverage and proper use of Drupal patterns.

### Target State

After implementation:

1. **Binary Authentication**: Only two authentication modes exist: `required` (enforce OAuth scopes) and `disabled` (no authentication). The ambiguous "optional" mode is removed.

2. **Functional Configuration**: All four configuration values are properly read from `mcp_server.settings` config and used by the factory, with validation to ensure correct types and ranges.

3. **Updated Tests**: All test coverage reflects the simplified authentication logic and validates configuration usage.

4. **Accurate Documentation**: README.md accurately describes the two authentication modes and working configuration system.

### Background

The architecture review (ARCHITECTURE_REVIEW.md) identified 5 potential simplifications. This plan addresses the 2 high-priority items:
- Finding #1: Authentication "Optional" Mode (HIGH PRIORITY)
- Finding #2: Unused Configuration Schema (HIGH PRIORITY)

The 3 low-priority items (scope caching, error marker detection, McpBridgeService complexity) are explicitly deferred as they represent acceptable trade-offs or necessary workarounds.

Since the module is unreleased, no migration paths or backward compatibility layers are needed.

## Technical Implementation Approach

### Component 1: Remove Optional Authentication Mode

**Objective**: Eliminate the ambiguous "optional" authentication mode to reduce cognitive complexity and clarify security semantics.

**Implementation Details**:

1. **Schema Update**: Modify `config/schema/mcp_server.schema.yml` to restrict `authentication_mode` to two values: `[required, disabled]`

2. **Entity Method Update**: Update `McpToolConfig::requiresAuthentication()` to return `true` only for "required" mode (disabled returns false)

3. **Bridge Service Simplification**: Remove lines 254-265 from `McpBridgeService::executeMcpTool()` that handle optional mode. The authentication flow becomes:
   - If mode is "disabled": Skip all checks, execute tool
   - If mode is "required": Check authentication, validate scopes (throw on failure), execute tool
   - Remove the third branch entirely

4. **Validation Method Update**: The `validateToolScopes()` method no longer needs the `$throwOnFailure` parameter - it always throws on failure. Simplify to single behavior.

5. **Test Updates**:
   - Remove test cases covering optional mode behavior
   - Add tests confirming only two modes are valid
   - Update integration tests to reflect binary authentication

**Files Modified**:
- `config/schema/mcp_server.schema.yml`
- `src/Entity/McpToolConfig.php`
- `src/McpBridgeService.php`
- `tests/src/Kernel/OAuthScopeIntegrationTest.php`
- `tests/src/Unit/McpBridgeServiceTest.php`

**Lines Removed**: Approximately 15-20 lines of branching logic

### Component 2: Implement Configuration Settings

**Objective**: Make the four defined configuration values (`server_name`, `server_version`, `pagination_limit`, `session_ttl`) functional by reading them from config and using them appropriately.

**Implementation Details**:

1. **Factory Service Definition Update**: Inject `config.factory` service into `McpServerFactory` to access `mcp_server.settings`

2. **Factory Constructor Update**: Add `ConfigFactoryInterface` parameter to constructor with readonly property

3. **Factory Configuration Reading**: In `create()` method, read all four config values:
   ```php
   $config = $this->configFactory->get('mcp_server.settings');
   $server_name = $config->get('server_name') ?? 'Drupal MCP Server';
   $server_version = $config->get('server_version') ?? '1.0.0';
   $pagination_limit = $config->get('pagination_limit') ?? 50;
   // session_ttl is passed to FileSessionStore
   ```

4. **Configuration Usage**:
   - `server_name` and `server_version`: Pass to `setServerInfo()`
   - `pagination_limit`: Pass to `setPaginationLimit()` with validation (must be positive integer)
   - `session_ttl`: Pass to `FileSessionStore` constructor (requires service definition update)

5. **Service Definition Update**: Modify `mcp_server.services.yml`:
   - Add `@config.factory` argument to `mcp_server.server_factory`
   - Update `mcp_server.session_store` to accept TTL parameter from factory

6. **Validation**: Add basic validation for configuration values:
   - `pagination_limit`: Must be integer > 0, max 1000
   - `session_ttl`: Must be integer > 0
   - String values: Non-empty

7. **Test Updates**:
   - Add unit tests for `McpServerFactory` that mock config values
   - Verify factory reads and applies configuration correctly
   - Test validation of out-of-range values
   - Integration tests with custom config values

**Files Modified**:
- `mcp_server.services.yml`
- `src/McpServerFactory.php`
- `tests/src/Unit/McpServerFactoryTest.php` (new file)
- `tests/src/Kernel/McpServerConfigTest.php` (new file)

**Lines Added**: Approximately 40-50 lines for config reading and validation

### Component 3: Documentation Updates

**Objective**: Update README.md to accurately reflect the simplified authentication system and functional configuration.

**Implementation Details**:

1. **Authentication Mode Documentation**:
   - Update section describing authentication modes
   - Remove references to "optional" mode
   - Clarify binary choice: required (OAuth enforced) vs disabled (no auth)
   - Add examples of when to use each mode

2. **Configuration Documentation**:
   - Verify the Configuration section (lines 268-298) accurately describes the four working settings
   - Add notes about validation ranges and defaults
   - Clarify that changes require cache rebuild

3. **Example Updates**: Update any code examples that reference "optional" mode

**Files Modified**:
- `README.md`

## Risk Considerations and Mitigation Strategies

### Technical Risks

- **Configuration Migration**: Existing config entities using "optional" mode will be invalid
    - **Mitigation**: Since module is unreleased, no migration needed. If any dev sites exist, manual config update is acceptable.

- **Session Store TTL Injection**: FileSessionStore may not accept TTL in constructor
    - **Mitigation**: Review FileSessionStore API before implementation. If not supported, document session_ttl as future enhancement and keep hardcoded.

### Implementation Risks

- **Test Coverage Gaps**: Removing optional mode might expose untested code paths
    - **Mitigation**: Run full test suite before and after changes. Add tests for newly simplified flows.

- **Breaking Existing Integrations**: Developers using "optional" mode will need to update
    - **Mitigation**: Acceptable since module is unreleased. Add clear upgrade notes in commit message.

## Success Criteria

### Primary Success Criteria

1. Configuration schema only accepts "required" and "disabled" authentication modes
2. McpBridgeService has 11+ fewer lines of authentication branching logic
3. McpServerFactory reads all four configuration values from mcp_server.settings
4. Server info (name, version) and pagination limit are configurable and functional
5. All existing tests pass with updated authentication logic
6. README.md accurately documents the two authentication modes

### Quality Assurance Metrics

1. PHPUnit test suite passes with 100% success rate
2. PHPStan static analysis passes with no new errors
3. Code coverage maintains or exceeds 91% ratio
4. Manual testing confirms configuration values are respected (change server name, verify in MCP response)

## Resource Requirements

### Development Skills

- Drupal configuration system and schema definitions
- Drupal service container and dependency injection
- PHP unit testing and mocking
- Drupal entity system and validation
- OAuth authentication patterns (to understand what's being simplified)

### Technical Infrastructure

- PHP 8.3+ with Drupal 11.x
- PHPUnit for test execution
- PHPStan for static analysis
- Drupal's config system and entity API
- MCP Server PHP SDK (understanding for configuration integration)

## Implementation Order

1. **Authentication Mode Removal** (Component 1): Cleaner to do this first as it's purely deletion/simplification
2. **Configuration Implementation** (Component 2): More complex, benefits from simplified codebase
3. **Documentation Updates** (Component 3): Final step to ensure docs match implementation

## Notes

- This work addresses 2 of 5 findings from the architecture review
- The other 3 findings are deliberately deferred:
  - Scope caching (Finding #3): Low priority, acceptable optimization
  - Error marker detection (Finding #4): Low priority, necessary SDK workaround
  - McpBridgeService complexity (Finding #5): Low priority, acceptable cohesion
- All changes are non-breaking due to unreleased status
- Configuration values should have sensible defaults to avoid deployment issues
- Consider adding admin UI form for configuration in future (not in scope for this plan)

## Task Dependencies

```mermaid
graph TD
    001[Task 01: Remove Optional Auth Mode] --> 002[Task 02: Update Auth Tests]
    001 --> 003[Task 03: Implement Config Settings]
    003 --> 004[Task 04: Add Config Tests]
    002 --> 005[Task 05: Update Documentation]
    004 --> 005
```

## Execution Blueprint

**Validation Gates:**
- Reference: `.ai/task-manager/config/hooks/POST_PHASE.md`

### ✅ Phase 1: Foundation - Authentication Simplification
**Parallel Tasks:**
- ✔️ Task 01: Remove Optional Auth Mode (authentication-simplification) - **COMPLETED**

**Rationale:** This foundational change must complete first as it simplifies the codebase for all subsequent work.

**Completion Notes:**
- Config schema updated to binary authentication (required/disabled)
- McpBridgeService simplified by removing 15+ lines of optional mode logic
- validateToolScopes() method simplified to always throw on failure
- McpToolConfig default changed from 'optional' to 'disabled'
- Linting: PASSED
- Tests: 2 errors, 5 failures (expected - will be fixed in Phase 2)

### ⚠️ Phase 2: Validation and Enhancement (PARTIALLY COMPLETE - BUGS FOUND)
**Parallel Tasks:**
- ✔️ Task 02: Update Auth Tests (authentication-simplification) - **COMPLETED** - depends on: 01
- ✔️ Task 03: Implement Config Settings (configuration-implementation) - **COMPLETED** - depends on: 01

**Rationale:** Tests and configuration can be developed in parallel once authentication is simplified.

**Completion Notes:**
- **Task 02**: Successfully removed all "optional" mode tests (3 test methods removed, 3 test methods updated)
- **Task 03**: Configuration system fully implemented - all 4 settings now functional with validation
- Session TTL: Successfully implemented via static factory method
- Linting: PASSED
- **CRITICAL ISSUE**: Task 02 revealed 5 test failures caused by bugs in Task 01:
  1. Scope handling broken in McpToolConfig::getScopeNames()
  2. OAuthScopeDiscoveryService returning empty arrays
  3. Scope validation not throwing expected exceptions
  4. OAuth metadata endpoint missing scopes_supported field

**Status**: Phase 2 tasks completed but cannot proceed to Phase 3/4 until Task 01 bugs are fixed.

### Phase 3: Comprehensive Testing
**Parallel Tasks:**
- Task 04: Add Config Tests (configuration-implementation) - depends on: 03

**Rationale:** Configuration tests require implementation to be complete.

### Phase 4: Documentation
**Parallel Tasks:**
- Task 05: Update Documentation (documentation) - depends on: 02, 04

**Rationale:** Documentation is final step to ensure accuracy after all implementation and tests are complete.

### Post-phase Actions
- After Phase 4: Run full test suite and static analysis
- Verify all acceptance criteria from Success Criteria section
- Manual testing of configuration changes

### Execution Summary
- Total Phases: 4
- Total Tasks: 5
- Maximum Parallelism: 2 tasks (in Phase 2)
- Critical Path Length: 4 phases
- Task Groups: 2 (authentication-simplification, configuration-implementation, documentation)

## Execution Summary

**Status**: ⚠️ Partially Completed (Phases 1-2 Complete, Phases 3-4 Blocked)
**Completed Date**: 2025-11-13

### Results

**Phases Completed:**
- ✅ **Phase 1**: Authentication simplification completed successfully
  - Removed "optional" authentication mode from codebase
  - Config schema restricted to binary modes (required/disabled)
  - McpBridgeService simplified by 15+ lines
  - Linting passed, commit created

- ✅ **Phase 2**: Test updates and configuration implementation completed
  - Test suite updated to remove all "optional" mode references (3 test methods removed, 3 updated)
  - Configuration system fully implemented - all 4 settings now functional
  - Session TTL implemented via static factory method
  - Linting passed, commit created

**Tasks Completed**: 3 of 5 (Task 01, 02, 03)
**Tasks Blocked**: 2 of 5 (Task 04, 05)

### Noteworthy Events

**Critical Issue Discovered:**
During Task 02 execution, comprehensive testing revealed 5 test failures caused by bugs introduced in Task 01:

1. **testRequiredAuthorization**: McpToolConfig::getScopeNames() returning empty array instead of configured scopes
2. **testScopeAggregation**: OAuthScopeDiscoveryService::getScopesSupported() returning empty instead of aggregated scopes
3. **testDisabledToolsIgnored**: Missing expected scope in results
4. **testRequiredModeWithScopesFailsWithoutToken**: Scope validation not throwing InsufficientScopeException
5. **testEndToEndWorkflows**: OAuth metadata endpoint missing scopes_supported field

**Root Cause:**
The bugs appear to be in scope field handling within McpToolConfig entity. When the entity was modified in Task 01 to remove "optional" mode handling, the scope retrieval/storage mechanism was inadvertently broken.

**Impact:**
- Cannot proceed to Phase 3 (Add Config Tests) - new tests would also fail
- Cannot proceed to Phase 4 (Update Documentation) - docs would be inaccurate
- Phase 2 tasks (02, 03) are complete and correct - the tests properly identify production code bugs

### Recommendations

**Immediate Action Required:**
1. **Fix Task 01 Scope Bugs**: Investigate and repair McpToolConfig::getScopeNames() and related scope handling
   - Review changes to src/Entity/McpToolConfig.php from Task 01
   - Ensure scope field is properly read and returned
   - Verify OAuthScopeDiscoveryService can aggregate scopes from configs

2. **Re-run Test Suite**: After fixing scope bugs, verify all 25 tests pass

3. **Resume Blueprint Execution**: Once tests are green, continue with:
   - Phase 3: Task 04 (Add Config Tests)
   - Phase 4: Task 05 (Update Documentation)
   - Post-execution: Archive plan

**Success Criteria for Completion:**
- All 25 tests passing (currently 20/25)
- All 5 tasks completed
- README.md updated to reflect binary authentication
- Plan archived with full execution summary

**Lessons Learned:**
- Task 01 should have run the full test suite before completion, not just noted expected failures
- Early comprehensive testing would have caught scope bugs immediately
- Breaking changes require extra scrutiny of related functionality (authentication → scopes)

**Estimated Remaining Effort:**
- Bug fixes: 30-60 minutes
- Task 04 (Config Tests): 30 minutes
- Task 05 (Documentation): 20 minutes
- Total: ~2 hours to complete plan
