---
id: 1
group: "authentication-simplification"
dependencies: []
status: "completed"
created: 2025-11-13
skills:
  - drupal-backend
  - php-refactoring
---
# Remove Optional Authentication Mode

## Objective
Remove the "optional" authentication mode from the codebase, simplifying authentication logic to a binary choice: required or disabled.

## Skills Required
- **Drupal Backend Development**: Understanding of Drupal config schema, entity systems, and service patterns
- **PHP Refactoring**: Safely removing code paths while maintaining test coverage

## Acceptance Criteria
- [ ] Config schema restricts authentication_mode to [required, disabled]
- [ ] McpBridgeService removes lines 254-265 (optional mode handling)
- [ ] validateToolScopes() method simplified to always throw on failure
- [ ] McpToolConfig::requiresAuthentication() updated for binary logic
- [ ] All references to "optional" mode removed from code

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

**Files to Modify:**
1. `config/schema/mcp_server.schema.yml` - Line 50: Update AllowedValues constraint
2. `src/McpBridgeService.php` - Lines 254-265: Remove optional authentication branch
3. `src/McpBridgeService.php` - Line 407: Remove $throwOnFailure parameter from validateToolScopes()
4. `src/Entity/McpToolConfig.php` - Update requiresAuthentication() logic if needed

**Key Changes:**
- Schema: `constraints: AllowedValues: [required, disabled]` (remove "optional")
- McpBridgeService::executeMcpTool(): Remove if ($authentication_mode === 'optional') block entirely
- McpBridgeService::validateToolScopes(): Remove $throwOnFailure parameter, always throw InsufficientScopeException

## Input Dependencies
None - this is a foundational simplification task.

## Output Artifacts
- Simplified authentication flow with binary logic (required/disabled)
- Reduced branching complexity in McpBridgeService
- Updated config schema with two authentication modes

## Implementation Notes

<details>
<summary>Detailed Implementation Steps</summary>

### Step 1: Update Config Schema
File: `config/schema/mcp_server.schema.yml`

Locate line 50 with:
```yaml
constraints:
  AllowedValues: [required, optional, disabled]
```

Change to:
```yaml
constraints:
  AllowedValues: [required, disabled]
```

### Step 2: Simplify validateToolScopes Method
File: `src/McpBridgeService.php`

Current signature (line 407):
```php
protected function validateToolScopes(
  McpToolConfig $config,
  string $mcpName,
  bool $throwOnFailure,
): void
```

New signature:
```php
protected function validateToolScopes(
  McpToolConfig $config,
  string $mcpName,
): void
```

In the method body (lines 428-457):
- Remove the `if ($throwOnFailure)` conditional
- Always throw InsufficientScopeException when validation fails
- Remove the else block with logging (lines 446-456)

### Step 3: Remove Optional Authentication Branch
File: `src/McpBridgeService.php`

In executeMcpTool() method, remove lines 254-265:
```php
// Handle optional authentication mode.
if ($authentication_mode === 'optional') {
  // If user is anonymous, execute without scope validation.
  if ($is_anonymous) {
    return $this->executeToolById($tool_config->getToolId(), $parameters);
  }

  // For authenticated users, validate scopes (logs warning on failure).
  $this->validateToolScopes($tool_config, $mcpName, FALSE);

  return $this->executeToolById($tool_config->getToolId(), $parameters);
}
```

Update the "required" mode call to validateToolScopes() on line 249:
```php
// Old: $this->validateToolScopes($tool_config, $mcpName, TRUE);
// New: $this->validateToolScopes($tool_config, $mcpName);
```

### Step 4: Verify Entity Logic
File: `src/Entity/McpToolConfig.php`

Check the `requiresAuthentication()` method. It should return true only when authentication_mode is "required":
```php
public function requiresAuthentication(): bool {
  return $this->getAuthenticationMode() === 'required';
}
```

This logic is already correct - disabled mode returns false.

### Step 5: Run Tests
Execute the test suite to identify any test failures:
```bash
cd /var/www/html && vendor/bin/phpunit web/modules/contrib/mcp_server/tests/
```

**Expected test failures:**
- Tests checking "optional" mode behavior
- Tests validating AllowedValues constraint accepts "optional"

These will be fixed in task 02.

### Step 6: Clear Cache
```bash
cd /var/www/html && vendor/bin/drush cache:rebuild
```

</details>
