---
id: 2
group: "authentication-simplification"
dependencies: [1]
status: "completed"
created: 2025-11-13
skills:
  - drupal-testing
  - phpunit
---
# Update Authentication Tests for Binary Mode

## Objective
Update test suite to reflect the simplified binary authentication system (required/disabled only), removing all test cases for the deprecated "optional" mode.

## Skills Required
- **Drupal Testing**: Understanding of Drupal test bases (Unit, Kernel, Functional)
- **PHPUnit**: Writing and maintaining PHP unit tests with mocks and assertions

## Acceptance Criteria
- [ ] Remove all test cases covering "optional" authentication mode
- [ ] Add tests confirming only "required" and "disabled" modes are valid
- [ ] Update validateToolScopes() test calls to remove $throwOnFailure parameter
- [ ] All tests pass with 100% success rate
- [ ] Test coverage maintains ≥91% ratio

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

**Files to Modify:**
1. `tests/src/Unit/McpBridgeServiceTest.php` - Remove optional mode test cases
2. `tests/src/Kernel/OAuthScopeIntegrationTest.php` - Update integration tests
3. Any other test files referencing "optional" authentication mode

**Test Framework:**
- PHPUnit with Drupal test bases
- Mock objects for services (EntityTypeManager, OAuthScopeValidator, etc.)
- Drupal's KernelTestBase for integration tests

## Input Dependencies
- Task 01 completed: Optional authentication mode removed from production code
- Updated McpBridgeService with simplified validateToolScopes() signature

## Output Artifacts
- Green test suite with no failures
- Test cases validating binary authentication behavior
- Updated integration tests for required/disabled modes

## Implementation Notes

<details>
<summary>Detailed Implementation Steps</summary>

### Step 1: Identify Tests to Update

Search for test files referencing "optional":
```bash
cd /var/www/html/web/modules/contrib/mcp_server
grep -r "optional" tests/ --include="*.php"
```

Expected files:
- `tests/src/Unit/McpBridgeServiceTest.php`
- `tests/src/Kernel/OAuthScopeIntegrationTest.php`

### Step 2: Update McpBridgeServiceTest.php

**Remove Test Methods:**
Look for test methods like:
- `testExecuteMcpToolWithOptionalAuthenticationAnonymous()`
- `testExecuteMcpToolWithOptionalAuthenticationAuthenticated()`
- `testExecuteMcpToolWithOptionalAuthenticationInsufficientScopes()`

Delete these entire methods.

**Update Remaining Tests:**
Find calls to `validateToolScopes()` and remove the third parameter:
```php
// Old:
$this->scopeValidator->validateScopes($required_scopes, $token_scopes, TRUE);

// New:
$this->scopeValidator->validateScopes($required_scopes, $token_scopes);
```

**Add Binary Validation Test:**
Add a new test confirming only two modes are valid:
```php
/**
 * Tests that only required and disabled authentication modes are valid.
 */
public function testAuthenticationModesRestricted(): void {
  $tool_config = $this->createMock(McpToolConfig::class);

  // Test valid modes
  $tool_config->method('getAuthenticationMode')->willReturn('required');
  $this->assertEquals('required', $tool_config->getAuthenticationMode());

  $tool_config->method('getAuthenticationMode')->willReturn('disabled');
  $this->assertEquals('disabled', $tool_config->getAuthenticationMode());

  // Note: Config schema validation prevents "optional" from being set
}
```

### Step 3: Update OAuthScopeIntegrationTest.php

**Remove Optional Mode Integration Tests:**
Look for test methods testing optional authentication behavior.
Delete test methods that:
- Set up config entities with `authentication_mode: optional`
- Test warning logs for insufficient scopes with optional mode

**Update Required Mode Tests:**
Ensure tests for "required" mode properly expect exceptions:
```php
// Verify InsufficientScopeException is thrown (no warning-only behavior)
$this->expectException(InsufficientScopeException::class);
$bridge->executeMcpTool('test_tool', []);
```

### Step 4: Update Config Schema Tests

If there are tests validating the config schema:
```php
/**
 * Tests that config schema only allows required and disabled modes.
 */
public function testAuthenticationModeConstraint(): void {
  $config = $this->config('mcp_server.mcp_tool_config.test');

  // Valid values should work
  $config->set('authentication_mode', 'required')->save();
  $config->set('authentication_mode', 'disabled')->save();

  // Invalid value should fail validation
  $this->expectException(\InvalidArgumentException::class);
  $config->set('authentication_mode', 'optional')->save();
}
```

### Step 5: Run Test Suite

Execute full test suite:
```bash
cd /var/www/html
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/
```

Verify:
- All tests pass
- No skipped tests
- No failures or errors

### Step 6: Check Test Coverage

If coverage reporting is available:
```bash
cd /var/www/html
vendor/bin/phpunit --coverage-text web/modules/contrib/mcp_server/tests/
```

Ensure coverage is ≥91% (maintain or improve current ratio).

### Step 7: Run Static Analysis

Verify no new issues introduced:
```bash
cd /var/www/html
vendor/bin/phpstan analyse web/modules/contrib/mcp_server/
```

</details>
