---
id: 3
group: "configuration-implementation"
dependencies: [1]
status: "completed"
created: 2025-11-13
completed: 2025-11-13
skills:
  - drupal-backend
  - drupal-configuration
---
# Implement Configuration Settings

## Objective
Make the four defined configuration values (server_name, server_version, pagination_limit, session_ttl) functional by reading them from mcp_server.settings config and applying them in McpServerFactory.

## Skills Required
- **Drupal Backend Development**: Service definitions, dependency injection, factory patterns
- **Drupal Configuration System**: Config objects, schema, defaults, validation

## Acceptance Criteria
- [ ] McpServerFactory reads all four config values from mcp_server.settings
- [ ] Server name and version are passed to setServerInfo()
- [ ] Pagination limit is passed to setPaginationLimit() with validation
- [ ] Session TTL investigation completed (document if not implementable)
- [ ] Config validation ensures positive integers and non-empty strings
- [ ] Service definition updated with config.factory dependency

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

**Files to Modify:**
1. `mcp_server.services.yml` - Add @config.factory argument to server_factory
2. `src/McpServerFactory.php` - Inject ConfigFactoryInterface, read config, apply values
3. `config/install/mcp_server.settings.yml` - Verify defaults are sensible

**Drupal APIs:**
- `ConfigFactoryInterface` for reading config
- Service container for dependency injection
- Server builder API from MCP SDK

**Validation Requirements:**
- `pagination_limit`: Integer, 1-1000
- `session_ttl`: Integer, > 0 (if implementable)
- `server_name`, `server_version`: Non-empty strings

## Input Dependencies
- Task 01 completed: Authentication simplification provides cleaner codebase

## Output Artifacts
- Functional configuration system
- McpServerFactory reading and applying all config values
- Validated configuration with sensible ranges
- Documentation of session_ttl status (implemented or deferred)

## Implementation Notes

<details>
<summary>Detailed Implementation Steps</summary>

### Step 1: Update Service Definition

File: `mcp_server.services.yml`

Locate the `mcp_server.server_factory` service definition:
```yaml
mcp_server.server_factory:
  class: Drupal\mcp_server\McpServerFactory
  arguments:
    - '@mcp_server.bridge'
    - '@logger.channel.mcp_server'
    - '@event_dispatcher'
    - '@mcp_server.session_store'
```

Add `@config.factory` as first argument:
```yaml
mcp_server.server_factory:
  class: Drupal\mcp_server\McpServerFactory
  arguments:
    - '@config.factory'
    - '@mcp_server.bridge'
    - '@logger.channel.mcp_server'
    - '@event_dispatcher'
    - '@mcp_server.session_store'
```

### Step 2: Update Factory Constructor

File: `src/McpServerFactory.php`

Current constructor (lines 34-39):
```php
public function __construct(
  private readonly McpBridgeService $mcpBridge,
  private readonly LoggerInterface $logger,
  private readonly EventDispatcherInterface $eventDispatcher,
  private readonly FileSessionStore $sessionStore,
) {}
```

Updated constructor:
```php
public function __construct(
  private readonly ConfigFactoryInterface $configFactory,
  private readonly McpBridgeService $mcpBridge,
  private readonly LoggerInterface $logger,
  private readonly EventDispatcherInterface $eventDispatcher,
  private readonly FileSessionStore $sessionStore,
) {}
```

Add use statement at top:
```php
use Drupal\Core\Config\ConfigFactoryInterface;
```

### Step 3: Read Configuration in create() Method

File: `src/McpServerFactory.php`

Current create() method (lines 47-71):
```php
public function create(): Server {
  $builder = Server::builder()
    ->setServerInfo('Drupal MCP Server', '1.0.0')
    ->setPaginationLimit(50)
    ->setLogger($this->logger)
    ->setEventDispatcher($this->eventDispatcher)
    ->setSession($this->sessionStore);
  // ... rest of method
}
```

Updated create() method:
```php
public function create(): Server {
  // Read configuration.
  $config = $this->configFactory->get('mcp_server.settings');
  $server_name = $config->get('server_name') ?: 'Drupal MCP Server';
  $server_version = $config->get('server_version') ?: '1.0.0';
  $pagination_limit = $config->get('pagination_limit') ?: 50;

  // Validate pagination limit.
  $pagination_limit = (int) $pagination_limit;
  if ($pagination_limit < 1 || $pagination_limit > 1000) {
    $this->logger->warning(
      'Invalid pagination_limit (@limit). Using default: 50',
      ['@limit' => $pagination_limit]
    );
    $pagination_limit = 50;
  }

  $builder = Server::builder()
    ->setServerInfo($server_name, $server_version)
    ->setPaginationLimit($pagination_limit)
    ->setLogger($this->logger)
    ->setEventDispatcher($this->eventDispatcher)
    ->setSession($this->sessionStore);
  // ... rest of method
}
```

### Step 4: Investigate session_ttl Implementation

The `session_ttl` config value is defined but needs investigation:

**Check FileSessionStore API:**
```php
// Review FileSessionStore constructor signature
// Location: vendor/mcp/sdk/src/Server/Session/FileSessionStore.php
```

**Possible outcomes:**

**A) If FileSessionStore accepts TTL:**
```php
// In mcp_server.services.yml, update session store service:
mcp_server.session_store:
  class: Mcp\Server\Session\FileSessionStore
  factory: ['Drupal\mcp_server\McpServerFactory', 'createSessionStore']
  arguments:
    - '%kernel.cache_dir%/mcp_sessions'

// In McpServerFactory, add static factory method:
public static function createSessionStore(string $cache_dir): FileSessionStore {
  $config = \Drupal::config('mcp_server.settings');
  $ttl = (int) ($config->get('session_ttl') ?: 3600);
  return new FileSessionStore($cache_dir, $ttl);
}
```

**B) If FileSessionStore does NOT accept TTL:**
Add a comment in create() method:
```php
// Note: session_ttl configuration is defined but not yet implemented.
// FileSessionStore from mcp/sdk does not currently support configurable TTL.
// This will be implemented when SDK adds TTL support.
```

And add TODO to README.md documentation.

### Step 5: Verify Default Configuration

File: `config/install/mcp_server.settings.yml`

Verify defaults are sensible:
```yaml
server_name: 'Drupal MCP Server'
server_version: '1.0.0'
pagination_limit: 50
session_ttl: 3600
```

These values look good. No changes needed.

### Step 6: Test Configuration Reading

Create a simple test:
```bash
cd /var/www/html
vendor/bin/drush php-eval "
  \$config = \Drupal::config('mcp_server.settings');
  print 'Server Name: ' . \$config->get('server_name') . PHP_EOL;
  print 'Version: ' . \$config->get('server_version') . PHP_EOL;
  print 'Pagination: ' . \$config->get('pagination_limit') . PHP_EOL;
  print 'Session TTL: ' . \$config->get('session_ttl') . PHP_EOL;
"
```

Expected output:
```
Server Name: Drupal MCP Server
Version: 1.0.0
Pagination: 50
Session TTL: 3600
```

### Step 7: Test Configuration Override

Modify config and verify it's respected:
```bash
cd /var/www/html

# Change server name
vendor/bin/drush config:set mcp_server.settings server_name "My Custom Server" -y

# Rebuild cache
vendor/bin/drush cache:rebuild

# Test MCP server (if STDIO transport available)
# The MCP listTools response should include "My Custom Server" in metadata
```

### Step 8: Clear Cache

```bash
cd /var/www/html && vendor/bin/drush cache:rebuild
```

</details>
