---
id: 4
group: "configuration-implementation"
dependencies: [3]
status: "completed"
created: 2025-11-13
skills:
  - drupal-testing
  - phpunit
---
# Add Configuration Tests

## Objective
Create comprehensive tests for McpServerFactory configuration reading, validation, and application to ensure config values are properly used.

## Skills Required
- **Drupal Testing**: Unit tests with mocked config, Kernel tests with real config
- **PHPUnit**: Test structure, mocking, assertions, data providers

## Acceptance Criteria
- [ ] Unit tests verify McpServerFactory reads config correctly
- [ ] Tests validate pagination_limit bounds (1-1000)
- [ ] Tests verify default values are used when config is empty
- [ ] Tests confirm server info and pagination are applied to builder
- [ ] Integration tests with actual config objects
- [ ] All tests pass with 100% success rate

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

**New Test Files:**
1. `tests/src/Unit/McpServerFactoryTest.php` - Unit tests with mocked config
2. `tests/src/Kernel/McpServerConfigTest.php` - Integration tests with real config

**Test Scenarios:**
- Default configuration values
- Custom configuration values
- Invalid pagination_limit (< 1, > 1000)
- Empty/null configuration values
- Configuration changes reflected in server

## Input Dependencies
- Task 03 completed: McpServerFactory implements configuration reading

## Output Artifacts
- New unit test file for McpServerFactory
- New kernel test file for config integration
- Comprehensive test coverage for configuration system
- Validated configuration behavior

## Implementation Notes

<details>
<summary>Detailed Implementation Steps</summary>

### Step 1: Create Unit Test File

File: `tests/src/Unit/McpServerFactoryTest.php`

```php
<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\mcp_server\McpBridgeService;
use Drupal\mcp_server\McpServerFactory;
use Drupal\Tests\UnitTestCase;
use Mcp\Server\Session\FileSessionStore;
use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Log\LoggerInterface;

/**
 * Tests for McpServerFactory configuration reading.
 *
 * @coversDefaultClass \Drupal\mcp_server\McpServerFactory
 * @group mcp_server
 */
class McpServerFactoryTest extends UnitTestCase {

  /**
   * The config factory mock.
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * The config object mock.
   */
  private ImmutableConfig $config;

  /**
   * Other service mocks.
   */
  private McpBridgeService $mcpBridge;
  private LoggerInterface $logger;
  private EventDispatcherInterface $eventDispatcher;
  private FileSessionStore $sessionStore;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create config mocks.
    $this->config = $this->createMock(ImmutableConfig::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->method('get')
      ->with('mcp_server.settings')
      ->willReturn($this->config);

    // Create service mocks.
    $this->mcpBridge = $this->createMock(McpBridgeService::class);
    $this->mcpBridge->method('getEnabledTools')->willReturn([]);

    $this->logger = $this->createMock(LoggerInterface::class);
    $this->eventDispatcher = $this->createMock(EventDispatcherInterface::class);
    $this->sessionStore = $this->createMock(FileSessionStore::class);
  }

  /**
   * Tests factory reads default configuration values.
   *
   * @covers ::create
   */
  public function testDefaultConfiguration(): void {
    // Configure mock to return default values.
    $this->config->method('get')->willReturnMap([
      ['server_name', 'Drupal MCP Server'],
      ['server_version', '1.0.0'],
      ['pagination_limit', 50],
      ['session_ttl', 3600],
    ]);

    $factory = new McpServerFactory(
      $this->configFactory,
      $this->mcpBridge,
      $this->logger,
      $this->eventDispatcher,
      $this->sessionStore
    );

    $server = $factory->create();
    $this->assertInstanceOf(\Mcp\Server::class, $server);
  }

  /**
   * Tests factory reads custom configuration values.
   *
   * @covers ::create
   */
  public function testCustomConfiguration(): void {
    // Configure mock to return custom values.
    $this->config->method('get')->willReturnMap([
      ['server_name', 'Custom Server Name'],
      ['server_version', '2.5.0'],
      ['pagination_limit', 100],
      ['session_ttl', 7200],
    ]);

    $factory = new McpServerFactory(
      $this->configFactory,
      $this->mcpBridge,
      $this->logger,
      $this->eventDispatcher,
      $this->sessionStore
    );

    $server = $factory->create();
    $this->assertInstanceOf(\Mcp\Server::class, $server);
  }

  /**
   * Tests factory handles invalid pagination limits.
   *
   * @dataProvider invalidPaginationProvider
   * @covers ::create
   */
  public function testInvalidPaginationLimit(int $invalid_limit): void {
    $this->config->method('get')->willReturnMap([
      ['server_name', 'Test Server'],
      ['server_version', '1.0.0'],
      ['pagination_limit', $invalid_limit],
      ['session_ttl', 3600],
    ]);

    // Logger should be called with warning.
    $this->logger->expects($this->once())
      ->method('warning')
      ->with(
        $this->stringContains('Invalid pagination_limit'),
        $this->arrayHasKey('@limit')
      );

    $factory = new McpServerFactory(
      $this->configFactory,
      $this->mcpBridge,
      $this->logger,
      $this->eventDispatcher,
      $this->sessionStore
    );

    $server = $factory->create();
    $this->assertInstanceOf(\Mcp\Server::class, $server);
  }

  /**
   * Data provider for invalid pagination limits.
   */
  public function invalidPaginationProvider(): array {
    return [
      'zero' => [0],
      'negative' => [-10],
      'too_large' => [1500],
    ];
  }

  /**
   * Tests factory uses defaults when config values are empty.
   *
   * @covers ::create
   */
  public function testEmptyConfigurationUsesDefaults(): void {
    // Configure mock to return null/empty values.
    $this->config->method('get')->willReturn(NULL);

    $factory = new McpServerFactory(
      $this->configFactory,
      $this->mcpBridge,
      $this->logger,
      $this->eventDispatcher,
      $this->sessionStore
    );

    $server = $factory->create();
    $this->assertInstanceOf(\Mcp\Server::class, $server);
  }

}
```

### Step 2: Create Kernel Integration Test

File: `tests/src/Kernel/McpServerConfigTest.php`

```php
<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Tests MCP server configuration integration.
 *
 * @group mcp_server
 */
class McpServerConfigTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['mcp_server', 'tool'];

  /**
   * The MCP server factory.
   */
  private $serverFactory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['mcp_server']);
    $this->serverFactory = $this->container->get('mcp_server.server_factory');
  }

  /**
   * Tests that factory service is available.
   */
  public function testFactoryServiceAvailable(): void {
    $this->assertNotNull($this->serverFactory);
  }

  /**
   * Tests that factory reads installed configuration.
   */
  public function testFactoryReadsInstalledConfig(): void {
    $config = $this->config('mcp_server.settings');

    // Verify default config is installed.
    $this->assertEquals('Drupal MCP Server', $config->get('server_name'));
    $this->assertEquals('1.0.0', $config->get('server_version'));
    $this->assertEquals(50, $config->get('pagination_limit'));
    $this->assertEquals(3600, $config->get('session_ttl'));
  }

  /**
   * Tests that factory respects configuration changes.
   */
  public function testFactoryRespectsConfigChanges(): void {
    // Change configuration.
    $this->config('mcp_server.settings')
      ->set('server_name', 'Test Server')
      ->set('server_version', '3.0.0')
      ->set('pagination_limit', 75)
      ->save();

    // Create new factory instance (simulating container rebuild).
    $factory = $this->container->get('mcp_server.server_factory');
    $server = $factory->create();

    // Verify server was created successfully.
    $this->assertInstanceOf(\Mcp\Server::class, $server);
  }

  /**
   * Tests pagination limit validation.
   */
  public function testPaginationLimitValidation(): void {
    // Set invalid pagination limit.
    $this->config('mcp_server.settings')
      ->set('pagination_limit', 2000)
      ->save();

    $factory = $this->container->get('mcp_server.server_factory');

    // Should not throw exception, but use default value (50).
    $server = $factory->create();
    $this->assertInstanceOf(\Mcp\Server::class, $server);
  }

}
```

### Step 3: Run New Tests

Execute the new test files:
```bash
cd /var/www/html

# Run unit tests
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/src/Unit/McpServerFactoryTest.php

# Run kernel tests
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/src/Kernel/McpServerConfigTest.php
```

Verify all tests pass.

### Step 4: Run Full Test Suite

Ensure no regressions:
```bash
cd /var/www/html
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/
```

### Step 5: Check Test Coverage

Verify coverage for McpServerFactory:
```bash
cd /var/www/html
vendor/bin/phpunit --coverage-text --filter=McpServerFactory web/modules/contrib/mcp_server/tests/
```

Target: > 90% coverage for McpServerFactory class.

</details>
