---
id: 1
group: "prompt-registration"
dependencies: []
status: "completed"
created: 2025-11-17
skills:
  - php
  - drupal
---

# Register Prompts with MCP SDK in McpServerFactory

## Objective

Modify `McpServerFactory::create()` to register enabled prompt configurations with the MCP SDK builder, making them discoverable through the MCP protocol's `prompts/list` and `prompts/get` endpoints.

## Skills Required

- **PHP 8.3+**: Working with closures and service dependencies
- **Drupal 11**: Understanding dependency injection and factory patterns

## Acceptance Criteria

- [ ] Prompt registration code added to `McpServerFactory::create()` after tool registration
- [ ] Each enabled prompt is registered using `$builder->addPrompt()`
- [ ] Handler closure returns prompt definition from `getMcpPrompt()`
- [ ] Code follows existing patterns used for tool registration
- [ ] No regressions in existing tool functionality

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

**File to modify**: `src/McpServerFactory.php`

**Location**: After tool registration loop (around line 88), before `return $builder->build()` (line 90)

**Code to add**:

```php
// Register prompts with handlers for metadata discovery.
$enabled_prompts = $this->mcpBridge->getEnabledPrompts();
foreach ($enabled_prompts as $prompt_data) {
  $builder->addPrompt(
    handler: fn(array $args) => $this->mcpBridge->getMcpPrompt($prompt_data['name']),
    name: $prompt_data['name'],
    description: $prompt_data['description'] ?? NULL,
  );
}
```

**Key points**:
- `getEnabledPrompts()` returns array of prompts with keys: name, title, description, arguments, messages
- Handler closure must return full prompt definition when called by SDK
- Description is optional (some prompts may only have title)
- This follows the same pattern as tool registration above it

## Input Dependencies

**Existing code**:
- `McpBridgeService::getEnabledPrompts()` (lines 449-497 in src/McpBridgeService.php)
- `McpBridgeService::getMcpPrompt()` (lines 509-562 in src/McpBridgeService.php)
- `Builder::addPrompt()` from MCP SDK (vendor/mcp/sdk/src/Server/Builder.php:410-419)

**Dependencies**: None - this is the only implementation task

## Output Artifacts

**Modified file**: `src/McpServerFactory.php` with prompt registration added

**Result**: MCP server instance that includes registered prompts in capability registry

## Implementation Notes

<details>
<summary>Detailed Implementation Steps</summary>

1. **Open the file**: `src/McpServerFactory.php`

2. **Locate the insertion point**: Find the tool registration loop ending around line 88:
   ```php
   foreach ($enabled_tools as $tool_data) {
     $builder->addTool(
       handler: fn() => NULL,
       name: $tool_data['id'],
       description: $tool_data['description'],
       inputSchema: $tool_data['inputSchema'],
     );
   }

   return $builder->build(); // Line 90 - insert BEFORE this
   ```

3. **Add prompt registration code** between the tool loop and the return statement:
   ```php
   // Register prompts with handlers for metadata discovery.
   $enabled_prompts = $this->mcpBridge->getEnabledPrompts();
   foreach ($enabled_prompts as $prompt_data) {
     $builder->addPrompt(
       handler: fn(array $args) => $this->mcpBridge->getMcpPrompt($prompt_data['name']),
       name: $prompt_data['name'],
       description: $prompt_data['description'] ?? NULL,
     );
   }
   ```

4. **Understand the handler closure**:
   - Unlike tools (which use dummy handlers), prompt handlers must return data
   - The SDK's `GetPromptHandler` calls this closure and formats the response
   - `getMcpPrompt()` already returns the correct format (tested in existing code)
   - Arguments `$args` will be passed by the SDK but our handler doesn't need them for static prompts

5. **Verify the data format**:
   - `getEnabledPrompts()` returns: `['name' => 'prompt_name', 'title' => '...', 'description' => '...', 'arguments' => [...], 'messages' => [...]]`
   - We only need `name` and `description` for registration
   - The handler closure will provide full details when prompts are accessed

6. **Note on caching**:
   - Both `getEnabledPrompts()` and `getMcpPrompt()` use Drupal cache
   - Cache is invalidated via tags when prompts are saved/deleted (already implemented in `McpPromptConfig`)
   - No additional cache handling needed in factory

7. **Error handling**:
   - `getEnabledPrompts()` catches exceptions and returns empty array (line 490-496)
   - Permission checking happens in bridge service (line 450-452)
   - No additional error handling needed in factory

</details>

**Testing approach**:
- After implementation, clear Drupal cache: `vendor/bin/drush cache:rebuild`
- Run existing tests to ensure no regressions: `vendor/bin/phpunit web/modules/contrib/mcp_server/tests/`
- Manual verification with MCP Inspector should show prompts listed
