---
id: 1
group: "prompt-handler-factory"
dependencies: []
status: "completed"
created: 2025-01-17
skills:
  - php
  - drupal-services
---
# Create PromptHandlerFactory Service

## Objective
Implement a `PromptHandlerFactory` service that dynamically generates closure handlers with parameter signatures matching MCP prompt argument configurations.

## Skills Required
- PHP 8.3+ (closures, reflection, type hints)
- Drupal service container and dependency injection

## Acceptance Criteria
- [ ] `PromptHandlerFactory` service class created at `src/Service/PromptHandlerFactory.php`
- [ ] Service accepts prompt name, arguments array, and McpBridgeService dependency
- [ ] Generates closures with parameters matching configured arguments (name, type, required/optional)
- [ ] Generated closures return prompt messages array from `McpBridgeService::getMcpPrompt()`
- [ ] Service registered in `mcp_server.services.yml`
- [ ] Code follows project style guidelines (final class, readonly properties, strict types)

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

<details>
<summary>Detailed Implementation Specifications</summary>

### Service Interface
Create `src/Service/PromptHandlerFactory.php`:
```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Service;

use Drupal\mcp_server\McpBridgeService;

final class PromptHandlerFactory {

  public function __construct(
    private readonly McpBridgeService $mcpBridge,
  ) {}

  /**
   * Creates a prompt handler closure with typed parameters.
   *
   * @param string $promptName
   *   The prompt name.
   * @param array<int, array{name: string, description: string, required: bool}> $arguments
   *   The prompt arguments from configuration.
   *
   * @return \Closure
   *   A closure with parameters matching the arguments configuration.
   */
  public function createHandler(string $promptName, array $arguments): \Closure {
    // Implementation here
  }

}
```

### Dynamic Closure Generation Strategy
Use PHP's variable functions to create closures with dynamic parameters:

```php
// For arguments: [{name: 'level', required: true}, {name: 'surface', required: false}]
// Generate:
$handler = function(string $level, ?string $surface = null) use ($promptName, $mcpBridge) {
    $providedArgs = array_filter(
        compact('level', 'surface'),
        fn($v) => $v !== null
    );
    return $mcpBridge->getMcpPrompt($promptName)['messages'] ?? [];
};
```

Use `eval()` with proper escaping or create a helper method that builds the parameter list programmatically.

### Type Mapping
Map argument configurations to PHP types:
- Required argument → `string $paramName`
- Optional argument → `?string $paramName = null`

(All parameters use `string` type for simplicity; MCP protocol handles type conversion)

### Service Registration
Add to `mcp_server.services.yml`:
```yaml
  mcp_server.prompt_handler_factory:
    class: Drupal\mcp_server\Service\PromptHandlerFactory
    arguments: ['@mcp_server.bridge']
```

</details>

## Input Dependencies
- Existing `McpBridgeService` for retrieving prompt messages
- Prompt configuration structure from `McpPromptConfig::getArguments()`

## Output Artifacts
- `src/Service/PromptHandlerFactory.php`
- Service registration in `mcp_server.services.yml`

## Implementation Notes
- **Security**: If using `eval()`, ensure all variable names are validated to contain only alphanumeric characters and underscores
- **Simplification**: Start with all parameters as `string` type - MCP protocol will handle conversions
- **Error Handling**: If argument configuration is invalid, log warning and return a simple fallback closure
- **Performance**: Closures are generated once during server creation, not on every request
