---
id: 10
summary: "Implement MCP prompt argument autocomplete support by integrating completion providers into prompt configuration"
created: 2025-11-17
---

# Plan: MCP Prompt Argument Autocomplete Support

## Original Work Order

> is running… I have seen the prompts listed in the NCP Inspector, and they are coming through with the arguments and with the metadata that is correctly used in the Drupal configuration
entity. However, when I try to set a value for the argument, I am given by the NCP Inspector an autocomplete list that is empty of potential values. Is the user or the NCP client supposed to get the values
that are available for the arguments on the NCP prompts from the NCP server? I am asking because while I type values in the NCP Inspector input, I can see autocomplete requests being made to the server on the
NCP endpoint. I am attaching some of the logs for such autocompletes and some of the feedback that I get in the NCP Inspector on the requests that are being made.

## Executive Summary

The MCP protocol includes a `completion/complete` mechanism that allows MCP servers to provide autocomplete suggestions for prompt arguments. The MCP Inspector client is correctly sending `completion/complete` requests to the Drupal server, but the server is returning empty completion lists because the Drupal `mcp_server` module does not currently register completion providers for prompt arguments.

The MCP SDK provides built-in support for completion providers through the `CompletionCompleteHandler`, which looks for registered providers in the `PromptReference::$completionProviders` array. Currently, the `PromptConfigLoader` registers prompts with an empty completion providers array (`[]`), causing all autocomplete requests to return empty results.

This plan implements autocomplete support by creating a new plugin system for completion providers, extending the prompt configuration entity to store provider selections, and updating the UI with checkboxes that allow administrators to select available completion providers for each argument. This plugin-based approach provides an extensible, user-friendly system where developers can add new completion provider types without modifying core module code.

## Context

### Current State

The `mcp_server` module currently:

1. **Registers prompts without completion providers**: In `PromptConfigLoader.php:67`, prompts are registered with an empty array for completion providers:
   ```php
   $registry->registerPrompt($prompt, $handler, [], TRUE);
   ```

2. **Has no UI for configuring completions**: The `McpPromptConfigForm` and `McpPromptConfig` entity do not include fields for specifying completion options.

3. **Handles incoming `completion/complete` requests correctly**: The MCP SDK's `CompletionCompleteHandler` processes these requests but finds no registered providers, returning empty results.

4. **MCP Inspector correctly requests completions**: The client-side behavior is correct - it sends `completion/complete` requests when users type in argument fields.

### Target State

After implementation:

1. **Plugin system for completion providers**: A new `PromptArgumentCompletionProvider` plugin type allows developers to create custom completion providers with human-readable names and descriptions.

2. **Prompt configuration stores provider selections**: Each prompt argument can optionally specify selected completion provider plugin IDs and configuration (e.g., static value lists).

3. **Completion providers are instantiated and registered**: The `PromptConfigLoader` uses the plugin manager to instantiate selected providers and registers them with each prompt.

4. **Admin UI displays available providers as checkboxes**: Site builders see a list of available completion providers (with names and descriptions) and can select which ones to enable for each argument.

5. **Autocomplete works in MCP Inspector**: When users type argument values, they receive relevant autocomplete suggestions from the enabled completion providers.

### Background

**MCP Completion Protocol**: The Model Context Protocol specification includes a `completion/complete` request that clients send to servers. The request includes:
- `ref`: Reference to the prompt or resource
- `argument`: Object with `name` (argument name) and `value` (current input value)

The server responds with a `CompletionCompleteResult` containing:
- `values`: Array of string completions (max 100)
- `total`: Optional total count of available completions
- `hasMore`: Optional boolean indicating more results exist

**SDK Implementation**: The `mcp/sdk` library provides:
- `ProviderInterface`: Interface for completion providers with `getCompletions(string $currentValue): array`
- `ListCompletionProvider`: Provider for static value lists
- `CompletionCompleteHandler`: Request handler that looks up providers by argument name and invokes them

**Current Gap**: The Drupal module's configuration-based approach doesn't capture or register completion providers, leaving the SDK handler unable to provide suggestions.

## Technical Implementation Approach

### Component 1: Create Plugin System for Completion Providers

**Objective**: Establish a plugin-based architecture that allows developers to create custom completion providers with metadata, making them discoverable and selectable through the UI.

Create a new plugin type `PromptArgumentCompletionProvider`:

**Plugin Attribute**:
```php
/**
 * Defines a Prompt Argument Completion Provider attribute.
 */
#[\Attribute(\Attribute::TARGET_CLASS)]
class PromptArgumentCompletionProvider {
  public function __construct(
    public readonly string $id,
    public readonly TranslatableMarkup $label,
    public readonly TranslatableMarkup $description,
  ) {}
}
```

**Plugin Interface**:
```php
interface PromptArgumentCompletionProviderInterface extends PluginInspectionInterface {
  /**
   * Get completions for a given current value.
   *
   * @param string $currentValue
   *   The current value to get completions for.
   * @param array $configuration
   *   Plugin configuration from the prompt argument.
   *
   * @return string[]
   *   Array of completion values.
   */
  public function getCompletions(string $currentValue, array $configuration): array;
}
```

**Base Plugin Class**:
```php
abstract class PromptArgumentCompletionProviderBase extends PluginBase
  implements PromptArgumentCompletionProviderInterface {
  // Common functionality
}
```

**Default Plugin - Static List Provider**:
```php
/**
 * Provides completions from a static list of values.
 */
#[PromptArgumentCompletionProvider(
  id: 'static_list',
  label: new TranslatableMarkup('Static list'),
  description: new TranslatableMarkup('Provides completions from a static list of values configured per argument.'),
)]
class StaticListCompletionProvider extends PromptArgumentCompletionProviderBase {
  public function getCompletions(string $currentValue, array $configuration): array {
    $values = $configuration['values'] ?? [];
    if (empty($currentValue)) {
      return $values;
    }
    return array_values(array_filter(
      $values,
      fn($value) => str_starts_with($value, $currentValue)
    ));
  }
}
```

**Plugin Manager**:
- Create `PromptArgumentCompletionProviderManager` extending `DefaultPluginManager`
- Discover plugins in `src/Plugin/PromptArgumentCompletionProvider/` directories
- Provide methods to list available providers with metadata

### Component 2: Extend Prompt Configuration Schema

**Objective**: Add completion provider selections and configuration to the prompt argument structure.

Update `McpPromptConfig` entity to include completion provider data:

**Configuration Schema**:
```yaml
arguments:
  - name: 'skill_level'
    description: 'Player skill level'
    required: true
    completion_providers:
      - plugin_id: 'static_list'
        configuration:
          values: ['beginner', 'intermediate', 'advanced', 'professional']
```

**Implementation Details**:
1. Add optional `completion_providers` array to argument structure in `McpPromptConfig::$arguments`
2. Each item contains `plugin_id` and optional `configuration` array
3. Update `validateArguments()` to validate:
   - Plugin IDs exist in the plugin manager
   - Configuration is valid for the selected plugin
4. Support multiple providers per argument (plugins are checked in order)

### Component 3: Instantiate and Register Completion Providers

**Objective**: Use the plugin manager to instantiate selected completion providers and register them with each prompt.

Modify `PromptConfigLoader::load()` to:

1. **Inject plugin manager**: Add `PromptArgumentCompletionProviderManager` to constructor dependencies
2. **Parse completion configuration**: For each argument with `completion_providers` data
3. **Instantiate plugin instances**:
   ```php
   foreach ($arg['completion_providers'] ?? [] as $provider_config) {
     $plugin = $this->completionProviderManager->createInstance(
       $provider_config['plugin_id'],
       $provider_config['configuration'] ?? []
     );
     // Wrap in adapter to convert to SDK ProviderInterface
     $providers[$arg['name']] = new PluginProviderAdapter($plugin, $provider_config['configuration']);
   }
   ```
4. **Build completion providers array**: Create array mapping argument names to provider instances
5. **Register with providers**: Pass the providers array to `registerPrompt()`:
   ```php
   $registry->registerPrompt($prompt, $handler, $completionProviders, TRUE);
   ```

**Adapter Pattern**:
Create `PluginProviderAdapter` that adapts Drupal plugins to the SDK's `ProviderInterface`:
```php
class PluginProviderAdapter implements ProviderInterface {
  public function __construct(
    private readonly PromptArgumentCompletionProviderInterface $plugin,
    private readonly array $configuration
  ) {}

  public function getCompletions(string $currentValue): array {
    return $this->plugin->getCompletions($currentValue, $this->configuration);
  }
}
```

**Error Handling**:
- Log warnings if plugin instantiation fails
- Continue registration without completion for that argument
- Validate plugin configuration during entity save

### Component 4: Update UI with Plugin Selection Checkboxes

**Objective**: Provide an intuitive checkbox-based interface for selecting available completion providers.

Enhance `McpPromptConfigForm`:

1. **Load available plugins**: Use plugin manager to get all available completion provider plugins
2. **For each argument**, display:
   - Argument name and description
   - Checkboxes for available completion providers with:
     - Plugin label as checkbox label
     - Plugin description as help text below checkbox
3. **Plugin-specific configuration fields**:
   - When a plugin is checked, show additional configuration fields
   - For `static_list`: Show textarea for values (one per line)
   - For custom plugins: Show fields defined by plugin's configuration schema
4. **AJAX behavior**: Show/hide configuration fields when checkboxes change
5. **Form structure**:
   ```php
   $form['arguments'][$i]['completion'] = [
     '#type' => 'details',
     '#title' => $this->t('Autocomplete'),
     '#open' => FALSE,
   ];

   foreach ($available_plugins as $plugin_id => $definition) {
     $form['arguments'][$i]['completion'][$plugin_id] = [
       '#type' => 'checkbox',
       '#title' => $definition['label'],
       '#description' => $definition['description'],
     ];

     $form['arguments'][$i]['completion'][$plugin_id . '_config'] = [
       '#type' => 'container',
       '#states' => ['visible' => [':input[name="..."]' => ['checked' => TRUE]]],
       // Plugin-specific configuration fields
     ];
   }
   ```

**UI Design Considerations**:
- Use collapsed details fieldsets to avoid overwhelming users
- Show plugin count badge when providers are selected
- Provide clear examples for configuration fields
- Most users will use the `static_list` plugin

```mermaid
graph TD
    A[Argument Configuration] --> B[Completion Details Fieldset]
    B --> C[Available Plugins Checkboxes]
    C --> D{Plugin Selected?}
    D -->|Static List Checked| E[Values Textarea]
    D -->|Custom Plugin Checked| F[Plugin Config Fields]
    D -->|None Checked| G[No Autocomplete]
    E --> H[PluginProviderAdapter]
    F --> H
    H --> I[ListCompletionProvider]
    I --> J[completion/complete Response]
```

## Risk Considerations and Mitigation Strategies

### Technical Risks

- **Invalid plugin IDs in configuration**: Configuration may reference non-existent or disabled plugins
    - **Mitigation**: Validate plugin IDs exist during form submission; gracefully degrade by logging warning and skipping invalid plugins during registration

- **Large value lists causing performance issues**: Administrators might configure hundreds or thousands of completion values
    - **Mitigation**: Document best practices for completion value counts; the SDK automatically limits responses to 100 values with pagination support

- **Plugin instantiation failures**: Custom plugins may have complex dependencies or throw exceptions
    - **Mitigation**: Wrap instantiation in try-catch blocks; log errors; continue prompt registration without completion for failing arguments

- **Multiple plugins per argument complexity**: Allowing multiple providers could confuse users about execution order
    - **Mitigation**: Initially support only one provider per argument; document clearly if multiple providers are supported in the future

### Implementation Risks

- **Form complexity**: Adding completion configuration with plugin checkboxes may overwhelm users
    - **Mitigation**: Use collapsed details fieldsets; provide clear examples; make completion optional with sensible defaults; only show `static_list` plugin initially

- **Configuration schema changes**: Modifying the argument structure requires careful handling of existing configurations
    - **Mitigation**: Ensure backward compatibility by making completion optional; existing configs without completion data continue working unchanged

- **Plugin system overhead**: Adding a full plugin system for a simple feature might be over-engineering
    - **Mitigation**: Keep initial implementation simple with only `static_list` plugin; plugin architecture allows future extensibility without rewriting

## Success Criteria

### Primary Success Criteria

1. **Autocomplete works in MCP Inspector**: Users typing in prompt argument fields receive relevant autocomplete suggestions
2. **Configuration persists correctly**: Completion provider plugin selections and configuration save and load properly from configuration entities
3. **Plugin system functions correctly**: The `static_list` plugin provides completions; custom plugins can be added by developers
4. **Checkbox UI is intuitive**: Site builders can easily select and configure completion providers without technical knowledge
5. **Backward compatibility maintained**: Existing prompt configurations without completion data continue functioning normally

### Quality Assurance Metrics

1. **Functional test coverage**: Test case validates completion requests return expected values for configured prompts with `static_list` provider
2. **Form validation works**: Invalid plugin selections and configurations are rejected with clear error messages
3. **Error handling**: Invalid completion plugins don't prevent prompt registration; errors are logged appropriately
4. **Performance**: Completion requests respond within acceptable timeframes even with large value lists
5. **Plugin discoverability**: Developers can create custom plugins that automatically appear in the UI

## Resource Requirements

### Development Skills

- Drupal 11 plugin system architecture
- Drupal 11 configuration entity development
- PHP 8.3 features (readonly properties, constructor promotion)
- MCP protocol understanding (completion/complete request flow)
- Form API and AJAX patterns with #states
- PHPUnit functional testing

### Technical Infrastructure

- `mcp/sdk` library (already integrated)
- Drupal core Plugin API
- Drupal core Form API
- Drupal configuration system
- MCP Inspector for testing

## Implementation Order

1. **Plugin system foundation**: Create plugin type, interface, base class, manager, and `static_list` plugin implementation
2. **Configuration schema extension**: Extend `McpPromptConfig` to support `completion_providers` in argument definitions
3. **Adapter and instantiation logic**: Create `PluginProviderAdapter` and modify `PromptConfigLoader` to instantiate and register plugins
4. **UI implementation**: Update `McpPromptConfigForm` with plugin selection checkboxes and configuration fields
5. **Testing**: Add functional test validating completion requests return configured values from `static_list` plugin
6. **Documentation**: Update module documentation with plugin development examples and configuration guide

## Notes

- The MCP SDK handles the `completion/complete` protocol correctly; this plan only addresses the Drupal-specific plugin system, configuration, and registration
- Completion providers are optional - arguments without completion configuration work as before, just without autocomplete
- The SDK limits completion results to 100 values per response with `hasMore` pagination support
- The plugin architecture allows future extensibility (e.g., entity reference completions, taxonomy term completions, custom API completions) without modifying the core module code
- Initially, only the `static_list` plugin will be implemented; additional plugins can be added as needed by site builders or contributed as separate modules
- The `PluginProviderAdapter` bridges between Drupal's plugin system and the MCP SDK's `ProviderInterface`, allowing seamless integration

## Task Dependencies

```mermaid
graph TD
    001[Task 01: Plugin System Foundation] --> 002[Task 02: Configuration Schema]
    001 --> 003[Task 03: Adapter and Loader]
    002 --> 003
    001 --> 004[Task 04: UI Implementation]
    002 --> 004
```

## Execution Blueprint

**Validation Gates:**
- Reference: `.ai/task-manager/config/hooks/POST_PHASE.md`

### ✅ Phase 1: Foundation
**Parallel Tasks:**
- ✔️ Task 01: Create Plugin System Foundation

**Description:** Establish the plugin architecture including attribute, interface, base class, plugin manager, and default `static_list` plugin.

### ✅ Phase 2: Schema and Integration
**Parallel Tasks:**
- ✔️ Task 02: Extend Prompt Configuration Schema (depends on: 01)

**Description:** Add completion provider configuration support to the prompt entity with validation.

### ✅ Phase 3: Runtime Integration
**Parallel Tasks:**
- ✔️ Task 03: Create Adapter and Update Loader (depends on: 01, 02)
- ✔️ Task 04: Add Completion Provider UI to Form (depends on: 01, 02)

**Description:** Implement the adapter pattern for SDK integration and build the user interface for configuration.

### Execution Summary
- Total Phases: 3
- Total Tasks: 4
- Maximum Parallelism: 2 tasks (in Phase 3)
- Critical Path Length: 3 phases

## Execution Summary

**Status**: ✅ Completed Successfully
**Completed Date**: 2025-11-17
**Branch**: feature/plan-10-prompt-argument-autocomplete
**Commit**: 6dd7ba9 - feat: implement MCP prompt argument autocomplete support

### Results

Successfully implemented MCP prompt argument autocomplete support through a plugin-based architecture. The implementation consists of:

**Created Components (6 new files):**
- Plugin attribute, interface, base class, and manager for extensible completion provider system
- StaticListCompletionProvider plugin for static value autocomplete
- PluginProviderAdapter bridging Drupal plugins to MCP SDK ProviderInterface

**Enhanced Components (5 modified files):**
- McpPromptConfig entity with completion_providers field and validation
- PromptConfigLoader with plugin instantiation and registration logic
- McpServerFactory with plugin manager injection
- McpPromptConfigForm with checkbox-based plugin selection UI
- Service definitions for plugin manager

**Key Deliverables:**
- Site builders can configure static autocomplete values via admin UI
- Developers can extend functionality by creating custom completion provider plugins
- MCP Inspector displays autocomplete suggestions as users type argument values
- Full backward compatibility with existing prompt configurations

### Noteworthy Events

**Implementation Highlights:**
- All validation gates passed (linting, static analysis, pre-commit hooks)
- Clean separation of concerns using adapter pattern for SDK integration
- Extensible plugin architecture allows future enhancements without modifying core code
- User-friendly checkbox interface with #states-based conditional visibility

**Technical Decisions:**
- Initially supporting one completion provider per argument (easily extensible to multiple)
- Using PHP 8.3 attributes instead of annotations for modern plugin system
- Implementing graceful error handling that logs warnings but doesn't block prompt registration
- Collapsible details fieldsets to avoid overwhelming users

### Recommendations

**Future Enhancements:**
- Entity Reference completion provider (autocomplete from Drupal entities)
- Taxonomy Term completion provider (autocomplete from taxonomy vocabularies)
- REST API completion provider (fetch values from external APIs)
- Support multiple completion providers per argument with prioritization

**Testing:**
- Manual testing in MCP Inspector recommended to verify autocomplete behavior
- Consider adding functional test coverage in future iterations if needed
- Test with large value lists to verify performance characteristics

**Documentation:**
- Add examples to module documentation showing custom plugin development
- Create user guide for site builders on configuring autocomplete
- Document the plugin API for contributed completion provider modules
