---
id: 1
group: "plugin-system"
dependencies: []
status: "completed"
created: "2025-11-17"
skills:
  - "drupal-backend"
  - "php"
---
# Create Plugin System Foundation

## Objective
Establish the plugin architecture for prompt argument completion providers, including the attribute class, interface, base class, plugin manager, and the default `static_list` plugin implementation.

## Skills Required
- **drupal-backend**: Drupal 11 plugin system architecture, DefaultPluginManager
- **php**: PHP 8.3 attributes, readonly properties, constructor promotion

## Acceptance Criteria
- [ ] Plugin attribute `PromptArgumentCompletionProvider` created with `id`, `label`, and `description` properties
- [ ] Plugin interface `PromptArgumentCompletionProviderInterface` defines `getCompletions()` method
- [ ] Base class `PromptArgumentCompletionProviderBase` implements common functionality
- [ ] Plugin manager `PromptArgumentCompletionProviderManager` extends `DefaultPluginManager` and discovers plugins
- [ ] Default plugin `StaticListCompletionProvider` implements static list filtering
- [ ] Plugin system can be instantiated and lists available plugins

## Technical Requirements

**File Structure**:
- `src/Annotation/PromptArgumentCompletionProvider.php` - Attribute class
- `src/Plugin/PromptArgumentCompletionProviderInterface.php` - Interface
- `src/Plugin/PromptArgumentCompletionProviderBase.php` - Base class
- `src/Plugin/PromptArgumentCompletionProviderManager.php` - Plugin manager
- `src/Plugin/PromptArgumentCompletionProvider/StaticListCompletionProvider.php` - Default plugin

**Key APIs**:
- PHP 8 `#[\Attribute]` syntax
- Drupal `DefaultPluginManager`
- Drupal `PluginInspectionInterface`
- Drupal `TranslatableMarkup`

## Input Dependencies
None - this is the foundation task

## Output Artifacts
- Complete plugin system that can discover and instantiate completion provider plugins
- Working `static_list` plugin for testing
- Plugin manager service definition in `mcp_server.services.yml`

## Implementation Notes

<details>
<summary>Detailed Implementation Guide</summary>

### Step 1: Create Plugin Attribute

Create `src/Annotation/PromptArgumentCompletionProvider.php`:

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Annotation;

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Defines a Prompt Argument Completion Provider attribute.
 */
#[\Attribute(\Attribute::TARGET_CLASS)]
class PromptArgumentCompletionProvider {

  /**
   * Constructs a PromptArgumentCompletionProvider attribute.
   *
   * @param string $id
   *   The plugin ID.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $label
   *   The human-readable name.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $description
   *   The description.
   */
  public function __construct(
    public readonly string $id,
    public readonly TranslatableMarkup $label,
    public readonly TranslatableMarkup $description,
  ) {}

}
```

### Step 2: Create Plugin Interface

Create `src/Plugin/PromptArgumentCompletionProviderInterface.php`:

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;

/**
 * Interface for Prompt Argument Completion Provider plugins.
 */
interface PromptArgumentCompletionProviderInterface extends PluginInspectionInterface {

  /**
   * Get completions for a given current value.
   *
   * @param string $current_value
   *   The current value to get completions for.
   * @param array $configuration
   *   Plugin configuration from the prompt argument.
   *
   * @return string[]
   *   Array of completion values.
   */
  public function getCompletions(string $current_value, array $configuration): array;

}
```

### Step 3: Create Base Plugin Class

Create `src/Plugin/PromptArgumentCompletionProviderBase.php`:

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin;

use Drupal\Component\Plugin\PluginBase;

/**
 * Base class for Prompt Argument Completion Provider plugins.
 */
abstract class PromptArgumentCompletionProviderBase extends PluginBase implements PromptArgumentCompletionProviderInterface {

  /**
   * {@inheritdoc}
   */
  abstract public function getCompletions(string $current_value, array $configuration): array;

}
```

### Step 4: Create Plugin Manager

Create `src/Plugin/PromptArgumentCompletionProviderManager.php`:

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Manages Prompt Argument Completion Provider plugins.
 */
final class PromptArgumentCompletionProviderManager extends DefaultPluginManager {

  /**
   * Constructs a PromptArgumentCompletionProviderManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct(
      'Plugin/PromptArgumentCompletionProvider',
      $namespaces,
      $module_handler,
      'Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderInterface',
      'Drupal\mcp_server\Annotation\PromptArgumentCompletionProvider'
    );
    $this->setCacheBackend($cache_backend, 'mcp_server_prompt_argument_completion_provider_plugins');
  }

}
```

### Step 5: Create Static List Plugin

Create `src/Plugin/PromptArgumentCompletionProvider/StaticListCompletionProvider.php`:

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin\PromptArgumentCompletionProvider;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mcp_server\Attribute\PromptArgumentCompletionProvider;
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderBase;

/**
 * Provides completions from a static list of values.
 */
#[PromptArgumentCompletionProvider(
  id: 'static_list',
  label: new TranslatableMarkup('Static list'),
  description: new TranslatableMarkup('Provides completions from a static list of values configured per argument.'),
)]
final class StaticListCompletionProvider extends PromptArgumentCompletionProviderBase {

  /**
   * {@inheritdoc}
   */
  public function getCompletions(string $current_value, array $configuration): array {
    $values = $configuration['values'] ?? [];
    if (empty($current_value)) {
      return $values;
    }
    return array_values(array_filter(
      $values,
      fn($value) => str_starts_with($value, $current_value)
    ));
  }

}
```

### Step 6: Register Plugin Manager Service

Add to `mcp_server.services.yml`:

```yaml
  plugin.manager.prompt_argument_completion_provider:
    class: Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager
    parent: default_plugin_manager
```

### Step 7: Verification

After implementation:
1. Clear cache: `vendor/bin/drush cache:rebuild`
2. Verify plugin discovery works by instantiating the manager
3. Verify `static_list` plugin can be created and returns completions

</details>
