---
id: 2
group: "configuration-schema"
dependencies: [1]
status: "completed"
created: "2025-11-17"
skills:
  - "drupal-backend"
---
# Extend Prompt Configuration Schema

## Objective
Add `completion_providers` field to the prompt argument structure in `McpPromptConfig` entity, allowing administrators to store selected completion provider plugin IDs and their configurations.

## Skills Required
- **drupal-backend**: Drupal configuration entities, validation, schema definition

## Acceptance Criteria
- [ ] `McpPromptConfig::$arguments` supports optional `completion_providers` array per argument
- [ ] Each `completion_providers` item contains `plugin_id` and optional `configuration`
- [ ] `validateArguments()` validates plugin IDs exist in plugin manager
- [ ] `validateArguments()` validates plugin configuration structure
- [ ] Existing configurations without completion providers continue working
- [ ] Configuration exports/imports correctly with completion provider data

## Technical Requirements

**Modified Files**:
- `src/Entity/McpPromptConfig.php` - Add field and validation

**Configuration Schema Example**:
```yaml
arguments:
  - name: 'skill_level'
    description: 'Player skill level'
    required: true
    completion_providers:
      - plugin_id: 'static_list'
        configuration:
          values: ['beginner', 'intermediate', 'advanced', 'professional']
```

**Key APIs**:
- Drupal config entity validation
- Plugin manager `getDefinitions()` and `hasDefinition()`

## Input Dependencies
- Task 1: Plugin manager service to validate plugin IDs

## Output Artifacts
- Extended `McpPromptConfig` entity supporting completion provider configuration
- Validation logic that prevents invalid plugin references

## Implementation Notes

<details>
<summary>Detailed Implementation Guide</summary>

### Step 1: Update Entity Documentation

In `src/Entity/McpPromptConfig.php`, update the `$arguments` property documentation:

```php
/**
 * Array of argument definitions.
 *
 * Each argument has: name (string), description (string),
 * required (bool), completion_providers (array, optional).
 *
 * Completion providers structure:
 * - plugin_id (string): The completion provider plugin ID
 * - configuration (array, optional): Plugin-specific configuration
 *
 * @var array<int, array{
 *   name: string,
 *   description: string,
 *   required: bool,
 *   completion_providers?: array<int, array{
 *     plugin_id: string,
 *     configuration?: array
 *   }>
 * }>
 */
protected array $arguments = [];
```

### Step 2: Inject Plugin Manager

Add plugin manager to entity dependencies:

```php
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager;

// In the entity class, add method to get plugin manager
protected function getCompletionProviderManager(): PromptArgumentCompletionProviderManager {
  return \Drupal::service('plugin.manager.prompt_argument_completion_provider');
}
```

### Step 3: Extend validateArguments()

Enhance the existing `validateArguments()` method:

```php
protected function validateArguments(): void {
  // Existing validation for name, description, required...

  // Add completion provider validation
  $plugin_manager = $this->getCompletionProviderManager();
  $available_plugins = array_keys($plugin_manager->getDefinitions());

  foreach ($this->arguments as $index => $argument) {
    // Validate completion_providers structure if present
    if (isset($argument['completion_providers'])) {
      if (!is_array($argument['completion_providers'])) {
        throw new \InvalidArgumentException(
          sprintf('Argument "%s" completion_providers must be an array.', $argument['name'])
        );
      }

      foreach ($argument['completion_providers'] as $provider_index => $provider) {
        // Validate plugin_id exists
        if (empty($provider['plugin_id']) || !is_string($provider['plugin_id'])) {
          throw new \InvalidArgumentException(
            sprintf('Argument "%s" completion provider at index %d must have a valid plugin_id.',
              $argument['name'], $provider_index)
          );
        }

        // Validate plugin exists
        if (!in_array($provider['plugin_id'], $available_plugins)) {
          throw new \InvalidArgumentException(
            sprintf('Argument "%s" references unknown completion provider plugin "%s".',
              $argument['name'], $provider['plugin_id'])
          );
        }

        // Validate configuration is array if present
        if (isset($provider['configuration']) && !is_array($provider['configuration'])) {
          throw new \InvalidArgumentException(
            sprintf('Argument "%s" completion provider configuration must be an array.',
              $argument['name'])
          );
        }
      }
    }
  }
}
```

### Step 4: Add Helper Method

Add a getter method for completion providers:

```php
/**
 * Gets completion providers for a specific argument.
 *
 * @param string $argument_name
 *   The argument name.
 *
 * @return array
 *   Array of completion provider configurations.
 */
public function getArgumentCompletionProviders(string $argument_name): array {
  foreach ($this->arguments as $argument) {
    if ($argument['name'] === $argument_name) {
      return $argument['completion_providers'] ?? [];
    }
  }
  return [];
}
```

### Step 5: Ensure Backward Compatibility

Verify that:
1. Existing configs without `completion_providers` load correctly
2. The field is truly optional (not required)
3. Empty `completion_providers` arrays are handled gracefully

### Step 6: Test Configuration Export/Import

Create a test prompt configuration with completion providers and verify:
1. Configuration exports to YAML correctly
2. Configuration imports from YAML correctly
3. Validation runs on import and rejects invalid plugin IDs

</details>
