---
id: 3
group: "provider-integration"
dependencies: [1, 2]
status: "completed"
created: "2025-11-17"
skills:
  - "drupal-backend"
  - "php"
---
# Create Adapter and Update Loader

## Objective
Create `PluginProviderAdapter` to bridge Drupal plugins to MCP SDK's `ProviderInterface`, and update `PromptConfigLoader` to instantiate completion provider plugins and register them with prompts.

## Skills Required
- **drupal-backend**: Service injection, plugin instantiation, Drupal service patterns
- **php**: Adapter pattern, readonly properties, error handling

## Acceptance Criteria
- [ ] `PluginProviderAdapter` implements MCP SDK's `ProviderInterface`
- [ ] Adapter correctly delegates to Drupal plugin's `getCompletions()` method
- [ ] `PromptConfigLoader` injects plugin manager service
- [ ] Loader instantiates plugins for each argument's completion providers
- [ ] Completion providers array passed to `registerPrompt()` method
- [ ] Plugin instantiation failures are logged and handled gracefully
- [ ] Prompts without completion providers continue working unchanged

## Technical Requirements

**New Files**:
- `src/Capability/Loader/PluginProviderAdapter.php` - Adapter class

**Modified Files**:
- `src/Capability/Loader/PromptConfigLoader.php` - Add plugin instantiation
- `src/McpServerFactory.php` - Inject plugin manager into loader

**Key APIs**:
- MCP SDK `Mcp\Capability\Completion\ProviderInterface`
- Drupal plugin manager `createInstance()`
- PSR-3 logging

## Input Dependencies
- Task 1: Plugin manager and interface
- Task 2: Configuration schema with completion_providers

## Output Artifacts
- Working adapter that connects Drupal plugins to MCP SDK
- Updated loader that registers completion providers with prompts
- Autocomplete functionality working in MCP Inspector

## Implementation Notes

<details>
<summary>Detailed Implementation Guide</summary>

### Step 1: Create PluginProviderAdapter

Create `src/Capability/Loader/PluginProviderAdapter.php`:

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Capability\Loader;

use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderInterface;
use Mcp\Capability\Completion\ProviderInterface;

/**
 * Adapts Drupal completion provider plugins to MCP SDK ProviderInterface.
 */
final class PluginProviderAdapter implements ProviderInterface {

  /**
   * Constructs a PluginProviderAdapter.
   *
   * @param \Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderInterface $plugin
   *   The Drupal plugin instance.
   * @param array $configuration
   *   The plugin configuration.
   */
  public function __construct(
    private readonly PromptArgumentCompletionProviderInterface $plugin,
    private readonly array $configuration,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getCompletions(string $currentValue): array {
    return $this->plugin->getCompletions($currentValue, $this->configuration);
  }

}
```

### Step 2: Update PromptConfigLoader Constructor

In `src/Capability/Loader/PromptConfigLoader.php`, add plugin manager injection:

```php
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager;

/**
 * Constructs a PromptConfigLoader.
 *
 * @param \Drupal\mcp_server\McpBridgeService $mcpBridge
 *   The MCP bridge service.
 * @param \Psr\Log\LoggerInterface $logger
 *   The logger.
 * @param \Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager $completionProviderManager
 *   The completion provider plugin manager.
 */
public function __construct(
  private readonly McpBridgeService $mcpBridge,
  private readonly LoggerInterface $logger,
  private readonly PromptArgumentCompletionProviderManager $completionProviderManager,
) {}
```

### Step 3: Add Completion Provider Instantiation Logic

In the `load()` method, before registering the prompt, add:

```php
// Build completion providers array
$completion_providers = [];
foreach ($prompt_data['arguments'] ?? [] as $arg) {
  if (!empty($arg['completion_providers'])) {
    $providers_for_arg = [];

    foreach ($arg['completion_providers'] as $provider_config) {
      try {
        $plugin = $this->completionProviderManager->createInstance(
          $provider_config['plugin_id'],
          $provider_config['configuration'] ?? []
        );

        // Wrap in adapter
        $providers_for_arg[] = new PluginProviderAdapter(
          $plugin,
          $provider_config['configuration'] ?? []
        );

      } catch (\Throwable $e) {
        $this->logger->warning(
          'Failed to instantiate completion provider "@plugin_id" for argument "@arg": @message',
          [
            '@plugin_id' => $provider_config['plugin_id'],
            '@arg' => $arg['name'],
            '@message' => $e->getMessage(),
          ]
        );
      }
    }

    // If we successfully created any providers, use the first one
    // (Future: support multiple providers per argument)
    if (!empty($providers_for_arg)) {
      $completion_providers[$arg['name']] = $providers_for_arg[0];
    }
  }
}

// Register prompt with completion providers
$registry->registerPrompt($prompt, $handler, $completion_providers, TRUE);
```

### Step 4: Update McpServerFactory

In `src/McpServerFactory.php`, inject plugin manager into loader:

```php
$completionProviderManager = $this->container->get('plugin.manager.prompt_argument_completion_provider');
$promptLoader = new PromptConfigLoader(
  $this->mcpBridge,
  $this->logger,
  $completionProviderManager
);
```

### Step 5: Error Handling

Ensure the following error conditions are handled:
1. Plugin doesn't exist (caught by try-catch)
2. Plugin instantiation throws exception (caught by try-catch)
3. Plugin returns invalid data (trust the plugin, log if issues occur)
4. Empty completion providers array (handled by existing code)

### Step 6: Logging

Add debug logging to track provider registration:

```php
if (!empty($completion_providers)) {
  $this->logger->debug(
    'Registered @count completion providers for prompt "@prompt"',
    [
      '@count' => count($completion_providers),
      '@prompt' => $prompt_data['name'],
    ]
  );
}
```

### Step 7: Testing

After implementation:
1. Create a prompt config with `static_list` completion provider
2. Clear cache: `vendor/bin/drush cache:rebuild`
3. Open MCP Inspector and test autocomplete on the argument
4. Verify completions appear as you type
5. Check logs for any warnings/errors

</details>
