---
id: 4
group: "ui-implementation"
dependencies: [1, 2]
status: "completed"
created: "2025-11-17"
skills:
  - "drupal-backend"
---
# Add Completion Provider UI to Form

## Objective
Update `McpPromptConfigForm` to display available completion provider plugins as checkboxes for each argument, with plugin-specific configuration fields that appear based on selection.

## Skills Required
- **drupal-backend**: Drupal Form API, #states, AJAX, plugin metadata extraction

## Acceptance Criteria
- [ ] Form displays "Autocomplete" details fieldset for each argument
- [ ] Available completion provider plugins shown as checkboxes with labels and descriptions
- [ ] Checking `static_list` plugin reveals textarea for entering values
- [ ] Configuration fields use #states to show/hide based on checkbox selection
- [ ] Form values submit and save correctly to entity's `completion_providers` field
- [ ] Form loads existing completion provider configurations correctly
- [ ] Multiple arguments can have different completion providers configured
- [ ] Details fieldset is collapsed by default to avoid overwhelming users

## Technical Requirements

**Modified Files**:
- `src/Form/McpPromptConfigForm.php` - Add completion provider UI

**Key APIs**:
- Drupal Form API
- Form #states for conditional visibility
- Plugin manager `getDefinitions()` to list plugins
- AJAX callbacks (if needed for complex interactions)

## Input Dependencies
- Task 1: Plugin manager with available plugin definitions
- Task 2: Entity schema supporting completion_providers field

## Output Artifacts
- User-friendly form for configuring completion providers
- Configuration that persists correctly on save

## Implementation Notes

<details>
<summary>Detailed Implementation Guide</summary>

### Step 1: Inject Plugin Manager

In `src/Form/McpPromptConfigForm.php`, add plugin manager injection:

```php
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager;

/**
 * The completion provider plugin manager.
 */
protected PromptArgumentCompletionProviderManager $completionProviderManager;

/**
 * {@inheritdoc}
 */
public static function create(ContainerInterface $container) {
  $instance = parent::create($container);
  $instance->completionProviderManager = $container->get('plugin.manager.prompt_argument_completion_provider');
  return $instance;
}
```

### Step 2: Add Completion Fieldset to Arguments

In the `buildForm()` method, after building each argument's fields, add completion UI:

```php
// For each argument in the arguments table
foreach ($arguments as $i => $argument) {
  // ... existing argument fields (name, description, required) ...

  // Add completion providers fieldset
  $form['arguments'][$i]['completion'] = [
    '#type' => 'details',
    '#title' => $this->t('Autocomplete'),
    '#open' => FALSE,
    '#description' => $this->t('Configure autocomplete suggestions for this argument.'),
  ];

  // Get available plugins
  $plugin_definitions = $this->completionProviderManager->getDefinitions();

  // Get existing completion provider configuration
  $existing_providers = $argument['completion_providers'] ?? [];
  $selected_plugin_id = $existing_providers[0]['plugin_id'] ?? NULL;
  $existing_config = $existing_providers[0]['configuration'] ?? [];

  // Build checkbox for each plugin
  foreach ($plugin_definitions as $plugin_id => $definition) {
    $checkbox_name = "arguments[$i][completion][{$plugin_id}_enabled]";

    $form['arguments'][$i]['completion']["{$plugin_id}_enabled"] = [
      '#type' => 'checkbox',
      '#title' => $definition['label'],
      '#description' => $definition['description'],
      '#default_value' => ($selected_plugin_id === $plugin_id) ? 1 : 0,
    ];

    // Add plugin-specific configuration fields
    $form['arguments'][$i]['completion']["{$plugin_id}_config"] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ":input[name=\"{$checkbox_name}\"]" => ['checked' => TRUE],
        ],
      ],
    ];

    // Static list plugin configuration
    if ($plugin_id === 'static_list') {
      $form['arguments'][$i]['completion']["{$plugin_id}_config"]['values'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Values'),
        '#description' => $this->t('Enter one value per line. These will be offered as autocomplete suggestions.'),
        '#default_value' => isset($existing_config['values']) ? implode("\n", $existing_config['values']) : '',
        '#rows' => 5,
      ];
    }

    // Future plugins can add their own config fields here
  }
}
```

### Step 3: Handle Form Submission

In the `submitForm()` method, extract completion provider configuration:

```php
// Process each argument's completion providers
foreach ($arguments as $i => $argument) {
  $completion_providers = [];

  // Get plugin definitions to iterate through
  $plugin_definitions = $this->completionProviderManager->getDefinitions();

  foreach ($plugin_definitions as $plugin_id => $definition) {
    // Check if this plugin is enabled
    if (!empty($values['arguments'][$i]['completion']["{$plugin_id}_enabled"])) {
      $provider_config = [
        'plugin_id' => $plugin_id,
      ];

      // Extract plugin-specific configuration
      if ($plugin_id === 'static_list') {
        $values_text = $values['arguments'][$i]['completion']["{$plugin_id}_config"]['values'] ?? '';
        if (!empty($values_text)) {
          $values_array = array_filter(
            array_map('trim', explode("\n", $values_text)),
            fn($v) => $v !== ''
          );
          $provider_config['configuration'] = ['values' => array_values($values_array)];
        }
      }

      // Only allow one provider per argument for now
      $completion_providers[] = $provider_config;
      break;
    }
  }

  // Add completion providers to argument if any were selected
  if (!empty($completion_providers)) {
    $arguments[$i]['completion_providers'] = $completion_providers;
  }
}

// Set processed arguments back to entity
$entity->set('arguments', $arguments);
```

### Step 4: Add Validation

Add form validation to ensure:
1. Only one completion provider can be selected per argument
2. Required configuration fields are filled when a provider is selected

```php
public function validateForm(array &$form, FormStateInterface $form_state) {
  parent::validateForm($form, $form_state);

  $arguments = $form_state->getValue('arguments') ?? [];

  foreach ($arguments as $i => $argument) {
    $enabled_count = 0;
    $plugin_definitions = $this->completionProviderManager->getDefinitions();

    foreach ($plugin_definitions as $plugin_id => $definition) {
      if (!empty($argument['completion']["{$plugin_id}_enabled"])) {
        $enabled_count++;

        // Validate static_list has values
        if ($plugin_id === 'static_list') {
          $values = $argument['completion']["{$plugin_id}_config"]['values'] ?? '';
          if (trim($values) === '') {
            $form_state->setErrorByName(
              "arguments][$i][completion][{$plugin_id}_config][values",
              $this->t('Static list completion provider requires at least one value.')
            );
          }
        }
      }
    }

    // Only allow one provider per argument
    if ($enabled_count > 1) {
      $form_state->setErrorByName(
        "arguments][$i][completion",
        $this->t('Only one completion provider can be enabled per argument.')
      );
    }
  }
}
```

### Step 5: UI Polish

Add helpful UI elements:
1. Help text explaining what completion providers do
2. Example values for static_list provider
3. Visual indication when a provider is configured (badge/count)

### Step 6: Testing

After implementation:
1. Edit a prompt configuration
2. Verify completion fieldset appears for each argument
3. Select `static_list` and enter values
4. Save the form
5. Reload the form and verify values persist
6. Verify validation prevents empty static lists
7. Verify only one provider can be selected

</details>
