---
id: 10
summary: "Add human-readable labels and auto-generated machine names to prompt arguments per MCP 2025-06-18 spec"
created: 2025-11-18
---

# Plan: Prompt Argument Labels with Auto-Generated Machine Names

## Original Work Order

> Add Label (Title) and Auto-Generated Machine Name Support to Prompt Arguments
>
> Overview
>
> Rename argument name → machine_name, add required label field with Drupal's machine_name form element
> for auto-generation with AJAX preview. No backward compatibility.
>
> Changes Required
>
> 1. Configuration Schema (config/schema/mcp_server.schema.yml)
> - Rename name field to machine_name in argument mapping
> - Add required label field for human-readable title
> - Update field order: label, machine_name, description, required, completion_providers
>
> 2. Entity Class (src/Entity/McpPromptConfig.php)
> - Update PHPDoc to reflect new argument structure (label + machine_name)
> - Update getArgumentCompletionProviders() to use machine_name for lookups
>
> 3. Form UI (src/Form/McpPromptConfigForm.php)
> - Add label textfield (required) in buildArgumentElement()
> - Convert name field to machine_name form element with:
>   - Source: argument label field
>   - Live AJAX updates as user types
>   - Editable but defaults to auto-generated value
> - Update form element ordering: label first, then machine_name
>
> 4. MCP Protocol Output (src/Capability/Loader/PromptConfigLoader.php)
> - Update PromptArgument instantiation to use machine_name for the name parameter
> - Add title parameter using the label field (per MCP 2025-06-18 spec)
>
> 5. Code References
> - Update all $arg['name'] references to $arg['machine_name'] throughout codebase
>
> 6. Tests (tests/src/Functional/McpPromptConfigTest.php)
> - Update test fixtures to use label + machine_name
> - Add test for machine name auto-generation
> - Verify MCP protocol output includes both fields correctly
>
> 7. Validation
> - Ensure machine_name uniqueness within a prompt's arguments
> - Validate machine_name format (lowercase, underscores, no spaces)

## Executive Summary

This plan implements support for human-readable labels on prompt arguments per the MCP (Model Context Protocol) 2025-06-18 specification. The implementation refactors the existing `name` field to `machine_name` and introduces a required `label` field for display purposes, aligning with the MCP spec's `BaseMetadata` interface that distinguishes between programmatic identifiers (`name`) and human-readable titles (`title`).

The approach leverages Drupal's standard `machine_name` form element to provide live AJAX-based machine name generation from user-entered labels, creating a familiar and intuitive UX pattern already established in Drupal core. This eliminates the need for users to manually create machine-safe identifiers while maintaining full editability for advanced use cases.

Key benefits include improved UX in MCP clients (which can display human-friendly labels instead of technical identifiers), alignment with MCP protocol standards, and consistency with Drupal's established patterns for entity ID generation.

## Context

### Current State

Prompt arguments in the mcp_server module currently use a single `name` field that serves dual purposes:
- Machine identifier for programmatic references (lookups, completion providers)
- Display label in UI contexts (forms, MCP Inspector)

This creates usability issues:
- Users must enter machine-safe names (lowercase, underscores, no spaces)
- MCP clients receive technical identifiers like `search_query` instead of user-friendly labels like "Search Query"
- Module doesn't comply with MCP 2025-06-18 spec which defines separate `name` and `title` fields

Current argument structure:
```php
$arguments[] = [
  'name' => 'search_query',  // Used for both machine ID and display
  'description' => 'The search query to execute',
  'required' => TRUE,
  'completion_providers' => [...]
];
```

### Target State

After implementation, arguments will have distinct label and machine_name fields:

```php
$arguments[] = [
  'label' => 'Search Query',  // Human-readable, used for display
  'machine_name' => 'search_query',  // Auto-generated, used for programmatic reference
  'description' => 'The search query to execute',
  'required' => TRUE,
  'completion_providers' => [...]
];
```

MCP protocol output will align with spec:
```php
new PromptArgument(
  name: 'search_query',  // From machine_name
  title: 'Search Query',  // From label (NEW per 2025-06-18 spec)
  description: 'The search query to execute',
  required: TRUE,
);
```

Users will enter natural language labels ("Search Query", "Max Results", "File Path") and the system will auto-generate machine names (`search_query`, `max_results`, `file_path`) with the option to override.

### Background

The MCP 2025-06-18 schema introduced the `BaseMetadata` interface with distinct `name` and `title` fields:
- `name` (string, required): For programmatic/logical use
- `title` (string, optional): For UI contexts, optimized for human readability

This pattern is used throughout MCP for Prompts, Resources, Tools, and PromptArguments. The module currently only implements the `name` field, making it non-compliant with the latest spec and providing suboptimal UX in MCP clients.

Drupal's `machine_name` form element provides the perfect solution:
- Standard pattern across all Drupal entity types
- Live AJAX preview of machine name as user types
- Automatic conversion rules (lowercase, replace spaces with underscores, remove special chars)
- Editable override capability for power users
- Validation and uniqueness checking built-in

## Technical Implementation Approach

### Component 1: Configuration Schema Updates

**Objective**: Update the configuration schema to define the new argument structure with label and machine_name fields.

**Implementation**:
- Modify `config/schema/mcp_server.schema.yml` in the `mcp_server.mcp_prompt.arguments` mapping
- Rename `name` field to `machine_name` (type: string, label: 'Machine name')
- Add new `label` field (type: label, label: 'Label') as the first field
- Reorder fields for logical flow: label, machine_name, description, required, completion_providers

This establishes the data contract for all argument storage and provides Drupal's configuration system with proper validation rules.

### Component 2: Entity Class Modifications

**Objective**: Update the McpPromptConfig entity to reflect the new argument structure in code.

**Implementation**:
- Update PHPDoc in `src/Entity/McpPromptConfig.php` for the `$arguments` property to document new structure
- Modify `getArgumentCompletionProviders($argument_name)` method to use `machine_name` for array filtering instead of `name`
- Ensure all internal entity methods reference the correct field names

This ensures type safety, IDE autocomplete, and correct programmatic access to arguments.

### Component 3: Form UI Enhancement

**Objective**: Implement the Drupal machine_name form element for seamless label-to-machine-name conversion.

**Implementation** in `src/Form/McpPromptConfigForm.php`:

1. **Add label field** in `buildArgumentElement()`:
   ```php
   $form['label'] = [
     '#type' => 'textfield',
     '#title' => $this->t('Label'),
     '#default_value' => $arg['label'] ?? '',
     '#required' => TRUE,
   ];
   ```

2. **Convert name to machine_name element**:
   ```php
   $form['machine_name'] = [
     '#type' => 'machine_name',
     '#title' => $this->t('Machine name'),
     '#default_value' => $arg['machine_name'] ?? '',
     '#machine_name' => [
       'source' => ['arguments', $index, 'label'],
       'exists' => [$this, 'argumentMachineNameExists'],
     ],
     '#required' => TRUE,
   ];
   ```

3. **Implement validation callback** `argumentMachineNameExists()` to check uniqueness within the current prompt's arguments

4. **Reorder form elements**: label → machine_name → description → required → completion_providers

This provides the standard Drupal UX with live machine name generation and validation.

### Component 4: MCP Protocol Output Updates

**Objective**: Ensure MCP protocol responses include both `name` and `title` per the 2025-06-18 spec.

**Implementation** in `src/Capability/Loader/PromptConfigLoader.php`:

Update the `PromptArgument` instantiation in the prompt loading logic:
```php
$arguments[] = new PromptArgument(
  name: $arg['machine_name'],  // Changed from $arg['name']
  title: $arg['label'],  // NEW - implements MCP spec
  description: $arg['description'] ?? NULL,
  required: $arg['required'] ?? FALSE,
);
```

This ensures MCP clients receive properly formatted argument metadata with both programmatic identifiers and human-readable titles.

### Component 5: Codebase-Wide Reference Updates

**Objective**: Update all code references from `$arg['name']` to `$arg['machine_name']`.

**Implementation**:
- Search codebase for `$arg['name']`, `$argument['name']`, and similar patterns
- Update to `$arg['machine_name']` and `$argument['machine_name']`
- Key locations:
  - `src/Entity/McpPromptConfig.php::getArgumentCompletionProviders()`
  - `src/Capability/Loader/PromptConfigLoader.php::buildCompletionProviders()`
  - Any other argument array access points

This maintains consistency across the codebase and prevents runtime errors.

### Component 6: Test Updates

**Objective**: Update test fixtures and add new test coverage for machine name functionality.

**Implementation** in `tests/src/Functional/McpPromptConfigTest.php`:

1. Update all test argument fixtures to include both `label` and `machine_name`
2. Add test method to verify machine name auto-generation behavior
3. Add test to verify MCP protocol output includes `title` field
4. Verify argument lookups by machine_name work correctly

This ensures the new functionality is properly tested and prevents regressions.

### Component 7: Validation Logic

**Objective**: Ensure data integrity through proper validation of machine names.

**Implementation**:
- Add validation in form to ensure machine_name uniqueness within a prompt
- Validate machine_name format (pattern: `^[a-z0-9_]+$`)
- Check for empty machine_name values
- Add entity-level validation in `McpPromptConfig::validateArguments()` if needed

This prevents invalid configurations and duplicate machine names.

## Risk Considerations and Mitigation Strategies

### Technical Risks

- **Breaking Change Risk**: Renaming `name` to `machine_name` breaks existing saved configurations
  - **Mitigation**: Per user requirement, no backward compatibility needed. Module is in active development and not yet released. Document breaking change clearly for any alpha users.

- **AJAX Complexity**: Machine name form element requires proper AJAX callback configuration
  - **Mitigation**: Use Drupal's standard machine_name element which handles all AJAX complexity. Follow existing patterns from entity ID fields.

- **Form State Management**: Dynamically added arguments need proper AJAX form state handling
  - **Mitigation**: Existing form already handles dynamic arguments with AJAX. Machine name element integrates seamlessly with current implementation.

### Implementation Risks

- **Incomplete Reference Updates**: Missing some `$arg['name']` → `$arg['machine_name']` conversions could cause runtime errors
  - **Mitigation**: Use comprehensive grep/search to find all references. Run full test suite. Test MCP protocol output manually with MCP Inspector.

- **Validation Edge Cases**: Uniqueness validation needs to handle edge cases (removed arguments, reordering, etc.)
  - **Mitigation**: Leverage Drupal's form API validation system. Test with multiple add/remove/reorder operations.

### Quality Risks

- **Test Coverage Gaps**: New functionality might not be fully covered by existing tests
  - **Mitigation**: Add specific tests for label→machine_name conversion, uniqueness validation, and MCP protocol output format.

## Success Criteria

### Primary Success Criteria

1. Users can enter human-readable labels for arguments and see auto-generated machine names in real-time
2. MCP protocol output includes both `name` (from machine_name) and `title` (from label) fields per 2025-06-18 spec
3. All existing functionality (completion providers, argument lookups, form operations) works with new structure
4. MCP Inspector displays human-readable labels instead of technical identifiers

### Quality Assurance Metrics

1. All existing tests pass with updated fixtures
2. New tests cover machine name generation and validation
3. Manual testing confirms AJAX machine name generation works in browser
4. Code follows Drupal coding standards (phpcs passes)
5. PHPStan static analysis passes with no new errors

## Resource Requirements

### Development Skills

- PHP 8.3+ with strict typing and readonly properties
- Drupal 11 Form API and configuration entity system
- Drupal AJAX/form state management
- MCP protocol 2025-06-18 specification knowledge
- PHPUnit functional testing in Drupal

### Technical Infrastructure

- Drupal 11.1 environment
- PHP 8.3
- Composer for dependency management
- PHPUnit for testing
- DDEV for local development environment
- MCP Inspector for protocol validation

## Notes

**No Backward Compatibility**: Per user requirements, this is a breaking change with no migration path or backward compatibility layers. The module is in active development and not yet released, making this an acceptable approach.

**Form Element Pattern**: Using Drupal's `machine_name` form element provides significant benefits:
- Established UX pattern familiar to Drupal users
- Built-in AJAX handling
- Standard validation
- Accessibility compliance
- No custom JavaScript required

**MCP Spec Compliance**: Implementing the `title` field brings the module into full compliance with the MCP 2025-06-18 specification's `BaseMetadata` interface, future-proofing the implementation.

## Task Dependencies

```mermaid
graph TD
    001[Task 01: Update Configuration Schema] --> 002[Task 02: Update Entity Class]
    001 --> 003[Task 03: Form UI with Machine Name]
    001 --> 004[Task 04: Update MCP Protocol Output]
    002 --> 003
    002 --> 004
    003 --> 005[Task 05: Update Tests]
    004 --> 005
```

## Execution Blueprint

**Validation Gates:**
- Reference: `.ai/task-manager/config/hooks/POST_PHASE.md`

### ✅ Phase 1: Foundation - Schema Updates
**Parallel Tasks:**
- ✔️ Task 01: Update Configuration Schema (status: completed)

**Phase objective**: Establish the data structure contract for label and machine_name fields in configuration storage.

### ✅ Phase 2: Core Implementation
**Parallel Tasks:**
- ✔️ Task 02: Update Entity Class (depends on: 01, status: completed)
- ✔️ Task 03: Implement Form UI with Machine Name (depends on: 01, 02, status: completed)
- ✔️ Task 04: Update MCP Protocol Output (depends on: 01, 02, status: completed)

**Phase objective**: Implement the core functionality across entity layer, form UI, and MCP protocol output.

### ✅ Phase 3: Verification and Testing
**Parallel Tasks:**
- ✔️ Task 05: Update Tests (depends on: 01, 02, 03, 04, status: completed)

**Phase objective**: Verify all changes work correctly through automated tests and manual validation.

### Execution Summary
- Total Phases: 3
- Total Tasks: 5
- Maximum Parallelism: 3 tasks (in Phase 2)
- Critical Path Length: 3 phases
- Estimated Complexity: Low-Medium (all tasks ≤5.5 complexity score)

## Execution Summary

**Status**: ✅ Completed Successfully
**Completed Date**: 2025-11-18

### Results

Successfully implemented human-readable labels and auto-generated machine names for prompt arguments throughout the MCP server module, achieving full compliance with the MCP 2025-06-18 specification.

**Key Deliverables**:
1. **Configuration Schema**: Updated to support separate `label` and `machine_name` fields
2. **Entity Layer**: McpPromptConfig entity fully refactored to use new field structure
3. **Form UI**: Implemented Drupal's machine_name element with live AJAX auto-generation from labels
4. **MCP Protocol Output**: PromptArgument objects now include both `name` (machine_name) and `title` (label) fields
5. **MCP SDK Updates**: Enhanced PromptArgument class to support the title field per spec
6. **Comprehensive Testing**: All 27 tests passing, test fixtures updated, manual testing guide created

**Code Quality**:
- All linting checks passed (PHPCS, PHPStan, ESLint, Prettier)
- No spelling errors
- Zero test failures
- Clean git commits with conventional commit format

### Noteworthy Events

**Smooth Execution**: All three phases executed without blocking issues. Task dependencies were properly structured, allowing efficient parallel execution in Phase 2.

**MCP SDK Modification**: Updated the vendor package `mcp/sdk` to add `title` parameter support to PromptArgument class. This modification enables proper MCP 2025-06-18 spec compliance. Note: This change should be contributed upstream or managed through composer patches.

**Test Strategy Efficiency**: Tests were already updated as part of Task 3 (Form UI implementation), which streamlined Task 5 execution. The testing-qa-engineer agent efficiently identified and completed remaining test fixture updates across 5 configuration YAML files.

**No Backward Compatibility**: Per user requirements, this implementation includes breaking changes with no migration path. Appropriate for a module in active development that hasn't been released.

### Recommendations

1. **Manual Browser Testing**: Execute the scenarios documented in `MANUAL_TESTING.md` to verify AJAX functionality works correctly in a real browser environment

2. **MCP SDK Patch**: Consider creating a composer patch for the `mcp/sdk` package changes, or contribute the `title` field support upstream to the official MCP SDK repository

3. **Documentation Update**: Update module README.md to highlight the new label/machine_name functionality and MCP 2025-06-18 compliance

4. **Feature Demonstration**: Test with MCP Inspector to verify that human-readable labels now appear in the UI instead of technical machine names

5. **Future Enhancement**: Consider adding inline help text to the form explaining the auto-generation behavior of machine names from labels
