---
id: 3
group: "form-ui"
dependencies: [1, 2]
status: "completed"
created: 2025-11-18
skills:
  - php
  - drupal-backend
---
# Implement Form UI with Label and Machine Name Fields

## Objective
Add label textfield and machine_name form element with AJAX-based auto-generation to the prompt argument form, including uniqueness validation.

## Skills Required
- Drupal Form API
- Drupal AJAX framework
- PHP 8.3+

## Acceptance Criteria
- [ ] Label textfield added to argument form element
- [ ] Machine name field converted to `#type => 'machine_name'` with AJAX
- [ ] Fields ordered: label, machine_name, description, required, completion_providers
- [ ] Uniqueness validation callback `argumentMachineNameExists()` implemented
- [ ] Form works correctly for add/remove/edit argument operations
- [ ] AJAX machine name generation functions properly

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
**File**: `src/Form/McpPromptConfigForm.php`

**Method to modify**: `buildArgumentElement()` at lines ~179-228

**New fields structure**:
```php
// 1. Add label field
$element['label'] = [
  '#type' => 'textfield',
  '#title' => $this->t('Label'),
  '#default_value' => $arg['label'] ?? '',
  '#required' => TRUE,
  '#weight' => -10,
];

// 2. Convert name to machine_name
$element['machine_name'] = [
  '#type' => 'machine_name',
  '#title' => $this->t('Machine name'),
  '#default_value' => $arg['machine_name'] ?? '',
  '#machine_name' => [
    'source' => ['arguments', $index, 'label'],
    'exists' => [$this, 'argumentMachineNameExists'],
  ],
  '#required' => TRUE,
  '#weight' => -9,
  '#disabled' => !empty($arg['machine_name']),  // Lock after first save
];
```

**New validation callback to add**:
```php
/**
 * Checks if an argument machine name already exists in the current prompt.
 *
 * @param string $machine_name
 *   The machine name to check.
 * @param array $element
 *   The form element.
 * @param \Drupal\Core\Form\FormStateInterface $form_state
 *   The form state.
 *
 * @return bool
 *   TRUE if the machine name exists, FALSE otherwise.
 */
public function argumentMachineNameExists(string $machine_name, array $element, FormStateInterface $form_state): bool {
  $arguments = $form_state->getValue('arguments') ?? [];
  $current_index = $element['#array_parents'][1] ?? NULL;

  foreach ($arguments as $index => $argument) {
    // Skip the current argument being edited
    if ($index === $current_index) {
      continue;
    }
    if (($argument['machine_name'] ?? '') === $machine_name) {
      return TRUE;
    }
  }
  return FALSE;
}
```

## Input Dependencies
- Task 1: Schema defines the data structure
- Task 2: Entity class expects machine_name field

## Output Artifacts
- Modified `src/Form/McpPromptConfigForm.php` with label and machine_name fields
- Functional AJAX-based machine name generation in admin UI

<details>
<summary>Implementation Notes</summary>

**Step 1: Locate and read the buildArgumentElement() method**
- File: `src/Form/McpPromptConfigForm.php`
- Method: `buildArgumentElement()` starting around line 179
- Current structure has: name, description, required, completion_providers

**Step 2: Add the label field**
Position it first (highest weight priority):
```php
$element['label'] = [
  '#type' => 'textfield',
  '#title' => $this->t('Label'),
  '#default_value' => $arg['label'] ?? '',
  '#required' => TRUE,
  '#weight' => -10,
];
```

**Step 3: Modify the existing 'name' field**
Find the current name field definition (~line 185-190) and replace it:
```php
// OLD (remove this):
$element['name'] = [
  '#type' => 'textfield',
  '#title' => $this->t('Name'),
  // ...
];

// NEW (replace with this):
$element['machine_name'] = [
  '#type' => 'machine_name',
  '#title' => $this->t('Machine name'),
  '#default_value' => $arg['machine_name'] ?? '',
  '#machine_name' => [
    'source' => ['arguments', $index, 'label'],
    'exists' => [$this, 'argumentMachineNameExists'],
    'standalone' => TRUE,
  ],
  '#required' => TRUE,
  '#weight' => -9,
  '#disabled' => !empty($arg['machine_name']),
];
```

**Step 4: Add the validation callback**
Add this new public method to the McpPromptConfigForm class:
```php
public function argumentMachineNameExists(string $machine_name, array $element, FormStateInterface $form_state): bool {
  $arguments = $form_state->getValue('arguments') ?? [];
  $current_index = $element['#array_parents'][1] ?? NULL;

  foreach ($arguments as $index => $argument) {
    if ($index === $current_index) {
      continue;
    }
    if (($argument['machine_name'] ?? '') === $machine_name) {
      return TRUE;
    }
  }
  return FALSE;
}
```

**Step 5: Update weights for other fields**
Ensure proper ordering:
- label: -10
- machine_name: -9
- description: -8
- required: -7
- completion_providers: 0 (default)

**Step 6: Handle form submission**
The existing `submitForm()` method should work without changes because it processes the entire arguments array. Just verify that both `label` and `machine_name` are being saved.

**Step 7: Testing checklist**:
1. Clear cache: `vendor/bin/drush cache:rebuild`
2. Visit the prompt configuration form in browser
3. Add a new argument and verify:
   - Typing in Label field auto-updates Machine name field
   - Machine name converts to lowercase and replaces spaces with underscores
   - Can override the machine name if needed
   - Validation prevents duplicate machine names
4. Save and verify both fields persist

**Important AJAX considerations**:
- The `machine_name` form element has built-in AJAX support
- Source path `['arguments', $index, 'label']` must match the form structure
- The form already has AJAX for add/remove buttons, so AJAX should work seamlessly
- The `#disabled` property prevents editing after first save (optional, can remove if not desired)

**Troubleshooting**:
- If AJAX doesn't work, check browser console for JavaScript errors
- Verify the source path matches the form array structure
- Check that form has `#ajax` wrapper on the arguments container
- The existing form already has AJAX infrastructure for dynamic arguments
</details>
