---
id: 4
group: "mcp-protocol"
dependencies: [1, 2]
status: "completed"
created: 2025-11-18
skills:
  - php
  - drupal-backend
---
# Update MCP Protocol Output to Include Title Field

## Objective
Modify PromptArgument instantiation to use machine_name for the name parameter and add the title parameter using the label field, per MCP 2025-06-18 specification.

## Skills Required
- PHP 8.3+
- MCP protocol understanding
- Drupal service layer patterns

## Acceptance Criteria
- [ ] PromptArgument uses `$arg['machine_name']` for name parameter
- [ ] PromptArgument includes new `title: $arg['label']` parameter
- [ ] MCP protocol output conforms to 2025-06-18 spec
- [ ] Manual verification with MCP Inspector shows title field

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
**File**: `src/Capability/Loader/PromptConfigLoader.php`

**Current code** (~line 48-56):
```php
$arguments[] = new PromptArgument(
  name: $arg['name'],
  description: $arg['description'] ?? NULL,
  required: $arg['required'] ?? FALSE,
);
```

**Target code**:
```php
$arguments[] = new PromptArgument(
  name: $arg['machine_name'],  // Changed from 'name'
  title: $arg['label'],  // NEW - MCP 2025-06-18 spec
  description: $arg['description'] ?? NULL,
  required: $arg['required'] ?? FALSE,
);
```

**MCP Spec Reference** (2025-06-18 BaseMetadata):
- `name` (string, required): For programmatic/logical use
- `title` (string, optional): For UI contexts, optimized for human readability

## Input Dependencies
- Task 1: Schema defines label and machine_name structure
- Task 2: Entity class provides access to arguments with new structure

## Output Artifacts
- Modified `src/Capability/Loader/PromptConfigLoader.php` with MCP spec-compliant output
- Arguments in MCP protocol responses include both name and title fields

<details>
<summary>Implementation Notes</summary>

**Step 1: Locate the PromptArgument instantiation**
- File: `src/Capability/Loader/PromptConfigLoader.php`
- Look for the loop creating PromptArgument objects (~line 48-56)
- This is in the method that loads prompt configurations

**Step 2: Update the PromptArgument constructor call**
Replace:
```php
$arguments[] = new PromptArgument(
  name: $arg['name'],
  description: $arg['description'] ?? NULL,
  required: $arg['required'] ?? FALSE,
);
```

With:
```php
$arguments[] = new PromptArgument(
  name: $arg['machine_name'],
  title: $arg['label'],
  description: $arg['description'] ?? NULL,
  required: $arg['required'] ?? FALSE,
);
```

**Step 3: Verify PromptArgument constructor signature**
Check the PromptArgument class (likely from `mcp/sdk` package) to ensure it accepts the `title` parameter. The 2025-06-18 spec defines it as optional.

**Step 4: Search for other argument name references**
Use Grep to find any other places in this file where `$arg['name']` is accessed:
```bash
grep -n "\$arg\['name'\]" src/Capability/Loader/PromptConfigLoader.php
```

If found in other locations (like buildCompletionProviders method), update those as well.

**Step 5: Check buildCompletionProviders method**
Around line 110-152, this method likely uses argument names. Ensure it references machine_name:
```php
// Look for patterns like:
$argument_name = $arg['name'];  // Change to: $arg['machine_name']
```

**Step 6: Verification**
After making changes:
1. Clear cache: `vendor/bin/drush cache:rebuild`
2. Start MCP server or access MCP Inspector
3. View prompt arguments in MCP protocol output
4. Verify JSON includes both fields:
   ```json
   {
     "name": "search_query",
     "title": "Search Query",
     "description": "...",
     "required": true
   }
   ```

**Common patterns to update**:
- `$arg['name']` → `$arg['machine_name']`
- Anywhere arguments are iterated and name field is accessed
- Completion provider lookups that use argument names

**Testing approach**:
1. Create a test prompt with arguments via the UI
2. Query the MCP server's prompts/list endpoint
3. Inspect the JSON response
4. Confirm `title` field appears alongside `name`
5. Confirm `name` contains the machine_name value
6. Confirm `title` contains the label value

**Edge cases**:
- Empty label values (shouldn't happen due to form validation, but handle gracefully)
- Arguments created before this change (won't have label field - handle with `??` operator)
</details>
