---
id: 5
group: "testing"
dependencies: [1, 2, 3, 4]
status: "completed"
created: 2025-11-18
skills:
  - php
  - drupal-testing
---
# Update Tests for Label and Machine Name Fields

## Objective
Update test fixtures to use label and machine_name fields, and verify MCP protocol output includes title field correctly.

## Skills Required
- PHPUnit
- Drupal functional testing
- PHP 8.3+

## Acceptance Criteria
- [ ] All test argument fixtures updated with label and machine_name
- [ ] MCP protocol output assertions verify title field presence
- [ ] All existing tests pass
- [ ] Manual browser testing confirms AJAX functionality

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
**File**: `tests/src/Functional/McpPromptConfigTest.php`

**Test updates needed**:
1. Update argument fixtures from `name` to `label` + `machine_name`
2. Add assertions for MCP protocol title field
3. Test argument lookup by machine_name

**Current test fixture pattern**:
```php
'arguments' => [
  [
    'name' => 'query',
    'description' => 'The search query',
    'required' => TRUE,
  ],
]
```

**Target fixture pattern**:
```php
'arguments' => [
  [
    'label' => 'Search Query',
    'machine_name' => 'query',
    'description' => 'The search query',
    'required' => TRUE,
  ],
]
```

## Input Dependencies
- Tasks 1-4: All code changes must be complete before tests will pass

## Output Artifacts
- Modified `tests/src/Functional/McpPromptConfigTest.php` with passing tests
- Verified MCP protocol compliance via test assertions

<details>
<summary>Implementation Notes</summary>

**IMPORTANT TEST PHILOSOPHY**:
Per project guidelines, we have ONE Functional test with ONE test function. Use helper methods to organize test scenarios within that single test function.

**Step 1: Read the existing test file**
```bash
vendor/bin/phpunit --list-tests web/modules/contrib/mcp_server/tests/src/Functional/
```
Examine `McpPromptConfigTest.php` to understand current structure.

**Step 2: Update all argument fixtures**
Find all arrays with 'arguments' key and update from:
```php
'name' => 'value'
```
To:
```php
'label' => 'Human Readable Value',
'machine_name' => 'value',
```

Common locations:
- Test setup/data provider methods
- Inline test data arrays
- Fixture creation methods

**Step 3: Add MCP protocol output verification**
Within the single test function, add a helper method or inline assertions:
```php
// Example assertion pattern (adapt to actual test structure):
$mcp_output = $this->getMcpProtocolOutput();
$this->assertArrayHasKey('title', $mcp_output['arguments'][0]);
$this->assertEquals('Search Query', $mcp_output['arguments'][0]['title']);
$this->assertEquals('query', $mcp_output['arguments'][0]['name']);
```

**Step 4: Verify argument lookup functionality**
If tests verify completion providers or argument lookups, ensure they use machine_name:
```php
$providers = $prompt->getArgumentCompletionProviders('query');  // Uses machine_name
$this->assertNotEmpty($providers);
```

**Step 5: Run the tests**
```bash
cd /var/www/html
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/
```

**Step 6: Manual browser testing**
Since we only have ONE functional test per module policy, manual testing is important:
1. Clear cache: `vendor/bin/drush cache:rebuild`
2. Navigate to prompt configuration form
3. Add new prompt with arguments
4. Verify:
   - Label field exists and is required
   - Machine name field auto-generates from label
   - Can type "Search Query" in label, see "search_query" in machine_name
   - Can override machine name if needed
   - Save works correctly
   - Edit loads both fields correctly
   - MCP Inspector shows title field

**Common test patterns to update**:
```php
// OLD:
$args = [
  'name' => 'test',
  'description' => 'Test arg',
];

// NEW:
$args = [
  'label' => 'Test Argument',
  'machine_name' => 'test',
  'description' => 'Test arg',
];
```

**Edge cases to test manually** (since we can't add new test functions):
- Empty label (should fail validation)
- Duplicate machine names (should fail validation)
- Special characters in label (should convert in machine name)
- Editing existing arguments (should preserve machine_name)
- Adding/removing arguments dynamically
- AJAX updates of machine name field

**Test data best practices**:
- Use realistic labels: "Search Query", "Max Results", "File Path"
- Use valid machine names: "search_query", "max_results", "file_path"
- Include both required and optional arguments
- Test with completion providers attached

**If tests fail**:
1. Check that all code changes are complete (Tasks 1-4)
2. Clear all caches
3. Verify database schema is updated
4. Check for typos in field names (label vs machine_name)
5. Review form submission handling
6. Debug with dpm() or error logs if using devel module

**PHPStan validation**:
```bash
vendor/bin/phpstan analyse web/modules/contrib/mcp_server/tests/
```

**Code style check**:
```bash
vendor/bin/phpcs --standard=Drupal web/modules/contrib/mcp_server/tests/
```
</details>
