---
id: 3
group: "form-integration"
dependencies: [1, 2]
status: "completed"
created: "2025-11-18"
completed: "2025-11-18"
skills:
  - drupal-backend
  - forms
---
# Create PluginSelectionFormTrait for Reusable AJAX Plugin Forms

## Objective
Create a reusable trait that provides AJAX-based plugin selection with dynamic configuration form injection, following the pattern from ab_tests module's PluginSelectionFormTrait.

## Skills Required
- drupal-backend: Traits, plugin managers, SubformState
- forms: AJAX callbacks, form state management, dynamic forms

## Acceptance Criteria
- [x] Trait created at src/Form/PluginSelectionFormTrait.php
- [x] injectPluginSelector() method supports multiple cardinality (checkboxes)
- [x] AJAX callback returns plugin configuration wrapper
- [x] SubformState used for isolated plugin form processing
- [x] Trait methods handle validation and submission

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Create new trait file
- Implement AJAX callback for plugin selection
- Use SubformState::createForSubform() for plugin forms
- Support checkbox-based multiple plugin selection
- Handle plugin configuration extraction

## Input Dependencies
- Reference implementation: ab_tests module's PluginSelectionFormTrait
- Updated plugin architecture from Task 1
- Configurable plugins from Task 2
- FormStateInterface, SubformState from Drupal core

## Output Artifacts
- PluginSelectionFormTrait with all required methods
- AJAX-enabled plugin selector with configuration forms
- Reusable infrastructure for any configurable plugin type
- Documented trait requirements and usage

<details>
<summary>Implementation Notes</summary>

### Trait Structure
File: `src/Form/PluginSelectionFormTrait.php`

Required methods:
1. `injectPluginSelector()` - Main injection method
2. `pluginSelectionAjaxCallback()` - AJAX callback
3. `buildPluginConfigurationForm()` - Build single plugin config form
4. `validatePluginForm()` - Validate selected plugin forms
5. `updatePluginConfiguration()` - Process submission
6. `getSelectedPluginsFromFormState()` - Extract selections from form state
7. `getSelectedPluginsFromSettings()` - Extract selections from saved config

### Trait Requirements (Document in Docblock)
Implementing class must provide:
- `completionProviderManager()`: Returns plugin manager instance
- Use `StringTranslationTrait` for `$this->t()`

### injectPluginSelector() Method Signature
```php
protected function injectPluginSelector(
  array &$element,
  FormStateInterface $form_state,
  array $existing_configuration,
  string $field_name,
  int $argument_index,
  bool $multiple_cardinality = TRUE
): void
```

### Key Implementation Details

**Checkbox Structure:**
```php
$element[$field_name] = [
  '#type' => 'checkboxes',
  '#title' => $this->t('Completion providers'),
  '#options' => $plugin_options, // From plugin manager definitions
  '#default_value' => $selected_plugins,
  '#ajax' => [
    'callback' => [$this, 'pluginSelectionAjaxCallback'],
    'wrapper' => "argument-{$argument_index}-completion-config-wrapper",
    'event' => 'change',
  ],
];
```

**AJAX Wrapper:**
```php
$element['config_wrapper'] = [
  '#type' => 'container',
  '#prefix' => "<div id=\"argument-{$argument_index}-completion-config-wrapper\">",
  '#suffix' => '</div>',
];
```

**Plugin Configuration Forms:**
For each selected plugin:
```php
$plugin = $plugin_manager->createInstance($plugin_id, $settings);
$subform = [];
$subform_state = SubformState::createForSubform($element['config_wrapper'][$plugin_id], $element, $form_state);
$subform = $plugin->buildConfigurationForm($subform, $subform_state);
$element['config_wrapper'][$plugin_id] = [
  '#type' => 'details',
  '#title' => $plugin_definition['label'],
  '#open' => TRUE,
] + $subform;
```

**AJAX Callback:**
```php
public function pluginSelectionAjaxCallback(array &$form, FormStateInterface $form_state) {
  $trigger = $form_state->getTriggeringElement();
  $parents = array_slice($trigger['#array_parents'], 0, -1);
  $parents[] = 'config_wrapper';
  $element = NestedArray::getValue($form, $parents);
  return $element;
}
```

**Validation Method:**
```php
protected function validatePluginForm(
  array &$form,
  FormStateInterface $form_state,
  string $field_name,
  int $argument_index,
  bool $multiple_cardinality = TRUE
): void {
  $selected_plugins = $this->getSelectedPluginsFromFormState($form_state, $field_name, $argument_index);

  foreach ($selected_plugins as $plugin_id) {
    $plugin = $this->completionProviderManager()->createInstance($plugin_id, []);
    $element_parents = ['arguments', $argument_index, 'config_wrapper', $plugin_id];
    $element = NestedArray::getValue($form, $element_parents);
    $subform_state = SubformState::createForSubform($element, $form, $form_state);
    $plugin->validateConfigurationForm($element, $subform_state);
  }
}
```

**Submission Method:**
```php
protected function updatePluginConfiguration(
  array &$form,
  FormStateInterface $form_state,
  string $field_name,
  int $argument_index,
  bool $multiple_cardinality = TRUE
): array {
  $selected_plugins = $this->getSelectedPluginsFromFormState($form_state, $field_name, $argument_index);
  $configurations = [];

  foreach ($selected_plugins as $plugin_id) {
    $plugin = $this->completionProviderManager()->createInstance($plugin_id, []);
    $element_parents = ['arguments', $argument_index, 'config_wrapper', $plugin_id];
    $element = NestedArray::getValue($form, $element_parents);
    $subform_state = SubformState::createForSubform($element, $form, $form_state);
    $plugin->submitConfigurationForm($element, $subform_state);

    $configurations[] = [
      'plugin_id' => $plugin_id,
      'settings' => $plugin->getConfiguration(),
    ];
  }

  return $configurations;
}
```

### Reference Pattern
Study `Drupal\ab_tests\Form\PluginSelectionFormTrait` for:
- Complete method implementations
- Form state handling patterns
- Error prevention strategies
- Documentation style

### Testing After Implementation
```bash
cd /var/www/html && vendor/bin/drush cache:rebuild
cd /var/www/html && vendor/bin/phpstan analyse web/modules/contrib/mcp_server/
```

</details>
