---
id: 12
summary: "Create an entity query autocomplete provider plugin that queries Drupal entities and returns absolute URLs with path alias resolution"
created: 2025-11-18
---

# Plan: Entity Query Autocomplete Provider

## Original Work Order

> I need you to create an auto-completion argument provider that works with entity queries and is configurable to allow you to select from a drop-down the entity type and bundle for the entity query.
>
> The output of the autocomplete provider for the prompt argument is not the entity ID but the absolute full URL to the entity. This includes resolving any path aliases that might exist.

## Plan Clarifications

| Question | Answer |
|----------|--------|
| Should the entity query filter by entity access? | Yes, check access for the anonymous user |
| How should autocomplete search work? | Both label and ID |
| Result limit? | Yes, limit to 10 |
| Bundle selector required or optional? | Optional |

## Executive Summary

This plan creates a new `EntityQueryCompletionProvider` plugin that extends the existing prompt argument completion provider system in the MCP Server module. The provider will enable MCP prompts to autocomplete entity references by querying Drupal's entity system with configurable entity type and optional bundle filtering.

The key innovation is that instead of returning entity IDs, this provider returns absolute URLs to entities, including proper path alias resolution. This makes the autocomplete results immediately useful for external tools and AI agents that need to reference Drupal content via URLs rather than internal IDs.

The implementation follows the established plugin architecture pattern used by `StaticListCompletionProvider`, ensuring consistency with the existing codebase while adding sophisticated entity querying capabilities with access control, label/ID searching, and performance optimizations.

## Context

### Current State

The MCP Server module currently has:
- A plugin system for prompt argument completion providers (`PromptArgumentCompletionProviderInterface`)
- One existing provider (`StaticListCompletionProvider`) that provides static string completions
- Plugin configuration forms that allow per-argument customization
- Integration with the MCP protocol for exposing autocomplete capabilities to AI clients

However, there's no way to autocomplete entity references, which is a common need when building prompts that interact with Drupal content (nodes, users, taxonomy terms, media, etc.).

### Target State

After implementation, users will be able to:
1. Configure a prompt argument to use the new `EntityQueryCompletionProvider`
2. Select an entity type (node, user, taxonomy_term, media, etc.) from a dropdown
3. Optionally select a specific bundle (e.g., "article" for nodes, "tags" for taxonomy terms)
4. Have autocomplete suggestions return absolute URLs to entities that:
   - Match the search term (by label or ID)
   - The anonymous user has access to view
   - Have path aliases resolved (e.g., `/blog/my-article` instead of `/node/123`)
   - Are limited to 10 results for performance

### Background

**Drupal Entity System**: Entities in Drupal are content objects with standardized APIs. Common entity types include:
- `node`: Content items (articles, pages, etc.)
- `user`: User accounts
- `taxonomy_term`: Category/tag terms
- `media`: Media items (images, documents, etc.)
- Each entity type may have multiple bundles (e.g., nodes can be "article", "page", "blog")

**URL Generation with Aliases**: Drupal's path alias system allows `/node/123` to be aliased as `/blog/my-article`. The `Url::fromRoute()` and `->setAbsolute()` methods handle this conversion automatically.

**Access Control**: Entity access must be checked to ensure only viewable entities are suggested. We'll use the anonymous user context for consistent, safe access checking.

**Performance Considerations**: Entity queries can be expensive. We'll limit results to 10 items and use efficient query methods.

## Technical Implementation Approach

### Component 1: Entity Query Completion Provider Plugin

**Objective**: Create a new plugin class that implements the completion provider interface with entity query capabilities.

```mermaid
classDiagram
    PromptArgumentCompletionProviderBase <|-- EntityQueryCompletionProvider
    PromptArgumentCompletionProviderInterface <.. EntityQueryCompletionProvider

    class EntityQueryCompletionProvider {
        +getCompletions(string, array) array
        +buildConfigurationForm(array, FormStateInterface) array
        +validateConfigurationForm(array, FormStateInterface) void
        +submitConfigurationForm(array, FormStateInterface) void
        -buildEntityQuery(string, array) QueryInterface
        -formatEntityAsUrl(EntityInterface) string
    }
```

The plugin will:
- Extend `PromptArgumentCompletionProviderBase` for standard plugin functionality
- Use dependency injection to access required Drupal services:
  - `entity_type.manager`: For entity type definitions and queries
  - `entity_type.bundle.info`: For bundle information
  - `path_alias.manager`: For resolving path aliases
  - `current_user`: To switch to anonymous user context for access checks
- Implement the `#[PromptArgumentCompletionProvider]` attribute with metadata
- Use `final` class modifier following the module's coding standards

### Component 2: Configuration Form

**Objective**: Provide a user-friendly form for configuring entity type and optional bundle selection.

The configuration form will include:

1. **Entity Type Selector**: A select dropdown populated with all available entity types
   - Dynamically loaded from `EntityTypeManager::getDefinitions()`
   - Filtered to show only content entity types (not config entities)
   - Display labels (e.g., "Content" for `node`, "User" for `user`)

2. **Bundle Selector**: An optional select dropdown for choosing a specific bundle
   - Uses `#states` API for conditional visibility based on selected entity type
   - Populated via AJAX callback when entity type changes
   - Shows "All bundles" as default option
   - Hidden for entity types without bundles (e.g., `user`)

3. **Form State Management**:
   - Default configuration: `['entity_type' => '', 'bundle' => '']`
   - AJAX callbacks for dynamic bundle population
   - Validation to ensure entity type is selected
   - Proper form submission handling to store configuration

### Component 3: Entity Query Logic

**Objective**: Build efficient entity queries that filter by type, bundle, access, and search term.

The query logic will:

1. **Base Query Construction**:
   ```php
   $query = $this->entityTypeManager
     ->getStorage($entity_type)
     ->getQuery()
     ->accessCheck(TRUE)
     ->range(0, 10);
   ```

2. **Bundle Filtering** (if configured):
   ```php
   if (!empty($bundle)) {
     $query->condition($bundle_key, $bundle);
   }
   ```

3. **Search Term Filtering** (label or ID):
   - For label search: Use the entity type's label key (e.g., `title` for nodes, `name` for users)
   - For ID search: Use the entity ID key
   - Combine with OR condition group to support both
   - Use `CONTAINS` operator for partial matching on labels

4. **Access Context Switching**:
   - Use `AccountSwitcher` to temporarily switch to anonymous user
   - Ensures consistent access checking regardless of current user
   - Properly restore original user context after query

### Component 4: URL Formatting with Alias Resolution

**Objective**: Convert entity objects to absolute URLs with path aliases resolved.

Implementation strategy:

1. **Load Entities**: Use query results to load full entity objects
2. **Generate URLs**: For each entity:
   ```php
   $url = $entity->toUrl('canonical', ['absolute' => TRUE])->toString();
   ```
3. **Alias Resolution**: The `toUrl()` method automatically uses `PathAliasManager` to resolve aliases
4. **Fallback Handling**: Entities without canonical URLs (rare) return empty string and are filtered out
5. **Result Formatting**: Return array of URL strings ready for autocomplete

```mermaid
flowchart LR
    A[Entity IDs] --> B[Load Entities]
    B --> C[Check toUrl Support]
    C --> D[Generate Absolute URL]
    D --> E[Path Alias Resolved]
    E --> F[Return URL String]
```

### Component 5: Dependency Injection

**Objective**: Properly inject required Drupal services following best practices.

The plugin will use:
- Constructor property promotion with `readonly` properties
- `create()` factory method for service container integration
- Type hints for all injected services
- Services needed:
  - `EntityTypeManagerInterface $entityTypeManager`
  - `EntityTypeBundleInfoInterface $bundleInfo`
  - `AccountSwitcherInterface $accountSwitcher`

## Risk Considerations and Mitigation Strategies

### Technical Risks

- **Performance with Large Entity Sets**: Querying thousands of entities could be slow
  - **Mitigation**: Hard limit to 10 results, indexed fields for searching, efficient query conditions

- **Entity Types Without Canonical URLs**: Some entities may not have canonical routes
  - **Mitigation**: Check for `toUrl()` support, gracefully skip entities without URLs, filter empty results

- **Path Alias Resolution Edge Cases**: Some URLs might not have aliases or have complex patterns
  - **Mitigation**: Let Drupal's URL generation handle all cases automatically, test with various entity types

- **Access Control Complexity**: Different entity types have different access logic
  - **Mitigation**: Use Drupal's built-in `accessCheck(TRUE)` on queries and anonymous user context for consistency

### Implementation Risks

- **AJAX Form Callbacks**: Dynamic bundle selector requires proper AJAX implementation
  - **Mitigation**: Follow Drupal Form API patterns, use `#ajax` callbacks with proper wrapper elements

- **Bundle Key Variations**: Different entity types use different bundle key names
  - **Mitigation**: Use `EntityType::getBundleEntityType()` and `EntityType::getKey('bundle')` for dynamic discovery

- **Entity Type Without Bundles**: User, File, and some other entities don't have bundles
  - **Mitigation**: Conditional logic to hide bundle selector for applicable entity types

### Integration Risks

- **Compatibility with Existing Providers**: Must work within existing plugin architecture
  - **Mitigation**: Follow exact same pattern as `StaticListCompletionProvider`, use same base class and interfaces

## Success Criteria

### Primary Success Criteria

1. **Functional Plugin**: Entity query completion provider successfully registered and available in plugin manager
2. **Configuration UI**: Form allows selecting entity type and optional bundle with proper AJAX updates
3. **Accurate Results**: Autocomplete returns correct absolute URLs with path aliases for matching entities
4. **Access Control**: Only returns entities viewable by anonymous users
5. **Search Functionality**: Matches entities by both label (partial match) and exact ID

### Quality Assurance Metrics

1. **Code Quality**: Passes PHPStan level 1 analysis and PHPCS Drupal coding standards
2. **Type Safety**: Full type declarations, strict types enabled, no mixed types
3. **Performance**: Query execution under 500ms for typical datasets
4. **Test Coverage**: Unit tests for query building logic, functional tests for end-to-end autocomplete flow
5. **Documentation**: Proper docblocks explaining entity query logic and URL generation

## Resource Requirements

### Development Skills

- **Drupal Plugin System**: Understanding of plugin architecture, annotations/attributes, plugin managers
- **Entity API**: Knowledge of entity queries, entity type manager, bundle info, URL generation
- **Form API**: Experience with AJAX forms, conditional fields, form state management
- **Dependency Injection**: Familiarity with service container, factory methods, constructor injection
- **PHP 8.3 Features**: Constructor property promotion, readonly properties, attributes

### Technical Infrastructure

- **Drupal Core Services**:
  - `entity_type.manager`
  - `entity_type.bundle.info`
  - `account_switcher`
  - URL generator (via entity `toUrl()` method)
- **Testing Framework**: PHPUnit with Drupal test traits for functional testing
- **Code Quality Tools**: PHPStan, PHPCS with Drupal standards

## Implementation Order

This implementation follows a logical progression:

1. **Plugin Class Structure**: Create the plugin file with basic structure, attributes, and dependency injection
2. **Configuration Form**: Implement the form with entity type selector and dynamic bundle field
3. **Entity Query Logic**: Build the query construction and filtering logic
4. **URL Generation**: Implement entity loading and URL formatting with alias resolution
5. **Testing**: Create tests to validate functionality and edge cases
6. **Documentation**: Add inline documentation and update any relevant module documentation

## Notes

- This plugin will be the second completion provider in the module, establishing patterns for future data-driven providers
- The anonymous user access context ensures safe, consistent results regardless of who triggers the autocomplete
- The 10-result limit balances usability (enough options) with performance (fast responses)
- Optional bundle selection provides flexibility while still allowing precise filtering when needed
- URL output format makes results immediately useful for external tools, APIs, and AI agents

## Task Dependencies

```mermaid
graph TD
    001[Task 001: Create Entity Query Completion Provider Plugin] --> 002[Task 002: Add Integration Tests]
```

## Execution Blueprint

**Validation Gates:**
- Reference: `.ai/task-manager/config/hooks/POST_PHASE.md`

### Phase 1: Core Implementation
**Parallel Tasks:**
- Task 001: Create Entity Query Completion Provider Plugin

**Phase Objective**: Implement the core plugin functionality with all features (configuration form, entity querying, URL generation with path alias resolution).

**Dependencies**: None (first phase)

### Phase 2: Quality Assurance
**Parallel Tasks:**
- Task 002: Add Integration Tests (depends on: 001)

**Phase Objective**: Validate the implementation with comprehensive integration tests covering all critical functionality and edge cases.

**Dependencies**: Task 001 must be completed

### Post-phase Actions

After each phase completion:
1. Run PHPCS and PHPStan to ensure code quality
2. Execute functional tests to verify no regressions
3. Clear Drupal cache to ensure plugin discovery
4. Validate against phase-specific criteria defined in POST_PHASE.md

### Execution Summary
- Total Phases: 2
- Total Tasks: 2
- Maximum Parallelism: 1 task per phase
- Critical Path Length: 2 phases
- Estimated Complexity:
  - Task 001: 6.0/10 (Moderate complexity with multiple Drupal APIs)
  - Task 002: 4.0/10 (Standard integration testing)

## Execution Summary

**Status**: ✅ Completed Successfully
**Completed Date**: 2025-11-18

### Results

Successfully implemented the `EntityQueryCompletionProvider` plugin for the MCP Server module, enabling entity-based autocomplete with URL output. Key deliverables:

1. **Plugin Implementation** (`src/Plugin/PromptArgumentCompletionProvider/EntityQueryCompletionProvider.php`):
   - Configurable entity type and bundle selection via AJAX-enabled form
   - Entity queries filtered by type, bundle, access (anonymous user), and search term (label/ID)
   - Returns absolute URLs with path aliases automatically resolved
   - Results limited to 10 items for performance
   - Passes PHPCS and PHPStan quality checks

2. **Integration Tests** (added to `tests/src/Functional/McpServerFunctionalTest.php`):
   - 7 test scenarios covering all critical functionality
   - 26 assertions validating entity filtering, search, access control, and URL generation
   - All tests pass successfully

3. **Bug Fixes Discovered During Testing**:
   - Fixed `PromptArgumentCompletionProviderManager::createInstance()` to support dependency injection
   - Fixed access control timing in entity query (must switch to anonymous user before query execution)
   - Added additional entity-level access check for defense-in-depth

### Noteworthy Events

**Access Control Discovery**: During test implementation, discovered that account switching must occur BEFORE building the entity query, not just before execution. The initial implementation switched context after query construction, which caused access checks to use the wrong user context. This was identified through failing test cases and corrected to ensure proper anonymous user access control.

**Plugin Manager Enhancement**: The existing `PromptArgumentCompletionProviderManager` did not implement the `createInstance()` method required for proper dependency injection via the plugin's `create()` factory method. This was added to support the new plugin's service dependencies.

**Path Alias Integration**: Drupal's URL generation system (`Entity::toUrl()`) automatically handles path alias resolution without requiring explicit `PathAliasManager` injection, simplifying the implementation while maintaining full functionality.

### Recommendations

1. **Future Enhancement**: Consider adding a configuration option to allow site administrators to choose between anonymous user access and current user access for the entity query, providing more flexibility for different use cases.

2. **Performance Monitoring**: For sites with very large entity counts, consider adding query result caching with appropriate cache tags to improve performance for frequently used autocomplete configurations.

3. **Documentation**: Add a user guide demonstrating how to configure the entity query completion provider for common scenarios (e.g., node autocomplete, taxonomy term autocomplete, media autocomplete).

4. **Additional Providers**: The pattern established by this plugin can be extended to create other dynamic completion providers (e.g., view results, custom database queries, external API calls).
