---
id: 1
group: "entity-autocomplete"
dependencies: []
status: "completed"
created: 2025-11-18
skills:
  - drupal-plugin-development
  - php-dependency-injection
---
# Create Entity Query Completion Provider Plugin

## Objective

Implement the core `EntityQueryCompletionProvider` plugin class with dependency injection, configuration form, entity querying logic, and URL generation with path alias resolution.

## Skills Required

- **Drupal Plugin Development**: Plugin system architecture, attributes, plugin managers, base classes
- **PHP Dependency Injection**: Service container, factory methods, constructor promotion

## Acceptance Criteria

- [ ] Plugin class created at `src/Plugin/PromptArgumentCompletionProvider/EntityQueryCompletionProvider.php`
- [ ] Plugin uses `#[PromptArgumentCompletionProvider]` attribute with id `entity_query`
- [ ] Dependency injection implemented with all required services
- [ ] Configuration form with entity type dropdown and conditional bundle selector
- [ ] Entity query logic filters by type, bundle, access (anonymous), and search term (label/ID)
- [ ] Results return absolute URLs with path aliases resolved
- [ ] Results limited to 10 items
- [ ] Passes PHPCS and PHPStan checks

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

### Services to Inject

```php
- EntityTypeManagerInterface $entityTypeManager
- EntityTypeBundleInfoInterface $bundleInfo
- AccountSwitcherInterface $accountSwitcher
- AccountInterface $anonymousUser (tagged service: @account.anonymous)
```

### Plugin Attribute

```php
#[PromptArgumentCompletionProvider(
  id: 'entity_query',
  label: new TranslatableMarkup('Entity query'),
  description: new TranslatableMarkup('Provides entity autocomplete with URL output.'),
)]
```

### Configuration Schema

```php
defaultConfiguration() {
  return [
    'entity_type' => '',
    'bundle' => '',
  ];
}
```

### Form Elements

1. **Entity Type Selector**:
   - Type: `select`
   - Options: All content entity types (use `$entityTypeManager->getDefinitions()` filtered by `ContentEntityTypeInterface`)
   - Required: TRUE

2. **Bundle Selector**:
   - Type: `select`
   - Options: Populated via AJAX based on entity type selection
   - States: Visible only when entity type is selected
   - Default: Empty (all bundles)

### Entity Query Logic

```php
public function getCompletions(string $current_value, array $configuration): array {
  $entity_type = $configuration['entity_type'] ?? '';
  $bundle = $configuration['bundle'] ?? '';

  // Build base query
  $query = $this->entityTypeManager
    ->getStorage($entity_type)
    ->getQuery()
    ->accessCheck(TRUE)
    ->range(0, 10);

  // Add bundle condition if specified
  if ($bundle) {
    $bundle_key = $this->entityTypeManager
      ->getDefinition($entity_type)
      ->getKey('bundle');
    if ($bundle_key) {
      $query->condition($bundle_key, $bundle);
    }
  }

  // Add search filter (label OR ID)
  if ($current_value !== '') {
    $label_key = $this->entityTypeManager
      ->getDefinition($entity_type)
      ->getKey('label');

    $or_group = $query->orConditionGroup();

    if ($label_key) {
      $or_group->condition($label_key, $current_value, 'CONTAINS');
    }

    // Check if current value is numeric for ID search
    if (is_numeric($current_value)) {
      $id_key = $this->entityTypeManager
        ->getDefinition($entity_type)
        ->getKey('id');
      $or_group->condition($id_key, (int) $current_value);
    }

    $query->condition($or_group);
  }

  // Execute with anonymous user context
  $this->accountSwitcher->switchTo($this->anonymousUser);
  $entity_ids = $query->execute();
  $this->accountSwitcher->switchBack();

  // Load entities and generate URLs
  $entities = $this->entityTypeManager
    ->getStorage($entity_type)
    ->loadMultiple($entity_ids);

  $urls = [];
  foreach ($entities as $entity) {
    if ($entity->hasLinkTemplate('canonical')) {
      $urls[] = $entity->toUrl('canonical', ['absolute' => TRUE])->toString();
    }
  }

  return $urls;
}
```

### URL Generation

- Use `$entity->toUrl('canonical', ['absolute' => TRUE])->toString()`
- Path aliases are automatically resolved by Drupal's URL generation
- Filter out entities without canonical link template

## Input Dependencies

None (first task in the plan).

## Output Artifacts

- `src/Plugin/PromptArgumentCompletionProvider/EntityQueryCompletionProvider.php` - Fully functional plugin class

## Implementation Notes

<details>
<summary>Detailed Implementation Guidance</summary>

### File Structure

Create the file at: `src/Plugin/PromptArgumentCompletionProvider/EntityQueryCompletionProvider.php`

### Class Declaration

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin\PromptArgumentCompletionProvider;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountSwitcherInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mcp_server\Attribute\PromptArgumentCompletionProvider;
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

#[PromptArgumentCompletionProvider(
  id: 'entity_query',
  label: new TranslatableMarkup('Entity query'),
  description: new TranslatableMarkup('Provides entity autocomplete based on entity queries. Returns absolute URLs with path aliases resolved.'),
)]
final class EntityQueryCompletionProvider extends PromptArgumentCompletionProviderBase {

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityTypeBundleInfoInterface $bundleInfo,
    private readonly AccountSwitcherInterface $accountSwitcher,
    private readonly AccountInterface $anonymousUser,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('account_switcher'),
      new \Drupal\Core\Session\AnonymousUserSession(),
    );
  }

  // ... implement methods
}
```

### AJAX Callback for Bundle Selector

```php
public function updateBundleOptions(array &$form, FormStateInterface $form_state) {
  return $form['bundle'];
}
```

### Form Building with AJAX

```php
public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
  // Entity type selector
  $entity_type_options = [];
  foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
    if ($entity_type instanceof \Drupal\Core\Entity\ContentEntityTypeInterface) {
      $entity_type_options[$entity_type_id] = $entity_type->getLabel();
    }
  }

  $form['entity_type'] = [
    '#type' => 'select',
    '#title' => $this->t('Entity type'),
    '#options' => $entity_type_options,
    '#default_value' => $this->configuration['entity_type'] ?? '',
    '#required' => TRUE,
    '#ajax' => [
      'callback' => [$this, 'updateBundleOptions'],
      'wrapper' => 'bundle-wrapper',
    ],
  ];

  // Bundle selector (conditional via AJAX)
  $form['bundle'] = [
    '#type' => 'select',
    '#title' => $this->t('Bundle (optional)'),
    '#options' => $this->getBundleOptions($form_state),
    '#default_value' => $this->configuration['bundle'] ?? '',
    '#prefix' => '<div id="bundle-wrapper">',
    '#suffix' => '</div>',
    '#states' => [
      'visible' => [
        ':input[name="entity_type"]' => ['!value' => ''],
      ],
    ],
  ];

  return $form;
}

private function getBundleOptions(FormStateInterface $form_state): array {
  $entity_type = $form_state->getValue('entity_type') ?? $this->configuration['entity_type'] ?? '';

  if (empty($entity_type)) {
    return ['' => $this->t('- Select entity type first -')];
  }

  $bundles = $this->bundleInfo->getBundleInfo($entity_type);
  $options = ['' => $this->t('- All bundles -')];

  foreach ($bundles as $bundle_id => $bundle_info) {
    $options[$bundle_id] = $bundle_info['label'];
  }

  return $options;
}
```

### Edge Cases to Handle

1. **Entity types without bundles** (e.g., `user`, `file`): Bundle selector should show "All bundles" option
2. **Entities without canonical URLs**: Skip them using `hasLinkTemplate('canonical')` check
3. **Empty search term**: Return first 10 entities of the type/bundle
4. **No matching entities**: Return empty array
5. **Invalid entity type in configuration**: Handle gracefully in validation

### Performance Notes

- The 10-result limit prevents expensive queries
- `accessCheck(TRUE)` uses entity access handlers efficiently
- Anonymous user context ensures consistent, safe results
- AJAX callbacks only rebuild the bundle field, not the entire form

### Testing Considerations

- Test with various entity types: node, user, taxonomy_term, media
- Test bundle filtering with nodes (article vs page)
- Test search by label (partial match)
- Test search by ID (exact match)
- Test URL generation with path aliases
- Test access control (entities not viewable by anonymous should not appear)

</details>
