---
id: 2
group: "entity-autocomplete"
dependencies: [1]
status: "completed"
created: 2025-11-18
skills:
  - drupal-testing
---
# Add Integration Tests for Entity Query Completion Provider

## Objective

Create integration tests within the existing functional test to validate the `EntityQueryCompletionProvider` plugin functionality, including entity querying, URL generation, path alias resolution, and access control.

## Skills Required

- **Drupal Testing**: Functional tests, test fixtures, entity creation, assertions

## Acceptance Criteria

- [ ] Tests added to existing `McpServerFunctionalTest` class
- [ ] Test entities created with path aliases
- [ ] Test validates entity type and bundle filtering
- [ ] Test validates search by label (partial match)
- [ ] Test validates search by ID (exact match)
- [ ] Test validates URL generation with path alias resolution
- [ ] Test validates access control (anonymous user context)
- [ ] Test validates 10-result limit
- [ ] All tests pass when run with `vendor/bin/phpunit`

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

### Meaningful Test Strategy Guidelines

Your critical mantra for test generation is: "write a few tests, mostly integration".

**When TO Write Tests:**
- Custom business logic and algorithms
- Critical user workflows and data transformations
- Edge cases and error conditions for core functionality
- Integration points between different system components
- Complex validation logic or calculations

**When NOT to Write Tests:**
- Third-party library functionality (already tested upstream)
- Framework features (React hooks, Express middleware, etc.)
- Simple CRUD operations without custom logic
- Getter/setter methods or basic property access
- Configuration files or static data
- Obvious functionality that would break immediately if incorrect

### Test Location

Add test methods to the existing functional test class:
`tests/src/Functional/McpServerFunctionalTest.php`

### Test Setup Requirements

```php
// Create test entities in setUp() method
protected function setUp(): void {
  parent::setUp();

  // Create content type for testing
  $this->drupalCreateContentType(['type' => 'article', 'name' => 'Article']);

  // Create test nodes with path aliases
  $node1 = $this->drupalCreateNode([
    'type' => 'article',
    'title' => 'Test Article One',
    'status' => 1,
  ]);
  $this->createPathAlias('/node/' . $node1->id(), '/blog/test-article-one');

  $node2 = $this->drupalCreateNode([
    'type' => 'article',
    'title' => 'Test Article Two',
    'status' => 1,
  ]);
  $this->createPathAlias('/node/' . $node2->id(), '/blog/test-article-two');

  // Create unpublished node (should not appear in anonymous results)
  $this->drupalCreateNode([
    'type' => 'article',
    'title' => 'Unpublished Article',
    'status' => 0,
  ]);
}
```

### Test Scenarios

#### Test 1: Entity Type Filtering

```php
public function testEntityQueryCompletionProviderEntityTypeFiltering() {
  // Create provider instance with node configuration
  // Query for completions
  // Assert only node URLs returned
  // Assert URLs use path aliases
}
```

#### Test 2: Bundle Filtering

```php
public function testEntityQueryCompletionProviderBundleFiltering() {
  // Create provider with bundle='article'
  // Create page node for comparison
  // Query for completions
  // Assert only article nodes returned
}
```

#### Test 3: Search by Label

```php
public function testEntityQueryCompletionProviderSearchByLabel() {
  // Query with partial label match "Test Article"
  // Assert both matching nodes returned
  // Query with "One"
  // Assert only first node returned
}
```

#### Test 4: Search by ID

```php
public function testEntityQueryCompletionProviderSearchById() {
  // Query with node ID
  // Assert correct node URL returned
}
```

#### Test 5: Access Control

```php
public function testEntityQueryCompletionProviderAccessControl() {
  // Query for completions (should use anonymous user)
  // Assert unpublished node not in results
  // Assert only published nodes returned
}
```

#### Test 6: Result Limit

```php
public function testEntityQueryCompletionProviderResultLimit() {
  // Create 15 test nodes
  // Query for completions with empty search
  // Assert exactly 10 results returned
}
```

#### Test 7: Path Alias Resolution

```php
public function testEntityQueryCompletionProviderPathAliasResolution() {
  // Query for completion
  // Assert URL contains alias path (/blog/test-article-one)
  // Assert URL does not contain node path (/node/123)
  // Assert URL is absolute (starts with base URL)
}
```

### Helper Methods

```php
/**
 * Creates a path alias.
 */
protected function createPathAlias(string $source, string $alias): void {
  $path_alias_storage = \Drupal::entityTypeManager()
    ->getStorage('path_alias');
  $path_alias_storage->create([
    'path' => $source,
    'alias' => $alias,
    'langcode' => 'en',
  ])->save();
}

/**
 * Creates an instance of EntityQueryCompletionProvider.
 */
protected function createEntityQueryProvider(array $config): PromptArgumentCompletionProviderInterface {
  $plugin_manager = \Drupal::service('plugin.manager.prompt_argument_completion_provider');
  return $plugin_manager->createInstance('entity_query', $config);
}
```

## Input Dependencies

- Task 1: `EntityQueryCompletionProvider` plugin class must be implemented

## Output Artifacts

- Updated `tests/src/Functional/McpServerFunctionalTest.php` with additional test methods
- Test coverage for all critical functionality

## Implementation Notes

<details>
<summary>Detailed Testing Guidance</summary>

### Test Organization

Since the module only allows ONE Functional test with ONE test method, you need to integrate these test scenarios as helper methods within the single test function:

```php
public function testMcpServerFunctionality() {
  // Existing tests...

  // Add new entity query provider tests
  $this->doTestEntityQueryProviderBasicFunctionality();
  $this->doTestEntityQueryProviderAccessControl();
  $this->doTestEntityQueryProviderPathAliases();
  // etc.
}
```

### Assertions to Use

```php
// URL format assertions
$this->assertStringContainsString('http', $url, 'URL is absolute');
$this->assertStringContainsString('/blog/test-article-one', $url, 'Path alias resolved');
$this->assertStringNotContainsString('/node/', $url, 'Node path not used');

// Result count assertions
$this->assertCount(10, $results, 'Results limited to 10');
$this->assertNotEmpty($results, 'Results returned');

// Content assertions
$this->assertContains($expected_url, $results, 'Expected URL in results');
```

### Edge Cases to Test

1. **Empty entity type**: Should not crash, should return empty array
2. **Invalid entity type**: Should handle gracefully
3. **Entity type without bundles**: Should work correctly
4. **Empty search term**: Should return first 10 entities
5. **Search with no matches**: Should return empty array
6. **Entities without canonical URLs**: Should be filtered out

### Performance Testing

While not required, consider adding a performance assertion:

```php
$start = microtime(TRUE);
$results = $provider->getCompletions($search_term, $config);
$duration = microtime(TRUE) - $start;
$this->assertLessThan(0.5, $duration, 'Query completes within 500ms');
```

### Test Data Reuse

Store test node IDs in class properties for reuse across helper methods:

```php
private array $testNodeIds = [];

protected function setUp(): void {
  parent::setUp();
  $node = $this->drupalCreateNode([...]);
  $this->testNodeIds[] = $node->id();
}
```

### Cleanup

Drupal's test framework handles entity cleanup automatically, but if you create custom services or modify global state, ensure proper teardown.

</details>
