---
id: 13
summary: "Implement MCP resource template plugin system for exposing Drupal content entities with canonical URLs via JSON:API serialization"
created: 2025-11-19
---

# Plan: Content Entity Resource Templates for MCP Server

## Original Work Order

> I want you to take a look at the @.ai/task-manager/scratch/mcp-resource-ideas.md and start implementing resource templates for content entity types. You should only provide resources for entity types that have a URL template for canonical. The way to implement these resource templates would be using a plugin. A new plugin type will need to be defined in order to have different types of resources. So the first plugin for resource templates will be for content entities. As you can see in the document for @.ai/task-manager/scratch/mcp-resource-ideas.md, each plugin will define a custom schema. In this case, we'll use the custom schema `drupal://entity/...`. Note that listing and accessing revisions is out of the scope for now. Right now, the resource will only be accessing a content entity that has a canonical URL template using its ID. The URI should be `drupal://entity/<entity_type>/{entity_id}`. Ex: `drupal://entity/node/{entity_id}`, `drupal://entity/media/{entity_id}`, ...

## Plan Clarifications

| Question | Answer |
|----------|--------|
| Should the resource plugin system support resource listing? | Individual entity access only (drupal://entity/node/{nid}) |
| Should resource access respect Drupal's entity access control? | Enforce Drupal entity access (check entity->access('view')) |
| Should resources return raw entity data, rendered output, or both? | Return JSON:API serialization of the entity |
| Should there be a configuration UI for enabling resources? | Yes, with fields for resource type, title, and dependencies (JSON:API listed as dependency for Content Entity Resource Template) |
| Should tests be written? | Yes, include comprehensive tests following project testing guidelines |

## Executive Summary

This plan implements a pluggable MCP resource system for Drupal, starting with content entity resources. The implementation creates a new plugin type (`ResourceTemplateInterface`) that allows different types of MCP resources to be discovered and exposed through the MCP server. The first concrete implementation will be a Content Entity Resource Template plugin that provides access to individual Drupal entities (nodes, media, taxonomy terms, etc.) that have canonical URL templates.

Resources will use the URI scheme `drupal://entity/{entity_type}/{entity_id}` (e.g., `drupal://entity/node/123`) and return JSON:API-serialized entity data. This approach leverages Drupal's robust JSON:API serialization infrastructure for consistent, standard data formatting while respecting entity access controls per the security framework outlined in the MCP resource ideas document.

A configuration entity system will allow site administrators to enable/disable specific resource types, with each resource displaying its dependencies (such as the JSON:API module requirement for content entity resources). This follows the established pattern in the MCP server module where tools and prompts are managed through configuration entities.

## Context

### Current State

The MCP server module currently provides:
- Tool integration through `McpBridgeService` with `McpToolConfig` configuration entities
- Prompt management through `McpPromptConfig` configuration entities
- OAuth scope validation and authentication
- Discovery and execution patterns for MCP capabilities

However, there is no system for exposing Drupal's content as MCP resources. The MCP specification defines resources as a way to provide contextual data (files, database schemas, application-specific information) to AI assistants, but this capability is not yet implemented in the module.

The `.ai/task-manager/scratch/mcp-resource-ideas.md` document outlines 100+ potential MCP resources for Drupal, organized by security classification levels (PUBLIC, RESTRICTED, PRIVILEGED, SENSITIVE, DANGEROUS). Content entities are identified as high-priority resources with varying security levels depending on publication status.

### Target State

After implementation, the module will have:

1. **Plugin System**: A new `ResourceTemplate` plugin type that allows different resource providers to be discovered and registered with the MCP server
2. **Content Entity Plugin**: A concrete plugin implementation that exposes content entities with canonical URLs through the `drupal://entity/{entity_type}/{entity_id}` URI scheme
3. **Configuration UI**: An admin interface for enabling/disabling resource types with dependency tracking
4. **Access Control**: Integration with Drupal's entity access system to enforce view permissions
5. **JSON:API Serialization**: Resources return properly serialized entity data using Drupal's JSON:API module
6. **Test Coverage**: Unit and kernel tests validating plugin discovery, access control, and serialization

This enables AI assistants connected to the MCP server to discover and retrieve Drupal content entities (nodes, media, taxonomy terms, etc.) with proper permission checking and standard data formats.

### Background

The MCP (Model Context Protocol) specification defines three main capability types: tools, prompts, and resources. While tools and prompts are implemented, resources remain unaddressed. Resources are particularly valuable for providing site context to AI assistants.

According to the resource ideas document, content entities should be exposed with:
- **PUBLIC** security level for published entities (requires only `access content` permission)
- **RESTRICTED** level for unpublished content (requires specific view permissions)
- Mandatory entity access checks to prevent information disclosure
- Field-level filtering for sensitive data (though JSON:API handles this through its field access system)

The decision to use JSON:API serialization (rather than raw entity arrays or rendered HTML) provides several benefits:
- **Standardized format**: JSON:API is an established specification (https://jsonapi.org/)
- **Built-in access control**: JSON:API respects field-level access
- **Relationship handling**: JSON:API properly serializes entity references and relationships
- **Consistency**: Same format used by Drupal's decoupled/headless capabilities

## Technical Implementation Approach

```mermaid
graph TB
    A[MCP Server] -->|discovers| B[ResourceTemplate Plugins]
    B -->|registers| C[Content Entity Resource Template]
    C -->|queries| D[Entity Type Manager]
    D -->|returns| E[Content Entities with Canonical URLs]
    C -->|serializes via| F[JSON:API Serializer]
    G[McpResourceConfig Entity] -->|enables/disables| C
    H[Admin UI] -->|manages| G
    I[Current User] -->|checks access| J[Entity Access System]
    C -->|enforces| J

    style C fill:#90EE90
    style F fill:#FFD700
    style J fill:#FF6B6B
```

### Component 1: Resource Template Plugin System

**Objective**: Create a plugin architecture that allows different types of MCP resources to be discovered, configured, and registered with the MCP server.

**Implementation Details**:

1. **Plugin Interface** (`src/Plugin/ResourceTemplateInterface.php`):
   ```php
   interface ResourceTemplateInterface extends PluginInspectionInterface {
     public function getResourceType(): string;
     public function getTitle(): TranslatableMarkup;
     public function getDescription(): ?TranslatableMarkup;
     public function getDependencies(): array;
     public function getUriTemplate(): string;
     public function getResources(): array;
     public function getResourceContent(string $uri): ?array;
     public function checkAccess(string $uri, AccountInterface $account): AccessResultInterface;
   }
   ```

2. **Plugin Base** (`src/Plugin/ResourceTemplateBase.php`):
   - Abstract base class implementing common functionality
   - Helper methods for URI parsing and validation
   - Default access control returning AccessResult::forbidden()

3. **Plugin Manager** (`src/Plugin/ResourceTemplateManager.php`):
   - Extends `DefaultPluginManager`
   - Discovers plugins via annotations or PHP attributes
   - Provides methods to get all available resource types
   - Caches discovered plugins

4. **Plugin Attribute** (`src/Attribute/ResourceTemplate.php`):
   ```php
   #[Attribute(Attribute::TARGET_CLASS)]
   class ResourceTemplate {
     public function __construct(
       public readonly string $id,
       public readonly TranslatableMarkup $label,
       public readonly TranslatableMarkup $description,
       public readonly array $dependencies = [],
     ) {}
   }
   ```

Following the established pattern from the Tool plugin system (which uses the Tool module's plugin infrastructure), but creating a custom plugin type specific to MCP resources.

### Component 2: Content Entity Resource Template Plugin

**Objective**: Implement the first concrete resource template plugin that exposes Drupal content entities with canonical URL templates via the `drupal://entity/{entity_type}/{entity_id}` URI scheme.

**Implementation Details**:

1. **Plugin Class** (`src/Plugin/ResourceTemplate/ContentEntityResourceTemplate.php`):
   ```php
   #[ResourceTemplate(
     id: 'content_entity',
     label: new TranslatableMarkup('Content Entity Resources'),
     description: new TranslatableMarkup('Provides access to content entities with canonical URLs'),
     dependencies: ['jsonapi'],
   )]
   final class ContentEntityResourceTemplate extends ResourceTemplateBase {
     // Implementation
   }
   ```

2. **Entity Discovery**:
   - Query `EntityTypeManagerInterface` for all entity types
   - Filter for content entities using `EntityTypeInterface::isSubclassOf(ContentEntityInterface::class)`
   - Further filter for entities with `canonical` link template
   - Examples: node, media, taxonomy_term, user, block_content

3. **Resource Generation**:
   - For each eligible entity type, generate resource metadata:
     ```php
     [
       'uri' => 'drupal://entity/node/{nid}',
       'name' => 'Node - {nid}',
       'description' => 'A content node entity',
       'mimeType' => 'application/vnd.api+json',
     ]
     ```
   - Use entity type labels and descriptions for human-readable metadata

4. **URI Handling**:
   - Parse URI format: `drupal://entity/{entity_type}/{entity_id}`
   - Extract entity type and ID from URI
   - Validate entity type exists and has canonical URL
   - Load entity using `EntityTypeManagerInterface::getStorage($entity_type)->load($id)`

5. **Access Control**:
   - Check if entity exists (return `AccessResult::forbidden()` if not)
   - Call `$entity->access('view', $account, TRUE)` to get access result
   - Return the access result (preserves cacheability metadata)
   - This enforces Drupal's entity access system including:
     - Publication status (published vs unpublished)
     - Entity-level access hooks
     - Custom access control modules

6. **JSON:API Serialization**:
   - Inject `jsonapi.serializer` service (requires JSON:API module)
   - Use `Serializer::normalize($entity, 'api_json')` to get JSON:API format
   - Return normalized array with:
     - Entity attributes (fields)
     - Relationships (entity references)
     - Links (including canonical URL)
     - Meta information
   - Handle serialization exceptions gracefully

### Component 3: Resource Configuration Entity

**Objective**: Create a configuration entity type that allows administrators to enable/disable resource types and view their dependencies.

**Implementation Details**:

1. **Configuration Entity** (`src/Entity/McpResourceConfig.php`):
   ```php
   #[ConfigEntityType(
     id: 'mcp_resource_config',
     label: new TranslatableMarkup('MCP Resource Configuration'),
     config_prefix: 'resource',
     admin_permission: 'administer mcp server',
     entity_keys: [
       'id' => 'id',
       'label' => 'label',
       'status' => 'status',
     ],
     config_export: [
       'id',
       'label',
       'resource_template_id',
       'description',
       'dependencies',
     ],
   )]
   final class McpResourceConfig extends ConfigEntityBase {
     public readonly string $resource_template_id;
     public readonly ?string $description;
     public readonly array $dependencies;

     public function getResourceTemplateId(): string;
     public function getDescription(): ?string;
     public function getDependencies(): array;
   }
   ```

2. **Entity Properties**:
   - `id`: Machine name (e.g., `content_entity_node`)
   - `label`: Human-readable label (e.g., "Content Entity - Node")
   - `status`: Enabled/disabled boolean
   - `resource_template_id`: Plugin ID (e.g., `content_entity`)
   - `description`: Optional description from plugin
   - `dependencies`: Array of module dependencies (e.g., `['jsonapi']`)

3. **Integration with Plugin System**:
   - When a resource config is enabled, query the plugin for resources
   - Validate that required dependencies are met (modules enabled)
   - If dependencies missing, prevent enabling or show warning

### Component 4: Resource Management UI

**Objective**: Provide an administrative interface for managing resource configurations with dependency tracking.

**Implementation Details**:

1. **List Builder** (`src/McpResourceConfigListBuilder.php`):
   - Extends `ConfigEntityListBuilder`
   - Displays table with columns:
     - Label
     - Resource Type (plugin ID)
     - Dependencies (with status indicators)
     - Enabled/Disabled status
     - Operations (Edit, Delete, Enable/Disable)
   - Show warning icon for resources with unmet dependencies
   - Use same pattern as `McpToolConfigListBuilder`

2. **Configuration Form** (`src/Form/McpResourceConfigForm.php`):
   - Extends `EntityForm`
   - Fields:
     - Label (textfield, required)
     - Resource Template (select, from available plugins, required)
     - Description (textarea, read-only, populated from plugin)
     - Dependencies (item list, read-only, populated from plugin)
     - Status (checkbox, "Enabled")
   - Validate dependencies on save
   - Show error if required modules not enabled
   - Use `PluginSelectionFormTrait` for plugin selection

3. **Admin Routes** (`mcp_server.routing.yml`):
   ```yaml
   entity.mcp_resource_config.collection:
     path: '/admin/config/services/mcp-server/resources'
     defaults:
       _entity_list: 'mcp_resource_config'
       _title: 'MCP Resource Templates'
     requirements:
       _permission: 'administer mcp server'

   entity.mcp_resource_config.add_form:
     path: '/admin/config/services/mcp-server/resources/add'
     defaults:
       _entity_form: 'mcp_resource_config.add'
       _title: 'Add Resource Configuration'
     requirements:
       _entity_create_access: 'mcp_resource_config'
   ```

4. **Menu Links** (`mcp_server.links.menu.yml`):
   - Add "Resources" tab under MCP Server configuration
   - Place alongside existing "Tools" and "Prompts" tabs

### Component 5: Integration with MCP Server

**Objective**: Register enabled resources with the MCP server instance so they can be discovered and accessed by clients.

**Implementation Details**:

1. **Resource Bridge Service** (`src/McpResourceBridgeService.php`):
   - Similar to `McpBridgeService` for tools/prompts
   - Methods:
     - `getEnabledResources(): array` - Returns all enabled resource metadata
     - `getResourceContent(string $uri): ?array` - Fetches content for specific URI
     - `checkResourceAccess(string $uri, AccountInterface $account): AccessResultInterface`
   - Loads enabled `McpResourceConfig` entities
   - Queries respective plugins for resource data
   - Caches results with appropriate cache tags

2. **Server Factory Integration** (`src/McpServerFactory.php`):
   - In `create()` method, after registering tools and prompts:
     ```php
     // Register resources
     $enabled_resources = $this->resourceBridge->getEnabledResources();
     foreach ($enabled_resources as $resource_data) {
       $builder->addResource(
         handler: fn($uri) => $this->resourceBridge->getResourceContent($uri),
         uri: $resource_data['uri'],
         name: $resource_data['name'],
         description: $resource_data['description'],
         mimeType: $resource_data['mimeType'],
       );
     }
     ```

3. **Capability Loader** (optional, if MCP SDK supports it):
   - Create `src/Capability/Loader/ResourceConfigLoader.php`
   - Implements loader interface from MCP SDK
   - Dynamically loads resources from configuration
   - Pattern similar to `PromptConfigLoader`

### Component 6: Testing Strategy

**Objective**: Ensure all components work correctly through comprehensive automated tests following project guidelines.

**Implementation Details**:

1. **Unit Tests**:
   - **ResourceTemplateManagerTest**: Plugin discovery and instantiation
   - **McpResourceConfigTest**: Configuration entity methods and validation
   - Test plugin attribute parsing
   - Test URI template parsing logic

2. **Kernel Tests**:
   - **ContentEntityResourceTemplateTest**:
     - Entity type discovery (finds node, media, taxonomy_term)
     - URI generation for different entity types
     - Entity loading and validation
     - Access control enforcement (published vs unpublished, anonymous vs authenticated)
     - JSON:API serialization (verify structure, fields, relationships)
   - **McpResourceBridgeServiceTest**:
     - Enabled resource retrieval
     - Resource content fetching
     - Access checking
     - Cache tag invalidation
   - **ResourceIntegrationTest**:
     - End-to-end resource access flow
     - Configuration entity CRUD operations
     - Dependency validation

3. **Test Data Setup**:
   - Create test content: published node, unpublished node, media entity
   - Create test users with different permissions
   - Enable/disable JSON:API module in tests
   - Use existing patterns from `McpToolConfig` tests

4. **Testing Constraints** (per project guidelines):
   - ONE kernel test class maximum for the feature
   - Use helper methods instead of multiple test methods
   - Test business logic, not framework features
   - Focus on module-specific behavior (plugin system, access control, serialization)

## Risk Considerations and Mitigation Strategies

### Technical Risks

- **JSON:API Dependency**: Content entity resources require JSON:API module
    - **Mitigation**: Clearly mark dependency in plugin metadata and configuration UI. Validate module is enabled before allowing resource configuration. Provide clear error messages if dependency is missing.

- **Serialization Performance**: JSON:API normalization can be expensive for entities with many fields/relationships
    - **Mitigation**: Leverage JSON:API's built-in caching. Consider pagination for future listing features. Document performance characteristics in admin UI.

- **Entity Access Complexity**: Drupal's entity access system has many layers (hooks, modules, custom access controllers)
    - **Mitigation**: Use `$entity->access('view', $account, TRUE)` to leverage all access checks. Return detailed access results with cacheability metadata. Test with common access control scenarios.

### Security Risks

- **Information Disclosure**: Exposing unpublished or restricted content to unauthorized users
    - **Mitigation**: Mandatory entity access checks on every resource request. Never bypass access control. Follow security classification from resource ideas document (PUBLIC for published, RESTRICTED for unpublished). Add audit logging for resource access.

- **Enumeration Attacks**: Attackers could iterate through entity IDs to discover content
    - **Mitigation**: Enforce access control on every request (even for non-existent entities, return same forbidden response). Consider UUIDs instead of sequential IDs in future iterations. Document this risk in admin UI.

- **Field-level PII Leakage**: Entity fields might contain sensitive user data
    - **Mitigation**: JSON:API respects field-level access through its normalization process. Document that site builders should properly configure field permissions. Test with fields containing sensitive data.

### Implementation Risks

- **Plugin Architecture Complexity**: New developers might find plugin system confusing
    - **Mitigation**: Provide clear documentation and code examples. Follow existing Drupal plugin patterns. Include comprehensive PHPDoc annotations.

- **Configuration Entity Overhead**: Managing resources through config entities adds administrative complexity
    - **Mitigation**: Provide sensible defaults. Consider auto-creating configs for common entity types on module install. Add bulk enable/disable operations.

- **Breaking Changes**: Future MCP SDK updates might change resource API
    - **Mitigation**: Abstract MCP SDK interaction behind services. Keep resource registration logic centralized in factory. Add comprehensive tests that will catch API changes.

### Integration Risks

- **MCP SDK Compatibility**: MCP SDK resource features might differ from expectations
    - **Mitigation**: Review current MCP SDK documentation thoroughly before implementation. Create thin adapter layer if needed. Test with actual MCP clients.

- **Existing Code Conflicts**: Integration with existing server factory might require refactoring
    - **Mitigation**: Follow established patterns from tool/prompt integration. Review existing code carefully. Ensure backward compatibility.

## Success Criteria

### Primary Success Criteria

1. **Plugin System Operational**: ResourceTemplate plugin type is discoverable, with ContentEntityResourceTemplate plugin successfully found and instantiated
2. **Entity Resources Accessible**: Content entities with canonical URLs (node, media, taxonomy_term) are accessible via `drupal://entity/{entity_type}/{entity_id}` URIs
3. **Access Control Enforced**: Anonymous users cannot access unpublished nodes, authenticated users with proper permissions can access published and unpublished content based on their permissions
4. **JSON:API Serialization Working**: Resources return properly formatted JSON:API responses with attributes, relationships, and links
5. **Configuration UI Functional**: Administrators can enable/disable resource types, view dependencies, and receive appropriate warnings for unmet dependencies
6. **MCP Server Integration**: Enabled resources are registered with MCP server and discoverable by MCP clients

### Quality Assurance Metrics

1. **Test Coverage**: All kernel tests pass, covering plugin discovery, access control, serialization, and configuration management
2. **Code Quality**: PHPStan analysis passes at level 1, PHPCS passes Drupal and DrupalPractice standards
3. **Security Validation**: All resource requests enforce entity access checks, no bypasses exist, security tests verify access denial for unauthorized users
4. **Documentation**: All public methods have PHPDoc, plugin attributes are complete, admin UI has help text explaining dependencies
5. **Performance**: Resource requests complete within reasonable time (<500ms for simple entities), JSON:API caching is utilized

## Resource Requirements

### Development Skills

- **Drupal Plugin API**: Understanding of plugin managers, discovery, attributes, and instantiation
- **Entity API**: Knowledge of entity type manager, content entities, access control, and link templates
- **JSON:API Module**: Familiarity with normalization, serialization, and JSON:API specification
- **Configuration Entities**: Experience with ConfigEntityBase, entity forms, and list builders
- **Drupal Testing**: Ability to write kernel tests with test content and user setup
- **MCP Protocol**: Understanding of MCP resource specification and URI templates

### Technical Infrastructure

- **JSON:API Module**: Required for entity serialization (already in Drupal core)
- **MCP SDK**: PHP MCP server SDK (already included via Composer)
- **Tool API Module**: For plugin pattern reference (already a dependency)
- **PHPUnit**: For automated testing (already configured)
- **PHPStan & PHPCS**: For code quality validation (already configured)

### External Dependencies

- **MCP SDK Resource APIs**: Rely on `$builder->addResource()` method or equivalent resource registration API
- **JSON:API Serializer Service**: Requires `jsonapi.serializer` service from JSON:API module
- **Entity Access System**: Depends on Drupal core's entity access control infrastructure

## Integration Strategy

The resource system integrates with existing MCP server components through established patterns:

1. **Service Layer**: `McpResourceBridgeService` follows same pattern as `McpBridgeService` for tools/prompts
2. **Configuration Entities**: `McpResourceConfig` mirrors `McpToolConfig` and `McpPromptConfig` structures
3. **Factory Integration**: Resources registered in `McpServerFactory::create()` alongside tools/prompts
4. **Admin UI**: Resource management UI placed in same admin section as tools/prompts configuration

This consistency ensures developers familiar with existing tool/prompt systems can easily understand and extend resource functionality.

## Implementation Order

High-level sequence of implementation (detailed task breakdown will follow in blueprint generation):

1. **Foundation**: Plugin system infrastructure (interface, base class, manager, attribute)
2. **Content Entity Plugin**: Core plugin implementation with entity discovery and JSON:API serialization
3. **Configuration Entity**: McpResourceConfig entity definition and schema
4. **Admin UI**: List builder and configuration form for resource management
5. **Service Integration**: Resource bridge service and factory integration
6. **Testing**: Comprehensive kernel test suite
7. **Documentation**: Code documentation and admin help text

## Notes

### Alignment with Resource Ideas Document

This implementation addresses the following from `.ai/task-manager/scratch/mcp-resource-ideas.md`:

- **Content Entity Resources** (Section 1.1-1.8): Implements access to nodes, media, taxonomy terms, and other core content entities
- **Security Framework** (Section 2): Enforces entity access control per security classification guidelines
- **URI Scheme Design** (Section 6.1): Uses recommended `drupal://entity/{entity_type}/{entity_id}` pattern
- **Phase 1 Foundation** (Section 7): Content entities identified as must-have foundation resources
- **JSON Serialization** (Section 6.7): Uses JSON:API for standardized, consistent serialization

### Future Enhancements (Out of Scope)

The following features are explicitly out of scope for this plan but may be addressed in future iterations:

- **Resource Listing**: `drupal://entity/node/list` endpoints (mentioned in resource ideas document)
- **Revision Access**: `drupal://entity/node/{nid}/revisions` (explicitly excluded per requirements)
- **Configuration Entity Resources**: Separate plugin type for config entities (`config://node_type/article`)
- **Contrib Module Resources**: Webform, Search API, Paragraphs, etc. (Phase 2 in resource ideas)
- **Advanced Resources**: Relationship graphs, field usage statistics, content audits (Phase 3)
- **Resource Subscriptions**: Change notifications via MCP's resources/subscribe

### Dependency Considerations

The Content Entity Resource Template requires:
- **JSON:API module**: For entity serialization (ships with Drupal core, but must be enabled)
- **MCP SDK**: Resource registration API (already a dependency of mcp_server module)

Other potential resource plugins may have different dependencies (e.g., Views module for view-based resources).

---

## Task Dependency Visualization

```mermaid
graph TD
    T1[Task 1: Plugin System Infrastructure] --> T2[Task 2: Content Entity Plugin]
    T1 --> T3[Task 3: Resource Config Entity]
    T1 --> T4[Task 4: Admin UI]
    T2 --> T5[Task 5: Bridge Service Integration]
    T3 --> T4
    T3 --> T5
    T1 --> T5
    T2 --> T6[Task 6: Integration Testing]
    T3 --> T6
    T4 --> T6
    T5 --> T6

    style T1 fill:#90EE90
    style T6 fill:#FFD700
```

---

## Execution Blueprint

**Validation Gates:**
- Reference: `.ai/task-manager/config/hooks/POST_PHASE.md`

### ✅ Phase 1: Foundation
**Parallel Tasks:**
- ✔️ Task 1: Plugin System Infrastructure (drupal-plugin-system, php)

**Objective**: Establish the core plugin architecture that all other components depend on. This phase creates the plugin interface, base class, manager, and attribute system.

**Completion Criteria**: Plugin system is functional, ResourceTemplateManager service is registered, code passes PHPCS/PHPStan.

---

### ✅ Phase 2: Core Components
**Parallel Tasks:**
- ✔️ Task 2: Content Entity Plugin (depends on: 1)
- ✔️ Task 3: Resource Config Entity (depends on: 1)

**Objective**: Implement the first concrete resource plugin and the configuration entity system. These can be developed in parallel as they don't depend on each other.

**Completion Criteria**: ContentEntityResourceTemplate plugin is discoverable, McpResourceConfig entity is functional, both pass code quality checks.

---

### ✅ Phase 3: Integration Layer
**Parallel Tasks:**
- ✔️ Task 4: Admin UI (depends on: 1, 3)
- ✔️ Task 5: Bridge Service Integration (depends on: 1, 2, 3)

**Objective**: Build the administrative interface and integrate resources with the MCP server. The admin UI requires the config entity (task 3), while the bridge service requires both the plugin (task 2) and config entity (task 3).

**Completion Criteria**: Admin UI is accessible and functional, resources are registered with MCP server, dependency validation works.

---

### ✅ Phase 4: Validation
**Parallel Tasks:**
- ✔️ Task 6: Integration Testing (depends on: 2, 3, 4, 5)

**Objective**: Comprehensive testing of the entire resource system including all integration points.

**Completion Criteria**: All tests pass, code coverage is adequate, access control and serialization are validated.

---

### Post-phase Actions

After each phase completion:
1. Run `vendor/bin/drush cache:rebuild` to ensure services/plugins are discovered
2. Run `vendor/bin/phpcs` and `vendor/bin/phpstan` to validate code quality
3. Manually verify functionality through Drush or admin UI
4. Review git diff to ensure changes align with plan requirements

After Phase 4:
1. Run full test suite: `vendor/bin/phpunit web/modules/contrib/mcp_server/tests/`
2. Verify resources are accessible via MCP client (if available)
3. Test with actual content entities (nodes, media, taxonomy terms)
4. Document any deviations from original plan

---

### Execution Summary
- **Total Phases**: 4
- **Total Tasks**: 6
- **Maximum Parallelism**: 2 tasks (in Phases 2 and 3)
- **Critical Path Length**: 4 phases
- **Estimated Complexity**: All tasks ≤5.5 on composite complexity scale
- **Skills Required**: drupal-plugin-system, drupal-entity-api, jsonapi, drupal-config-entities, drupal-forms, drupal-admin-ui, drupal-services, mcp-sdk, phpunit, drupal-testing

## Execution Summary

**Status**: ✅ Completed Successfully
**Completed Date**: 2025-11-19

### Results

Successfully implemented a complete MCP resource template system for exposing Drupal content entities as MCP resources. All 6 tasks across 4 phases were completed with all validation gates passing.

**Deliverables:**
- **Plugin System Infrastructure**: ResourceTemplateInterface, ResourceTemplateBase, ResourceTemplateManager, ResourceTemplate attribute
- **Content Entity Plugin**: ContentEntityResourceTemplate discovering entities with canonical URLs, enforcing access control, JSON:API serialization
- **Configuration Entity**: McpResourceConfig for managing resource enablement with dependency tracking
- **Admin UI**: List builder and form for resource management with visual dependency indicators
- **MCP Integration**: McpResourceBridgeService and McpServerFactory integration making resources discoverable to MCP clients
- **Test Suite**: Comprehensive kernel tests (10 test methods, 194 assertions) validating all critical functionality

**Code Quality:**
- All code passes PHPCS (Drupal & DrupalPractice standards)
- All code passes PHPStan static analysis (level 5)
- All tests pass successfully
- 4 Git commits with conventional commit messages
- Pre-commit hooks validated all changes

### Noteworthy Events

**Bugs Fixed During Testing:**
During integration test development, 3 bugs were discovered and fixed in ContentEntityResourceTemplate:
1. Incorrect namespace import: Changed from Annotation to Attribute for ResourceTemplate class
2. Invalid entity type check method: Changed `isSubclassOf()` to `entityClassImplements()` for proper interface validation
3. Non-existent method call: Removed `getDescription()` call on entity type definitions

These bugs were caught early through test-driven validation and fixed before they could impact functionality.

**Parallel Execution Efficiency:**
Successfully executed tasks in parallel during Phases 2 and 3, improving overall execution time. The dependency management system correctly identified when tasks could run concurrently.

**Agent Specialization:**
Effectively utilized specialized agents (drupal-backend-specialist, testing-qa-engineer) matching task skills, resulting in high-quality implementations following project standards.

### Recommendations

**Immediate Next Steps:**
1. Create example resource configurations for common entity types (node, media, taxonomy_term) to demonstrate the system
2. Document the resource template plugin creation process for developers wanting to add custom resource types
3. Consider adding drush commands for resource management (list, enable, disable)

**Future Enhancements (Out of Scope):**
- Resource listing endpoints (`drupal://entity/node/list`)
- Revision access support (`drupal://entity/node/{nid}/revisions`)
- Configuration entity resources (separate plugin type)
- Contrib module resources (Webform, Search API, Paragraphs)
- Resource subscription/change notifications via MCP protocol

**Testing Expansion:**
- Add functional tests for admin UI interactions once UI testing infrastructure is in place
- Consider performance testing for resource discovery with large numbers of entities
- Add tests for error scenarios (missing JSON:API module, malformed configurations)

